// $Id: stublen.cpp,v 1.1.2.4 2001/03/19 16:51:14 andreasb Exp $

#include "baseincs.h"
#define OGR_CORE_INTERNAL_STRUCTURES
#include "ogr.h"

#define NODESPERRUN 1000000

extern "C" CoreDispatchTable* ogr_get_dispatch_table();
static const int OGR_length[] = { /* use: OGR_length[depth] */
/* marks */
  /*  1 */    0,   1,   3,   6,  11,  17,  25,  34,  44,  55,
  /* 11 */   72,  85, 106, 127, 151, 177, 199, 216, 246, 283,
  /* 21 */  333, 356, 372, 425, 480, 492, 553, 585, 623
};

int marks=25;
#define MAXMARKS 26
#define DIFFS (marks-1)
int len=6;
int minsum=71;
int maxdifflen=255;
int seed=1;
int start=0;
int stop=9999;
int *diffs, diffsum;

int exitnow = 0;

FILE *outfile=0;

static void usage()
{
  fprintf(stderr, "Usage:  runstub OF MARKS LENGTH MINDIFFSUM MAXDIFFLEN SEED START STOP\n\n"
        "This will generate random stubs, pass them through ogr_cycle and save them and\n"
        "the resulting nodecout to a file.\n\n"
        "OF\n"
        "     The name of the output text file.\n\n"
        "MARKS\n"
        "     The number of marks in the OGR.\n\n"
        "LENGTH\n"
        "     The number of diffs in a stub.\n\n"
        "MINDIFFSUM\n"
        "     This can be used to set a lower limit for the sum of the diffs.\n\n"
        "MAXDIFFLEN\n"
        "     This is the maximum length for a single diff.\n\n"
        "SEED\n"
        "     This is the seed for the random number generator.\n\n"
        "START, STOP\n"
        "     Normally you would set the to 0 and say 9999 if you want to process 10000\n"
        "     random stubs for a given SEED. But if you want to continue this series you\n"
        "     can set them to 10000 and say 19999 for the next 10000 stubs.\n\n");
}

WorkStub ws;
State st;
CoreDispatchTable *dispatch = ogr_get_dispatch_table();

int loadstub()
{
  int length = 0;
  ws.stub.marks = marks;
  ws.stub.length = len;
  ws.worklength = 0;

  for (int i = 0; i < STUB_MAX && i < len; i++)
  {
    ws.stub.diffs[i] = diffs[i];
    length += diffs[i];
  }

  printf("Loaded: %d/", ws.stub.marks);
  for (u32 i = 0; i < ws.stub.length; i++)
    printf("%s%d", i?"-":"", ws.stub.diffs[i]);
  printf("   (%d)\n", length);

  return 0;
}

void printstub(const State *state, char eol = '\n', int depth = -1)
{
  int len = 0;

  if (depth < 0)
    depth = state->depth;

  printf("%d(%d)/", state->maxmarks, depth);
  for (int i = 0; i < state->maxdepth && (i < depth || i < state->startdepth); ++i)
  {
    if (i == state->startdepth)
      printf("+");
    else if (i)
      printf("-");
    printf("%d", state->markpos[i+1] - state->markpos[i]);
    len += state->markpos[i+1] - state->markpos[i];
  }
  if (depth == state->startdepth)
    printf("+");
  if (depth < state->maxdepth)
    printf("...");
  // printf(" = %d   %c",len,eol);
  printf("          %c", eol);
}

int prandom(int max) // returns int between and including 1 and max
{
  return random()%max+1;
}

int is_golomb()
{
  int *count;
  int i,k;
  int sum;

  count=(int *)malloc(len*maxdifflen*sizeof(int));
  if (count==0) {
    fprintf(stderr, "Could not allocate memory.\n");
    exit(1);
  }

  memset(count, 0, 1024*sizeof(int));

  for(k=0; k<len; k++) {
    sum=0;
    for(i=k; i>=0; i--) {
      sum+=diffs[i];
      count[sum]++;
      if(count[sum]>1) {
        free(count);
        return 0;
      }
    }
  }
  free(count);
  return 1;
}

int is_toobig()
{
  int i, diffsum;

  diffsum=0;
  for(i=0; i<len; i++) {
    if ((OGR_length[DIFFS]-OGR_length[DIFFS-1-i]-diffsum) < diffs[i])
      return 1;
    diffsum+=diffs[i];
  }

  return 0;
}

/*
 * This function seems to be very inefficient, but it is not. It needs to be so
 * inefficient to ensure that each possible stub can be generated with the same
 * likelyhood.
 */
void generate_random_stub()
{
  int n;

  do {
    diffsum=0;
    for(n=0; n<len; n++) {
      diffs[n]=prandom(maxdifflen);
      diffsum+=diffs[n];
    }
  } while (!is_golomb() || is_toobig() || diffsum<minsum);
}

int process_stub()
{
  loadstub();

  int res = dispatch->init();
  //printf("ogr_init() = %d\n", res);
  if (res != CORE_S_OK)
    return 42;

  if (int r = dispatch->create(&ws, sizeof(ws), &st, sizeof(st)) != CORE_S_OK)
  {
    fprintf(stderr, "ERROR: ogr_create() = %d\n", r);
    return 2;
  }
  //printstub(&st);

// ============================================================================    

  unsigned long long totalnodes = 0;
  int r;
  do {
    int nodes = NODESPERRUN;
    r = dispatch->cycle(&st, &nodes, 0);
    totalnodes += nodes;
    //printf("%d nodes = %d\n", r, nodes);
    printf("%.0f ", (double)totalnodes); 
    printstub(&st, '\r');
    fflush(stdout);

    switch (r) {
      case CORE_S_OK :
          break;
      case CORE_S_CONTINUE :
          break;
      case CORE_S_SUCCESS :
          printstub(&st, '\n', st.depth+1);
          break;
    }
  } while (r == CORE_S_CONTINUE);

  printf("\nStub (%d)  ", r);
  printstub(&st);
  printf("Totalnodes %.0f %016LX\n", (double)totalnodes, totalnodes);

// ============================================================================ 

  fprintf(outfile, "%.0f\n", (double)totalnodes);
  fflush(outfile);

  return 0;  
}

void exitfn(void)
{
  if (outfile!=0)
    fclose(outfile);
}

void handler(int i)
{
  exitnow = 1;
  printf("\nCatched a signal! Stublen will stop soon!\n");
}

int main(int argc, char **argv)
{
  int n,i;

  atexit(exitfn);
  signal(SIGHUP, handler);
  signal(SIGINT, handler);
  signal(SIGTERM, handler);

  if(argc!=9) {
    usage();
    exit(1);
  }

  outfile=fopen(argv[1], "a");
  if (outfile==0) {
    fprintf(stderr, "Could not open output file %s.\n", argv[1]);
    exit(1);
  }

  marks=atoi(argv[2]);
  if(marks>MAXMARKS) {
    fprintf(stderr, "MARKS should not be greater than %d.\n", MAXMARKS);
    exit(1);
  }
  len=atoi(argv[3]);
  if(len>marks-2) {
    fprintf(stderr, "It is no use to have LENGTH=%d if MARKS=%d.\n", len, marks);
    exit(1);
  }
  minsum=atoi(argv[4]);
  maxdifflen=atoi(argv[5]);
  seed=atoi(argv[6]);
  start=atoi(argv[7]);
  stop=atoi(argv[8]);

  diffs=(int *)malloc(len*sizeof(int));
  if (diffs==0) {
    fprintf(stderr, "Could not allocate memory.\n");
    exit(1);
  }

  srandom(seed);

  for(n=0; n<=stop && !exitnow; n++) {
    generate_random_stub();
    if (n>=start) {
      printf("\n%d : ", n);
      fprintf(outfile, "%d %d %d %d %d %5d : ", marks, len, minsum, maxdifflen, seed, n);
      for(i=0; i<len; i++)
        fprintf(outfile, "%3d ", diffs[i]);
      fprintf(outfile, ": %3d : ", diffsum);
      process_stub();
    }
  }
}
