/*
 * Copyright distributed.net 1997-2003 - All Rights Reserved
 * For use in distributed.net projects only.
 * Any other distribution or use of this source violates copyright.
 *
 * Created by Cyrus Patel (cyp@fb14.uni-mainz.de)
 *
 * Stripped down version for buffer utilities etc.
 * Logs to screen only.
*/
//#define TRACE

const char *logstuff_cpp(void) {
return "@(#)$Id: fakelog.cpp,v 1.2.4.1 2003/02/25 12:30:23 snake Exp $"; }

#include "cputypes.h"
#include "baseincs.h"  // basic (even if port-specific) #includes
#include "client.h"    // CONTEST_COUNT
#include "mail.h"      // MailMessage
#include "clitime.h"   // CliGetTimeString(NULL,1)
#include "pathwork.h"  // GetFullPathForFilename(), GetWorkingDirectory()
#include "problem.h"   // Problem object for logscreenpercent
#include "probman.h"   // GetProblemPointerFromIndex() for logscreenpercent
#include "bench.h"     // BenchGetBestRate() for logscreenpercent
#include "cpucheck.h"  // GetNumberOfDetectedProcessors() for logscreenpercent
#include "clicdata.h"  // CliGetContestNameFromID() for logscreenpercent
#include "console.h"   // for ConOut(), ConIsScreen(), ConIsGUI()
#include "clievent.h"  // ClientEventSyncPost()
#include "util.h"      // TRACE
#include "logstuff.h"  // keep the prototypes in sync

//-------------------------------------------------------------------------
#ifndef LOGTO_NONE       /* the following are declared in logstuff.h */
#define LOGTO_NONE       0x00
#define LOGTO_SCREEN     0x01
#define LOGTO_FILE       0x02
#define LOGTO_MAIL       0x04
#define LOGTO_RAWMODE    0x80
#endif

#ifndef LOGFILETYPE_NONE    /* the following are declared in logstuff.h */
#define LOGFILETYPE_NONE    0 //no logging to file
#define LOGFILETYPE_NOLIMIT 1 //unlimited (or limit == -1)
#define LOGFILETYPE_RESTART 2 //then logLimit is in KByte
#define LOGFILETYPE_FIFO    3 //then logLimit is in KByte (minimum 100K)
#define LOGFILETYPE_ROTATE  4 //then logLimit is in days
#endif

#define ASSERT_WIDTH_80     //show where badly formatted lines are cropping up
#define ASSUMED_SCREEN_WIDTH 80 //... until all platforms support ConGetSize()
// ========================================================================

static struct
{
  int initlevel;
  int loggingTo;            // LOGTO_xxx bitfields
  int spoolson;             // mail/file logging and time stamping is on/off.
  int crunchmeter;          // progress ind style (-1=def,0=off,1=abs,2=rel)
  int percbaton;            // percent baton is enabled

  void *mailmessage;         //handle returned from smtp_construct_message()
  char basedir[256];         //filled if user's 'logfile' is not qualified
  char logfile[128];         //filename from user
  FILE *logstream;           //open logfile if /dev/* and "no-limit"
  int  logfileType;          //rotate, restart, fifo, none
  unsigned int logfileLimit; //days when rotating or kbyte when fifo/restart
  unsigned int logfilestarted; // non-zero after the first logfile write 
                             //also used to mark len of the log fname without 
                             // ROTATE suffix

  int stdoutisatty;         //log screen can handle lines not ending in '\n'
  int stableflag;           //last log screen did end in '\n'
  int lastwasperc;          //last log screen was a percentbar
  unsigned int perc_callcount; //#of times LogScreenPercent() printed something
  unsigned int lastperc_done; //percentage of last LogScreenPercent() dot

} logstatics = {
  0,      // initlevel
  LOGTO_SCREEN,   // loggingTo
  0,      // spoolson
  0,      // crunchmeter
  0,      // percbaton
  NULL,   // *mailmessage
  {0},    // basedir[]
  {0},    // logfile[]
  NULL,   // logstream
  LOGFILETYPE_NONE, // logfileType
  0,      // logfileLimit
  0,      // logfilestarted
  0,      // stdoutisatty
  0,      // stableflag
  0,      // lastwasperc
  0,      // perc_callcount
  0       // lastperc_done
};      

// ========================================================================

// On NT/Alpha (and maybe some other platforms) the va_list type is a struct,
// not an int or a pointer-type.  Hence, NULL is not a valid va_list.  Pass
// a (va_list *) instead to avoid this problem
void LogWithPointer( int loggingTo, const char *format, va_list *arglist )
{
  char msgbuffer[1024]; //min 1024!!, but also think of other OSs stack!!
  unsigned int msglen = 0, sel;
  char *buffptr, *obuffptr;
  int old_loggingTo = loggingTo;

  msgbuffer[0]=0;
  loggingTo &= (logstatics.loggingTo|LOGTO_RAWMODE);

  if ( !format || !*format )
    loggingTo = LOGTO_NONE;

  if ( loggingTo != LOGTO_NONE )
  {
    if ( arglist == NULL )
      strcat( msgbuffer, format );
    else
      vsprintf( msgbuffer, format, *arglist );
    msglen = strlen( msgbuffer );

    if ( msglen == 0 )
      loggingTo = LOGTO_NONE;
    else if (msgbuffer[msglen-1] != '\n')
      loggingTo &= LOGTO_SCREEN|LOGTO_RAWMODE;  //screen only obviously
  }

  if (loggingTo != LOGTO_NONE && logstatics.spoolson /* timestamps */ &&
      (old_loggingTo & LOGTO_RAWMODE) == 0 )
  {
    buffptr = &msgbuffer[0];
    sel = 1;
    do
    {
      while (*buffptr == '\r' || *buffptr=='\n' )
        buffptr++;
      if (*buffptr == ' ' || *buffptr == '\t')
      {
        obuffptr = buffptr;
        while (*obuffptr == ' ' || *obuffptr == '\t')
          obuffptr++;
        memmove( buffptr, obuffptr, strlen( obuffptr )+1 );
      }
      if (*buffptr && *buffptr!='[' && *buffptr!='\r' && *buffptr!='\n' )
      {
        const char *timestamp = CliGetTimeString( NULL, sel );
        memmove( buffptr+(strlen(timestamp)+3), buffptr, strlen( buffptr )+1 );
        *buffptr++=((sel)?('['):(' '));
        while (*timestamp)
          *buffptr++ = *timestamp++;
        *buffptr++=((sel)?(']'):(' '));
        *buffptr=' ';
      }
      sel = 0;
      while (*buffptr && *buffptr != '\n' && *buffptr != '\r')
        buffptr++;
    } while (*buffptr);
    msglen = strlen( msgbuffer );
  }

  if (( loggingTo & LOGTO_SCREEN ) != 0)
  {
    int scrwidth = ASSUMED_SCREEN_WIDTH; /* assume this for consistancy */

    buffptr = &msgbuffer[0];
    if ((loggingTo & LOGTO_RAWMODE)==0)
    {
      if (logstatics.stableflag) /* previous print ended with '\n'|'\r' */
      {
        if (*buffptr=='\n') /* remove extraneous leading '\n' */
        {
          msglen--;
          buffptr++;
        }
      }    
      else  /* a linefeed is pending */
      {
        if (*buffptr == '\r')  /* curr print expects to overwrites previous */
        {                      /* so ensure the old line is clear */
          memmove( &msgbuffer[scrwidth], msgbuffer, msglen+1 );
          msglen += scrwidth;
          memset( msgbuffer, ' ', scrwidth );
          msgbuffer[0] = '\r';
        }
        else if (*buffptr!='\n') /* curr print expects to be on a newline */
        {                        /* so ensure it is */
          msglen++;
          memmove( msgbuffer+1, msgbuffer, msglen );
          msgbuffer[0] = '\n';
          logstatics.stableflag = 1;
        }  
      }
    }
    if (msglen)
    {
      logstatics.lastwasperc = 0; //perc bar looks for this
      logstatics.stableflag = ( buffptr[(msglen-1)] == '\n' || 
                                buffptr[(msglen-1)] == '\r' );
      printf(buffptr);
      fflush(stdout);
    }
  }

  return;
}

// ------------------------------------------------------------------------

#if (CLIENT_OS == OS_WIN16) && defined(__WATCOMC__)
#define MAKE_VA_LIST_PTR(__va_list) ((va_list *)(&(__va_list[0])))
#else
#define MAKE_VA_LIST_PTR(__va_list) (&__va_list)
#endif

void LogScreen( const char *format, ... )
{
  va_list argptr;
  va_start(argptr, format);
  LogWithPointer( LOGTO_SCREEN, format, MAKE_VA_LIST_PTR(argptr) );
  va_end(argptr);
  return;
}

void LogScreenRaw( const char *format, ... )
{
  va_list argptr;
  va_start(argptr, format);
  LogWithPointer( LOGTO_RAWMODE|LOGTO_SCREEN, format, MAKE_VA_LIST_PTR(argptr));
  va_end(argptr);
  return;
}

void Log( const char *format, ... )
{
  va_list argptr;
  va_start(argptr, format);
  LogWithPointer( LOGTO_SCREEN|LOGTO_FILE|LOGTO_MAIL, format, MAKE_VA_LIST_PTR(argptr));
  va_end(argptr);
  return;
}

void LogRaw( const char *format, ... )
{
  va_list argptr;
  va_start(argptr, format);
  LogWithPointer( LOGTO_RAWMODE|LOGTO_SCREEN|LOGTO_FILE|LOGTO_MAIL, format, MAKE_VA_LIST_PTR(argptr));
  va_end(argptr);
  return;
}

void LogTo( int towhat, const char *format, ... )
{
  va_list argptr;
  va_start(argptr, format);
  LogWithPointer( towhat, format, MAKE_VA_LIST_PTR(argptr) );
  va_end(argptr);
  return;
}
