/*
 * Copyright (c) 2019 Universita' di Firenze, Italy
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Tommaso Pecorella <tommaso.pecorella@unifi.it>
 */

#include "ns3/core-module.h"
#include "ns3/csma-module.h"
#include "ns3/internet-apps-module.h"
#include "ns3/internet-module.h"
#include "ns3/lr-wpan-module.h"
#include "ns3/mobility-module.h"
#include "ns3/propagation-module.h"
#include "ns3/sixlowpan-module.h"
#include "ns3/spectrum-module.h"

#include <fstream>

using namespace ns3;

int
main(int argc, char** argv)
{
    bool verbose = false;

    Packet::EnablePrinting();

    CommandLine cmd(__FILE__);
    cmd.AddValue("verbose", "turn on log components", verbose);
    cmd.Parse(argc, argv);

    if (verbose)
    {
        LogComponentEnable("Ping6Application", LOG_LEVEL_ALL);
        LogComponentEnable("LrWpanMac", LOG_LEVEL_ALL);
        LogComponentEnable("LrWpanPhy", LOG_LEVEL_ALL);
        LogComponentEnable("LrWpanNetDevice", LOG_LEVEL_ALL);
        LogComponentEnable("SixLowPanNetDevice", LOG_LEVEL_ALL);
    }

    uint32_t nWsnNodes = 4;
    NodeContainer wsnNodes;
    wsnNodes.Create(nWsnNodes);

    NodeContainer wiredNodes;
    wiredNodes.Create(1);
    wiredNodes.Add(wsnNodes.Get(0));

    MobilityHelper mobility;
    mobility.SetMobilityModel("ns3::ConstantPositionMobilityModel");
    mobility.SetPositionAllocator("ns3::GridPositionAllocator",
                                  "MinX",
                                  DoubleValue(0.0),
                                  "MinY",
                                  DoubleValue(0.0),
                                  "DeltaX",
                                  DoubleValue(80),
                                  "DeltaY",
                                  DoubleValue(80),
                                  "GridWidth",
                                  UintegerValue(10),
                                  "LayoutType",
                                  StringValue("RowFirst"));
    mobility.SetMobilityModel("ns3::ConstantPositionMobilityModel");
    mobility.Install(wsnNodes);

    LrWpanHelper lrWpanHelper;
    // Add and install the LrWpanNetDevice for each node
    NetDeviceContainer lrwpanDevices = lrWpanHelper.Install(wsnNodes);

    // Fake PAN association and short address assignment.
    // This is needed because the lr-wpan module does not provide (yet)
    // a full PAN association procedure.
    lrWpanHelper.AssociateToPan(lrwpanDevices, 0);

    InternetStackHelper internetv6;
    internetv6.Install(wsnNodes);
    internetv6.Install(wiredNodes.Get(0));

    SixLowPanHelper sixLowPanHelper;
    NetDeviceContainer sixLowPanDevices = sixLowPanHelper.Install(lrwpanDevices);

    CsmaHelper csmaHelper;
    NetDeviceContainer csmaDevices = csmaHelper.Install(wiredNodes);

    Ipv6AddressHelper ipv6;
    ipv6.SetBase(Ipv6Address("2001:cafe::"), Ipv6Prefix(64));
    Ipv6InterfaceContainer wiredDeviceInterfaces;
    wiredDeviceInterfaces = ipv6.Assign(csmaDevices);
    wiredDeviceInterfaces.SetForwarding(1, true);
    wiredDeviceInterfaces.SetDefaultRouteInAllNodes(1);

    ipv6.SetBase(Ipv6Address("2001:f00d::"), Ipv6Prefix(64));
    Ipv6InterfaceContainer wsnDeviceInterfaces;
    wsnDeviceInterfaces = ipv6.Assign(sixLowPanDevices);
    wsnDeviceInterfaces.SetForwarding(0, true);
    wsnDeviceInterfaces.SetDefaultRouteInAllNodes(0);

    for (uint32_t i = 0; i < sixLowPanDevices.GetN(); i++)
    {
        Ptr<NetDevice> dev = sixLowPanDevices.Get(i);
        dev->SetAttribute("UseMeshUnder", BooleanValue(true));
        dev->SetAttribute("MeshUnderRadius", UintegerValue(10));
    }

    uint32_t packetSize = 10;
    uint32_t maxPacketCount = 5;
    Time interPacketInterval = Seconds(1.);
    Ping6Helper ping6;

    ping6.SetLocal(wsnDeviceInterfaces.GetAddress(nWsnNodes - 1, 1));
    ping6.SetRemote(wiredDeviceInterfaces.GetAddress(0, 1));

    ping6.SetAttribute("MaxPackets", UintegerValue(maxPacketCount));
    ping6.SetAttribute("Interval", TimeValue(interPacketInterval));
    ping6.SetAttribute("PacketSize", UintegerValue(packetSize));
    ApplicationContainer apps = ping6.Install(wsnNodes.Get(nWsnNodes - 1));

    apps.Start(Seconds(1.0));
    apps.Stop(Seconds(10.0));

    AsciiTraceHelper ascii;
    lrWpanHelper.EnableAsciiAll(ascii.CreateFileStream("Ping-6LoW-lr-wpan-meshunder-lr-wpan.tr"));
    lrWpanHelper.EnablePcapAll(std::string("Ping-6LoW-lr-wpan-meshunder-lr-wpan"), true);

    csmaHelper.EnableAsciiAll(ascii.CreateFileStream("Ping-6LoW-lr-wpan-meshunder-csma.tr"));
    csmaHelper.EnablePcapAll(std::string("Ping-6LoW-lr-wpan-meshunder-csma"), true);

    Simulator::Stop(Seconds(10));

    Simulator::Run();
    Simulator::Destroy();
}
