// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

namespace acudriver {
using namespace ost;
using namespace std;

static Keydata::Define driver[] = {
	{"stack", "0"},
	{"events", "256"},
	{"priority", "1"},
        {"prifirmware", "ni2_usr.p4r"},
        {"brifirmware", "ets_bnet.ram"},
        {"prosody", "/home/aculab/dtk141/speech/download/speech/sp30a.smf"},
        {"briconfig", ""},
        {"priconfig", ""},
        {"netformat", "alaw"},
        {"buffersize","256"},
        {"apitrace","0"},
        {NULL, NULL}};

#ifdef	WIN32
#define	KEYS	"/bayonne/aculab"
#else
#define	KEYS	"/bayonne/driver/aculab"
#endif

Driver Driver::aculab;

Driver::Driver() : BayonneDriver(driver, KEYS, "aculab", true),
Thread(atoi(getLast("priority")) + 1, atoi(getLast("stack")) * 1024) {
#ifdef	WIN32
	const char *env = Process::getEnv("DRIVERCONFIG");

	if(env)
		loadFile(env, "aculab");
#else
	load("~bayonne/aculab");
#endif
}

unsigned Driver::getMaxPorts(void)
{
        const char *val = getLast("maxports");
        int v;
        if(!val)
                return MAXPORT;

        v = atoi(val);    
        if(v > MAXPORT)       
                return MAXPORT;

        return v;
}

void Driver::startDriver(void)
{
	int total_switches;
        unsigned this_port, total_ports = 0;
        struct restart_xparms restart_xparms;
        struct sm_download_parms sm_download_parms;
	int this_module, total_modules;
        int rc;	
	unsigned port, ts, id;
	unsigned long mask;

        pri_count = bri_count = 0;

        total_switches = sw_get_drvrs();
	total_ports = call_nports();

        if(total_switches < 1 || total_ports < 1) 
	{     
        	slog.error() << "Aculab Failure: no switch drivers" << endl;
                return;
        }


	for(this_port = 0; this_port < getMaxPorts(); ++this_port) 
                endpoint[this_port].ptype = endpoint_t::INVALID_PORT;


        for(this_port = 0; this_port < total_ports; ++this_port) {

                if(call_line(this_port) == L_BASIC_RATE) 
		{
                        endpoint[this_port].ptype = endpoint_t::BRI_PORT;
                        ++bri_count; 
                        restart_xparms.filenamep = (char *)getLast("brifirmware");                            
			restart_xparms.config_stringp = (char *)getLast("briconfig");
                }
                else {
        		endpoint[this_port].ptype = endpoint_t::PRI_PORT;
                        ++pri_count;

                        restart_xparms.filenamep = (char *)getLast("prifirmware");                      
                        restart_xparms.config_stringp = (char *)getLast("priconfig");
                        if (restart_xparms.config_stringp == NULL)
                                restart_xparms.config_stringp="";
                }
                if(call_is_download(this_port)) {
                        slog(Slog::levelInfo) << "aculab/" << this_port
                                << ": loading '" << restart_xparms.filenamep
                                << "'/'"<<restart_xparms.config_stringp
                                << "' to port  " << this_port << endl;

                        restart_xparms.net = this_port;
                        rc = call_restart_fmw(&restart_xparms);
                        if(rc) {
                		slog(Slog::levelError) << "aculab/" << this_port
                		<< ": firmware load failure: " << rc << endl;
                                endpoint[this_port].ptype = endpoint_t::FAILED_PORT;
                                return;
                        }
                }
        }

        if(system_init())
	{
		slog(Slog::levelError) << "aculab: system init failure" << endl;
                return;
        }

	total_modules = sm_get_modules();
	
        slog(Slog::levelInfo) << "aculab loading " << total_modules
                        << " prosody modules" << endl;

        for(this_module = 0; this_module < total_modules; ++this_module) 
	{
                if(sm_reset_module(this_module)) 
		{
                	slog(Slog::levelError) << "aculab module " << this_module
                                               << " failed, exiting..." << endl;
                        return;
                }
                sm_download_parms.module = this_module;
                sm_download_parms.id = 0;
                sm_download_parms.filename = (char *)getLast("prosady");
                slog(Slog::levelInfo) << "aculab module " << this_module
                		<< " downloading '"<<sm_download_parms.filename
                                << "'..." << endl;
                rc=sm_download_fmw(&sm_download_parms);
                if (rc != 0) 
		{
                        slog(Slog::levelError) << "aculab module download failed: " << rc << endl;
                        return;
                }
        }

	slog.info("aculab starting; switches=%d, calls=%d", total_switches, total_ports); 

	unsigned port_count = total_ports;

	imaps = new Session *[port_count * MAXCHANNELS];
	omaps = new Session *[port_count * MAXCHANNELS];
	ixmaps = new Session *[MAX_CH_IDX];
	tsmap = new char[port_count * MAXTIMESLOTS];
	pmaps = new Session *[port_count * MAXTIMESLOTS];
	devnodes = new devnode_t[port_count * MAXTIMESLOTS];

	memset(imaps, 0, sizeof(Session *) * port_count * MAXCHANNELS);
        memset(omaps, 0, sizeof(Session *) * port_count * MAXCHANNELS);
        memset(ixmaps, 0, sizeof(Session *) * MAX_CH_IDX);
	memset(tsmap, 0, port_count * MAXTIMESLOTS);
	memset(pmaps, 0, sizeof(Session *) * port_count * MAXTIMESLOTS);
	msgport = new BayonneMsgport(this);

        call_signal_info(&siginfo[0]);

        for(port = 0; port < port_count; ++port) 
	{
		slog(Slog::levelInfo) << "aculab("<<port<<",) signalling system: "
                                << siginfo[port].sigsys <<endl;
                mask = 1L;
                for(ts = 0; ts < MAXTIMESLOTS; ++ts) {
                        if(mask & siginfo[port].validvector) {     
                                tsmap[port * MAXTIMESLOTS + ts] = 1;
                        }
                        mask <<=1L;
                }
        }

        for(port = 0; port < port_count; ++port) {
                for(ts = 0; ts < MAXTIMESLOTS; ++ts) {
                        id = port * MAXTIMESLOTS + ts;
                        if(!tsmap[port * MAXTIMESLOTS + ts])
                                continue;

			if(ts_used >= ts_count)
				break;

			devnodes[id].timeslot = ts;
			devnodes[id].port = port;
			pmaps[id] = new Session(NULL, ts_used++, &devnodes[id]);
                        ++count;
                }
        }

	msgport->start();
	BayonneDriver::startDriver();
	Thread::start();
}

void Driver::stopDriver(void)
{
	if(running)
		BayonneDriver::stopDriver();
	terminate();
}

void Driver::run(void)
{
        struct state_xparms event_xparms;
        unsigned port, channel;
        int rc;
	Session *session;
	Event event;

	while(running)
	{
                event_xparms.handle = 0;
                event_xparms.timeout = 100L;

		Thread::yield();
                rc=call_event(&event_xparms);
                if(rc != 0) 
		{      
		        slog(Slog::levelError) << "aculab event failure; rc="<<rc<< endl;                               Thread::sleep(100);
                        continue;
                }

                if(!event_xparms.handle)
                        continue;

                port = call_handle_2_port(event_xparms.handle);
                channel = call_handle_2_chan(event_xparms.handle);

                if(port >= MAXPORT)
                        continue;

                if(channel >= MAXCHANNELS)
                        continue;


        	slog(Slog::levelDebug)<<"got aculab event "<<event_xparms.state<<
                               " on port "<<port<<",chan "<<channel<<endl;

                if (call_handle_2_io(event_xparms.handle) == 0)
                        session = imaps[port * MAXCHANNELS + channel];
                else 
                        session = omaps[port * MAXCHANNELS + channel];

                if(!session) 
		{
                	slog(Slog::levelWarning)<<": NO TRUNK FOR CHANNEL!"<<endl;
                        continue;
                }

		memset(&event, 0, sizeof(event));
		switch(event_xparms.state)
		{
                case EV_WAIT_FOR_INCOMING:
                case EV_WAIT_FOR_OUTGOING:
                case EV_OUTGOING_PROCEEDING:
                        break;
		case EV_DETAILS:
			event.id = CALL_INFO;
			break;
		case EV_INCOMING_CALL_DET:
			event.id = CALL_DETECT;
			break;
		case EV_CALL_CONNECTED:
			event.id = CALL_CONNECTED;
			break;
		case EV_PROGRESS:
		case EV_OUTGOING_RINGING:
			event.id = CALL_RINGING;
			break;
		case EV_HOLD:
			event.id = CALL_HOLDING;
			break;
		case EV_HOLD_REJECT:
			event.id = CALL_NOHOLD;
			break;
		case EV_TRANSFER_REJECT:
			event.id = CALL_FAILURE;
			break;
		case EV_IDLE:
			event.id = CALL_RELEASED;
			break;
		}

		if(event.id)
			session->queEvent(&event);
	}
}

} // namespace
