--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--


CREATE FUNCTION create_table_known_coins(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'known_coins';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(known_coin_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',denominations_serial INT8 NOT NULL'
      ',coin_pub BYTEA NOT NULL PRIMARY KEY CHECK (LENGTH(coin_pub)=32)'
      ',age_commitment_hash BYTEA CHECK (LENGTH(age_commitment_hash)=32)'
      ',denom_sig BYTEA NOT NULL'
      ',remaining taler_amount NOT NULL DEFAULT(0,0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'information about coins and their signatures, so we do not have to store the signatures more than once if a coin is involved in multiple operations'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination of the coin, determines the value of the original coin and applicable fees for coin-specific operations.'
    ,'denominations_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'EdDSA public key of the coin'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Value of the coin that remains to be spent'
    ,'remaining'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Optional hash of the age commitment for age restrictions as per DD 24 (active if denom_type has the respective bit set)'
    ,'age_commitment_hash'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'This is the signature of the exchange that affirms that the coin is a valid coin. The specific signature type depends on denom_type of the denomination.'
    ,'denom_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_known_coins(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'known_coins';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_known_coin_id_key'
    ' UNIQUE (known_coin_id)'
  );
END
$$;


CREATE FUNCTION foreign_table_known_coins()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'known_coins';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_denominations'
    ' FOREIGN KEY (denominations_serial) '
    ' REFERENCES denominations (denominations_serial) ON DELETE CASCADE'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('known_coins'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('known_coins'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('known_coins'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
