/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  HttpResponse,
  HttpResponseOk,
  RequestError,
} from "@gnu-taler/web-util/browser";
import { AuditorBackend, WithId } from "../declaration.js";
import { useBackendRequest, useMatchMutate } from "./backend.js";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import _useSWR, { SWRHook, useSWRConfig } from "swr";
import { useEntityContext } from "../context/entity.js";

const useSWR = _useSWR as unknown as SWRHook;

type YesOrNo = "yes" | "no";

interface Props {
  endpoint: string;
  entity: any;
}

export function getEntityList({ endpoint, entity }: Props): HttpResponse<any, AuditorBackend.ErrorDetail> {
  const { fetcher } = useBackendRequest();

  const { data: list, error: listError } = useSWR<
    HttpResponseOk<typeof entity>,
    RequestError<AuditorBackend.ErrorDetail>
  >([`monitoring/` + endpoint], fetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
  });

  if (listError) return listError.cause;

  if (list?.data != null) {
    return { ok: true, data: [list?.data] };
  }
  return { loading: true };
}
export interface EntityAPI {
  updateEntity: (
    id: string
  ) => Promise<void>;
}

export function useEntityAPI(): EntityAPI {
  const mutateAll = useMatchMutate();
  const { request } = useBackendRequest();
  const { endpoint }  = useEntityContext();
  const data = {"suppressed": true};

  const updateEntity = async (
    id: string,
  ): Promise<void> => {
    const r = await request(`monitoring/${endpoint}/${id}`, {
      method: "PATCH",
      data,
    });

    return await mutateAll(/.*\/monitoring.*/);
  };

  return { updateEntity };
}