/* Hey, Emacs, this a -*-C++-*- file !
 *
 * Copyright distributed.net 1997-2000 - All Rights Reserved
 * For use in distributed.net projects only.
 * Any other distribution or use of this source violates copyright.
 *
 * ----------------------------------------------------------------------
 * Supply macros to wrap around the system-dependent file open, creation,
 * and share locking operations.  Notice that you should attempt to do
 * native file-system share locking when possible, even though we
 * additionally implement a "buffer-level soft lock" within the buffer
 * file-format itself.
 *
 * Understand that sharing/locking operations on many Operating Systems
 * are only intended to be advisory locks and do not definitively ensure
 * exclusive access to the file.  This is acceptable, since OS-level
 * locking operations will only augment the existing "buffer-level
 * soft lock" operations that are still performed.
 *
 * Also be aware that UNIX flock() operations (or file regional access) that
 * are performed after the file open are inherently incompatible (do not
 * trigger a lock-conflict) with atomic locking-open operations.  This
 * limitation is also acceptable for the reason above for why advisory
 * and non-guaranteed locking is acceptable.
 * ----------------------------------------------------------------------
 */
#ifndef __BUFFSHIM_H__
#define __BUFFSHIM_H__ "@(#)$Id: buffshim.h,v 1.1.2.4 2000/04/14 13:00:54 cyp Exp $"


#if ((CLIENT_OS == OS_BEOS) || (CLIENT_OS == OS_NEXTSTEP) || \
     (CLIENT_OS == OS_MACOS) || (CLIENT_OS == OS_RISCOS))
  // These operating systems distinguish "binary" files.
  #define BUFFEROPEN( fn )  fopen( fn, "r+b" )
  #undef BUFFEROPENCREATE
  #define BUFFERCREATE( fn )   fopen( fn, "w+b" )
#elif (CLIENT_OS == OS_AMIGAOS) || (CLIENT_OS == OS_VMS)
  // These operating systems don't have an ftruncate equivalent.
  #define BUFFEROPEN( fn )  fopen( fn, "r+" )
  #undef BUFFEROPENCREATE
  #define BUFFERCREATE( fn )   fopen( fn, "w+" )
  #define ftruncate(h,sz)
#elif (CLIENT_OS == OS_SUNOS) && (CLIENT_CPU == CPU_68K) //Sun3
  #define BUFFEROPEN( fn )  fopen( fn, "r+" )
  #undef BUFFEROPENCREATE
  #define BUFFERCREATE( fn )   fopen( fn, "w+" )
  extern "C" int ftruncate(int, off_t); // Keep g++ happy.
#elif ((CLIENT_OS == OS_DOS) || (CLIENT_OS == OS_WIN32) || \
       (CLIENT_OS == OS_NETWARE) || (CLIENT_OS == OS_OS2) || \
       (CLIENT_OS == OS_WIN16))
  static FILE *__bufferopenlock(const char *fn, int bCreate)
  {
    #if (CLIENT_OS == OS_NETWARE)
    DIR *dir = opendir( fn );
    if ( dir != NULL && ( readdir( dir ) != NULL ) )
    {
      dir->d_cdatetime=(dir->d_cdatetime >> 16 | dir->d_cdatetime << 16);
      dir->d_adatetime=(dir->d_adatetime >> 16 | dir->d_adatetime << 16);
      long mdatetime  =(dir->d_date | dir->d_time << 16);
      dir->d_bdatetime=(dir->d_bdatetime >> 16 | dir->d_bdatetime << 16);

      SetFileInfo( (char *)(fn), 0x06, (dir->d_attr|_A_SHARE|_A_IMMPURG),
                   (char *)(&dir->d_cdatetime), (char *)(&dir->d_adatetime),
                   (char *)(&mdatetime), (char *)(&dir->d_bdatetime),
                   dir->d_uid  );
      closedir( dir );
      //technically could now use normal fopen().
    }
    #endif
    int handle;
    if (bCreate)
      handle = sopen(fn, O_RDWR|O_BINARY|O_CREAT,
                     SH_DENYRW, S_IREAD|S_IWRITE);
    else
      handle = sopen(fn, O_RDWR|O_BINARY, SH_DENYWR);

    if (handle != -1)
    {
      FILE *f = fdopen(handle, "r+b");
      if (f)
        return f;
      close (handle);
    }
    return (FILE *)0;
  }
  #define ftruncate(h,sz) chsize(h,sz)
  #define BUFFEROPEN( fn )  __bufferopenlock( fn, 0 )
  #define BUFFEROPENCREATE( fn )  __bufferopenlock( fn, 1 )
  #define BUFFERCREATE( fn ) fopen( fn, "w+b" )
#elif defined(LOCK_EX) && defined(LOCK_NB)    // flock supported.
  static FILE *__bufferopenlock(const char *fn, int bCreate)
  {
    int handle;

    // Attempt to open/create the file in exclusive access (O_EXLOCK) mode.
    // However, support for that flag is not provided on all platforms
    // (FreeBSD supports it, but Linux does not).  Since it is only an
    // advisory OS-level lock, that is acceptable so we try to do our best
    // to use it when it is possible.
    handle = open(fn, ((bCreate)?(O_CREAT):(0)) | 
                      #ifdef O_EXLOCK
                      O_EXLOCK | 
                      #endif
                      O_RDWR, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);

    // Now that the file handle is opened, attempt to do a traditional
    // flock advisory lock on the file.  It is understood that many
    // network file system implementations do not actually provide
    // support for flock operations.  But it is only an advisory lock
    // anyways, so we'll just do our best attempt to use it anyways.
    // If we successfully obtained an advisory lock, then attach a stdio
    // stream to the handle and return the new stream.
    if (handle != -1)
    {
      if (flock(handle, LOCK_EX | LOCK_NB) >= 0 ||
          errno != EWOULDBLOCK)
      {
        FILE *fp = fdopen(handle, "r+");
        if (fp != NULL) return fp;
      }
      close(handle);
    }
    return (FILE*) 0;
  }
  #define BUFFEROPEN( fn )   __bufferopenlock( fn, 0 )
  #define BUFFEROPENCREATE( fn )   __bufferopenlock( fn, 1 )
  #define BUFFERCREATE( fn )   fopen( fn, "w+" )
#else
  #define BUFFEROPEN( fn )  fopen( fn, "r+" )
  #undef BUFFEROPENCREATE
  #define BUFFERCREATE( fn )   fopen( fn, "w+" )
#endif

#endif /* __BUFFSHIM_H__ */
