module Utils : sig val float_repres : float -> string
                      end =
  struct
    (* Imported from typing/oprint.ml *)
    let valid_float_lexeme s =
      let l = String.length s in
      let rec loop i =
        if i >= l
        then s ^ "."
        else (match s.[i] with | '0' .. '9' | '-' -> loop (i + 1) | _ -> s)
      in loop 0
      
    let float_repres f =
      match classify_float f with
      | FP_nan -> "nan"
      | FP_infinite -> if f < 0.0 then "neg_infinity" else "infinity"
      | _ ->
          let float_val =
            let s1 = Printf.sprintf "%.12g" f
            in
              if f = (float_of_string s1)
              then s1
              else
                (let s2 = Printf.sprintf "%.15g" f
                 in
                   if f = (float_of_string s2)
                   then s2
                   else Printf.sprintf "%.18g" f)
          in valid_float_lexeme float_val
      
  end
  
module Debug :
  sig
    (****************************************************************************)
    (*                                                                          *)
    (*                                   OCaml                                  *)
    (*                                                                          *)
    (*                            INRIA Rocquencourt                            *)
    (*                                                                          *)
    (*  Copyright  2006   Institut National de Recherche  en  Informatique et   *)
    (*  en Automatique.  All rights reserved.  This file is distributed under   *)
    (*  the terms of the GNU Library General Public License, with the special   *)
    (*  exception on linking described in LICENSE at the top of the Camlp4      *)
    (*  source tree.                                                            *)
    (*                                                                          *)
    (****************************************************************************)
    (* Authors:
 * - Daniel de Rauglaudre: initial version
 * - Nicolas Pouillard: refactoring
 *)
    (* camlp4r *)
    type section = string
    
    val mode : section -> bool
      
    val printf : section -> ('a, Format.formatter, unit) format -> 'a
      
  end =
  struct
    (****************************************************************************)
    (*                                                                          *)
    (*                                   OCaml                                  *)
    (*                                                                          *)
    (*                            INRIA Rocquencourt                            *)
    (*                                                                          *)
    (*  Copyright  2006   Institut National de Recherche  en  Informatique et   *)
    (*  en Automatique.  All rights reserved.  This file is distributed under   *)
    (*  the terms of the GNU Library General Public License, with the special   *)
    (*  exception on linking described in LICENSE at the top of the Camlp4      *)
    (*  source tree.                                                            *)
    (*                                                                          *)
    (****************************************************************************)
    (* Authors:
 * - Daniel de Rauglaudre: initial version
 * - Nicolas Pouillard: refactoring
 *)
    (* camlp4r *)
    open Format
      
    module Debug = struct let mode _ = false
                             end
      
    type section = string
    
    let out_channel =
      try
        let f = Sys.getenv "CAMLP4_DEBUG_FILE"
        in
          open_out_gen [ Open_wronly; Open_creat; Open_append; Open_text ]
            0o666 f
      with | Not_found -> Pervasives.stderr
      
    module StringSet = Set.Make(String)
      
    let mode =
      try
        let str = Sys.getenv "CAMLP4_DEBUG" in
        let rec loop acc i =
          try
            let pos = String.index_from str i ':'
            in
              loop (StringSet.add (String.sub str i (pos - i)) acc) (pos + 1)
          with
          | Not_found ->
              StringSet.add (String.sub str i ((String.length str) - i)) acc in
        let sections = loop StringSet.empty 0
        in
          if StringSet.mem "*" sections
          then (fun _ -> true)
          else (fun x -> StringSet.mem x sections)
      with | Not_found -> (fun _ -> false)
      
    let formatter =
      let header = "camlp4-debug: " in
      let at_bol = ref true
      in
        make_formatter
          (fun buf pos len ->
             for i = pos to (pos + len) - 1 do
               if !at_bol then output_string out_channel header else ();
               let ch = buf.[i]
               in (output_char out_channel ch; at_bol := ch = '\n')
             done)
          (fun () -> flush out_channel)
      
    let printf section fmt = fprintf formatter ("%s: " ^^ fmt) section
      
  end
  
module Options :
  sig
    (****************************************************************************)
    (*                                                                          *)
    (*                                   OCaml                                  *)
    (*                                                                          *)
    (*                            INRIA Rocquencourt                            *)
    (*                                                                          *)
    (*  Copyright  2006   Institut National de Recherche  en  Informatique et   *)
    (*  en Automatique.  All rights reserved.  This file is distributed under   *)
    (*  the terms of the GNU Library General Public License, with the special   *)
    (*  exception on linking described in LICENSE at the top of the Camlp4      *)
    (*  source tree.                                                            *)
    (*                                                                          *)
    (****************************************************************************)
    (* Authors:
 * - Daniel de Rauglaudre: initial version
 * - Nicolas Pouillard: refactoring
 *)
    type spec_list = (string * Arg.spec * string) list
    
    val init : spec_list -> unit
      
    val add : string -> Arg.spec -> string -> unit
      
    (** Add an option to the command line options. *)
    val print_usage_list : spec_list -> unit
      
    val ext_spec_list : unit -> spec_list
      
    val parse : (string -> unit) -> string array -> string list
      
  end =
  struct
    (****************************************************************************)
    (*                                                                          *)
    (*                                   OCaml                                  *)
    (*                                                                          *)
    (*                            INRIA Rocquencourt                            *)
    (*                                                                          *)
    (*  Copyright  2006   Institut National de Recherche  en  Informatique et   *)
    (*  en Automatique.  All rights reserved.  This file is distributed under   *)
    (*  the terms of the GNU Library General Public License, with the special   *)
    (*  exception on linking described in LICENSE at the top of the Camlp4      *)
    (*  source tree.                                                            *)
    (*                                                                          *)
    (****************************************************************************)
    (* Authors:
 * - Daniel de Rauglaudre: initial version
 * - Nicolas Pouillard: refactoring
 *)
    type spec_list = (string * Arg.spec * string) list
    
    open Format
      
    let rec action_arg s sl =
      function
      | Arg.Unit f -> if s = "" then (f (); Some sl) else None
      | Arg.Bool f ->
          if s = ""
          then
            (match sl with
             | s :: sl ->
                 (try (f (bool_of_string s); Some sl)
                  with | Invalid_argument _ -> None)
             | [] -> None)
          else
            (try (f (bool_of_string s); Some sl)
             with | Invalid_argument _ -> None)
      | Arg.Set r -> if s = "" then (r := true; Some sl) else None
      | Arg.Clear r -> if s = "" then (r := false; Some sl) else None
      | Arg.Rest f -> (List.iter f (s :: sl); Some [])
      | Arg.String f ->
          if s = ""
          then (match sl with | s :: sl -> (f s; Some sl) | [] -> None)
          else (f s; Some sl)
      | Arg.Set_string r ->
          if s = ""
          then (match sl with | s :: sl -> (r := s; Some sl) | [] -> None)
          else (r := s; Some sl)
      | Arg.Int f ->
          if s = ""
          then
            (match sl with
             | s :: sl ->
                 (try (f (int_of_string s); Some sl) with | Failure _ -> None)
             | [] -> None)
          else (try (f (int_of_string s); Some sl) with | Failure _ -> None)
      | Arg.Set_int r ->
          if s = ""
          then
            (match sl with
             | s :: sl ->
                 (try (r := int_of_string s; Some sl)
                  with | Failure _ -> None)
             | [] -> None)
          else (try (r := int_of_string s; Some sl) with | Failure _ -> None)
      | Arg.Float f ->
          if s = ""
          then
            (match sl with
             | s :: sl -> (f (float_of_string s); Some sl)
             | [] -> None)
          else (f (float_of_string s); Some sl)
      | Arg.Set_float r ->
          if s = ""
          then
            (match sl with
             | s :: sl -> (r := float_of_string s; Some sl)
             | [] -> None)
          else (r := float_of_string s; Some sl)
      | Arg.Tuple specs ->
          let rec action_args s sl =
            (function
             | [] -> Some sl
             | spec :: spec_list ->
                 (match action_arg s sl spec with
                  | None -> action_args "" [] spec_list
                  | Some (s :: sl) -> action_args s sl spec_list
                  | Some sl -> action_args "" sl spec_list))
          in action_args s sl specs
      | Arg.Symbol (syms, f) ->
          (match if s = "" then sl else s :: sl with
           | s :: sl when List.mem s syms -> (f s; Some sl)
           | _ -> None)
      
    let common_start s1 s2 =
      let rec loop i =
        if (i == (String.length s1)) || (i == (String.length s2))
        then i
        else if s1.[i] == s2.[i] then loop (i + 1) else i
      in loop 0
      
    let parse_arg fold s sl =
      fold
        (fun (name, action, _) acu ->
           let i = common_start s name
           in
             if i == (String.length name)
             then
               (try
                  action_arg (String.sub s i ((String.length s) - i)) sl
                    action
                with | Arg.Bad _ -> acu)
             else acu)
        None
      
    let rec parse_aux fold anon_fun =
      function
      | [] -> []
      | s :: sl ->
          if ((String.length s) > 1) && (s.[0] = '-')
          then
            (match parse_arg fold s sl with
             | Some sl -> parse_aux fold anon_fun sl
             | None -> s :: (parse_aux fold anon_fun sl))
          else ((anon_fun s : unit); parse_aux fold anon_fun sl)
      
    let align_doc key s =
      let s =
        let rec loop i =
          if i = (String.length s)
          then ""
          else
            if s.[i] = ' '
            then loop (i + 1)
            else String.sub s i ((String.length s) - i)
        in loop 0 in
      let (p, s) =
        if (String.length s) > 0
        then
          if s.[0] = '<'
          then
            (let rec loop i =
               if i = (String.length s)
               then ("", s)
               else
                 if s.[i] <> '>'
                 then loop (i + 1)
                 else
                   (let p = String.sub s 0 (i + 1) in
                    let rec loop i =
                      if i >= (String.length s)
                      then (p, "")
                      else
                        if s.[i] = ' '
                        then loop (i + 1)
                        else (p, (String.sub s i ((String.length s) - i)))
                    in loop (i + 1))
             in loop 0)
          else ("", s)
        else ("", "") in
      let tab =
        String.make (max 1 ((16 - (String.length key)) - (String.length p)))
          ' '
      in p ^ (tab ^ s)
      
    let make_symlist l =
      match l with
      | [] -> "<none>"
      | h :: t ->
          (List.fold_left (fun x y -> x ^ ("|" ^ y)) ("{" ^ h) t) ^ "}"
      
    let print_usage_list l =
      List.iter
        (fun (key, spec, doc) ->
           match spec with
           | Arg.Symbol (symbs, _) ->
               let s = make_symlist symbs in
               let synt = key ^ (" " ^ s)
               in eprintf "  %s %s\n" synt (align_doc synt doc)
           | _ -> eprintf "  %s %s\n" key (align_doc key doc))
        l
      
    let remaining_args argv =
      let rec loop l i =
        if i == (Array.length argv) then l else loop (argv.(i) :: l) (i + 1)
      in List.rev (loop [] (!Arg.current + 1))
      
    let init_spec_list = ref []
      
    let ext_spec_list = ref []
      
    let init spec_list = init_spec_list := spec_list
      
    let add name spec descr =
      ext_spec_list := (name, spec, descr) :: !ext_spec_list
      
    let fold f init =
      let spec_list = !init_spec_list @ !ext_spec_list in
      let specs =
        List.sort (fun (k1, _, _) (k2, _, _) -> String.compare k2 k1)
          spec_list
      in List.fold_right f specs init
      
    let parse anon_fun argv =
      let remaining_args = remaining_args argv
      in parse_aux fold anon_fun remaining_args
      
    let ext_spec_list () = !ext_spec_list
      
  end
  
module Sig =
  struct
    (* camlp4r *)
    (****************************************************************************)
    (*                                                                          *)
    (*                                   OCaml                                  *)
    (*                                                                          *)
    (*                            INRIA Rocquencourt                            *)
    (*                                                                          *)
    (*  Copyright  2006   Institut National de Recherche  en  Informatique et   *)
    (*  en Automatique.  All rights reserved.  This file is distributed under   *)
    (*  the terms of the GNU Library General Public License, with the special   *)
    (*  exception on linking described in LICENSE at the top of the Camlp4      *)
    (*  source tree.                                                            *)
    (*                                                                          *)
    (****************************************************************************)
    (* Authors:
 * - Daniel de Rauglaudre: initial version
 * - Nicolas Pouillard: refactoring
 *)
    (** Camlp4 signature repository *)
    (** {6 Basic signatures} *)
    (** Signature with just a type. *)
    module type Type = sig type t
                            end
      
    (** Signature for errors modules, an Error modules can be registred with
    the {!ErrorHandler.Register} functor in order to be well printed. *)
    module type Error =
      sig
        type t
        
        exception E of t
          
        val to_string : t -> string
          
        val print : Format.formatter -> t -> unit
          
      end
      
    (** A signature for extensions identifiers. *)
    module type Id =
      sig
        (** The name of the extension, typically the module name. *)
        val name : string
          
        (** The version of the extension, typically $ Id$ with a versionning system. *)
        val version : string
          
      end
      
    (** A signature for warnings abstract from locations. *)
    module Warning (Loc : Type) =
      struct
        module type S =
          sig
            type warning = Loc.t -> string -> unit
            
            val default_warning : warning
              
            val current_warning : warning ref
              
            val print_warning : warning
              
          end
          
      end
      
    (** {6 Advanced signatures} *)
    (** A signature for locations. *)
    module type Loc =
      sig
        (** The type of locations.  Note that, as for OCaml locations,
      character numbers in locations refer to character numbers in the
      parsed character stream, while line numbers refer to line
      numbers in the source file. The source file and the parsed
      character stream differ, for instance, when the parsed character
      stream contains a line number directive. The line number
      directive will only update the file-name field and the
      line-number field of the position. It makes therefore no sense
      to use character numbers with the source file if the sources
      contain line number directives. *)
        type t
        
        (** Return a start location for the given file name.
      This location starts at the begining of the file. *)
        val mk : string -> t
          
        (** The [ghost] location can be used when no location
      information is available. *)
        val ghost : t
          
        (** {6 Conversion functions} *)
        (** Return a location where both positions are set the given position. *)
        val of_lexing_position : Lexing.position -> t
          
        (** Return an OCaml location. *)
        val to_ocaml_location : t -> Location.t
          
        (** Return a location from an OCaml location. *)
        val of_ocaml_location : Location.t -> t
          
        (** Return a location from ocamllex buffer. *)
        val of_lexbuf : Lexing.lexbuf -> t
          
        (** Return a location from [(file_name, start_line, start_bol, start_off,
      stop_line,  stop_bol,  stop_off, ghost)]. *)
        val of_tuple :
          (string * int * int * int * int * int * int * bool) -> t
          
        (** Return [(file_name, start_line, start_bol, start_off,
      stop_line,  stop_bol,  stop_off, ghost)]. *)
        val to_tuple :
          t -> (string * int * int * int * int * int * int * bool)
          
        (** [merge loc1 loc2] Return a location that starts at [loc1] and end at
      [loc2]. *)
        val merge : t -> t -> t
          
        (** The stop pos becomes equal to the start pos. *)
        val join : t -> t
          
        (** [move selector n loc]
      Return the location where positions are moved.
      Affected positions are chosen with [selector].
      Returned positions have their character offset plus [n]. *)
        val move : [ | `start | `stop | `both ] -> int -> t -> t
          
        (** [shift n loc] Return the location where the new start position is the old
      stop position, and where the new stop position character offset is the
      old one plus [n]. *)
        val shift : int -> t -> t
          
        (** [move_line n loc] Return the location with the old line count plus [n].
      The "begin of line" of both positions become the current offset. *)
        val move_line : int -> t -> t
          
        (** {6 Accessors} *)
        (** Return the file name *)
        val file_name : t -> string
          
        (** Return the line number of the begining of this location. *)
        val start_line : t -> int
          
        (** Return the line number of the ending of this location. *)
        val stop_line : t -> int
          
        (** Returns the number of characters from the begining of the stream
      to the begining of the line of location's begining. *)
        val start_bol : t -> int
          
        (** Returns the number of characters from the begining of the stream
      to the begining of the line of location's ending. *)
        val stop_bol : t -> int
          
        (** Returns the number of characters from the begining of the stream
      of the begining of this location. *)
        val start_off : t -> int
          
        (** Return the number of characters from the begining of the stream
      of the ending of this location. *)
        val stop_off : t -> int
          
        (** Return the start position as a Lexing.position. *)
        val start_pos : t -> Lexing.position
          
        (** Return the stop position as a Lexing.position. *)
        val stop_pos : t -> Lexing.position
          
        (** Generally, return true if this location does not come
      from an input stream. *)
        val is_ghost : t -> bool
          
        (** Return the associated ghost location. *)
        val ghostify : t -> t
          
        (** Return the location with the give file name *)
        val set_file_name : string -> t -> t
          
        (** [strictly_before loc1 loc2] True if the stop position of [loc1] is
      strictly_before the start position of [loc2]. *)
        val strictly_before : t -> t -> bool
          
        (** Return the location with an absolute file name. *)
        val make_absolute : t -> t
          
        (** Print the location into the formatter in a format suitable for error
      reporting. *)
        val print : Format.formatter -> t -> unit
          
        (** Print the location in a short format useful for debugging. *)
        val dump : Format.formatter -> t -> unit
          
        (** Same as {!print} but return a string instead of printting it. *)
        val to_string : t -> string
          
        (** [Exc_located loc e] is an encapsulation of the exception [e] with
      the input location [loc]. To be used in quotation expanders
      and in grammars to specify some input location for an error.
      Do not raise this exception directly: rather use the following
      function [Loc.raise]. *)
        exception Exc_located of t * exn
          
        (** [raise loc e], if [e] is already an [Exc_located] exception,
      re-raise it, else raise the exception [Exc_located loc e]. *)
        val raise : t -> exn -> 'a
          
        (** The name of the location variable used in grammars and in
      the predefined quotations for OCaml syntax trees. Default: [_loc]. *)
        val name : string ref
          
      end
      
    (** Abstract syntax tree minimal signature.
    Types of this signature are abstract.
    See the {!Camlp4Ast} signature for a concrete definition. *)
    module type Ast =
      sig
        (** {6 Syntactic categories as abstract types} *)
        type loc
        
        type meta_bool
        
        type 'a meta_option
        
        type 'a meta_list
        
        type ctyp
        
        type patt
        
        type expr
        
        type module_type
        
        type sig_item
        
        type with_constr
        
        type module_expr
        
        type str_item
        
        type class_type
        
        type class_sig_item
        
        type class_expr
        
        type class_str_item
        
        type match_case
        
        type ident
        
        type binding
        
        type rec_binding
        
        type module_binding
        
        type rec_flag
        
        type direction_flag
        
        type mutable_flag
        
        type private_flag
        
        type virtual_flag
        
        type row_var_flag
        
        type override_flag
        
        (** {6 Location accessors} *)
        val loc_of_ctyp : ctyp -> loc
          
        val loc_of_patt : patt -> loc
          
        val loc_of_expr : expr -> loc
          
        val loc_of_module_type : module_type -> loc
          
        val loc_of_module_expr : module_expr -> loc
          
        val loc_of_sig_item : sig_item -> loc
          
        val loc_of_str_item : str_item -> loc
          
        val loc_of_class_type : class_type -> loc
          
        val loc_of_class_sig_item : class_sig_item -> loc
          
        val loc_of_class_expr : class_expr -> loc
          
        val loc_of_class_str_item : class_str_item -> loc
          
        val loc_of_with_constr : with_constr -> loc
          
        val loc_of_binding : binding -> loc
          
        val loc_of_rec_binding : rec_binding -> loc
          
        val loc_of_module_binding : module_binding -> loc
          
        val loc_of_match_case : match_case -> loc
          
        val loc_of_ident : ident -> loc
          
        (** {6 Traversals} *)
        (** This class is the base class for map traversal on the Ast.
      To make a custom traversal class one just extend it like that:

      This example swap pairs expression contents:
      open Camlp4.PreCast;
      [class swap = object
        inherit Ast.map as super;
        method expr e =
          match super#expr e with
          \[ <:expr\@_loc< ($e1$, $e2$) >> -> <:expr< ($e2$, $e1$) >>
          | e -> e \];
      end;
      value _loc = Loc.ghost;
      value map = (new swap)#expr;
      assert (map <:expr< fun x -> (x, 42) >> = <:expr< fun x -> (42, x) >>);]
  *)
        class map :
          object ('self_type)
            method string : string -> string
            method list :
              'a 'b. ('self_type -> 'a -> 'b) -> 'a list -> 'b list
            method meta_bool : meta_bool -> meta_bool
            method meta_option :
              'a 'b.
                ('self_type -> 'a -> 'b) -> 'a meta_option -> 'b meta_option
            method meta_list :
              'a 'b. ('self_type -> 'a -> 'b) -> 'a meta_list -> 'b meta_list
            method loc : loc -> loc
            method expr : expr -> expr
            method patt : patt -> patt
            method ctyp : ctyp -> ctyp
            method str_item : str_item -> str_item
            method sig_item : sig_item -> sig_item
            method module_expr : module_expr -> module_expr
            method module_type : module_type -> module_type
            method class_expr : class_expr -> class_expr
            method class_type : class_type -> class_type
            method class_sig_item : class_sig_item -> class_sig_item
            method class_str_item : class_str_item -> class_str_item
            method with_constr : with_constr -> with_constr
            method binding : binding -> binding
            method rec_binding : rec_binding -> rec_binding
            method module_binding : module_binding -> module_binding
            method match_case : match_case -> match_case
            method ident : ident -> ident
            method override_flag : override_flag -> override_flag
            method mutable_flag : mutable_flag -> mutable_flag
            method private_flag : private_flag -> private_flag
            method virtual_flag : virtual_flag -> virtual_flag
            method direction_flag : direction_flag -> direction_flag
            method rec_flag : rec_flag -> rec_flag
            method row_var_flag : row_var_flag -> row_var_flag
            method unknown : 'a. 'a -> 'a
          end
          
        (** Fold style traversal *)
        class fold :
          object ('self_type)
            method string : string -> 'self_type
            method list :
              'a. ('self_type -> 'a -> 'self_type) -> 'a list -> 'self_type
            method meta_bool : meta_bool -> 'self_type
            method meta_option :
              'a.
                ('self_type -> 'a -> 'self_type) ->
                  'a meta_option -> 'self_type
            method meta_list :
              'a.
                ('self_type -> 'a -> 'self_type) ->
                  'a meta_list -> 'self_type
            method loc : loc -> 'self_type
            method expr : expr -> 'self_type
            method patt : patt -> 'self_type
            method ctyp : ctyp -> 'self_type
            method str_item : str_item -> 'self_type
            method sig_item : sig_item -> 'self_type
            method module_expr : module_expr -> 'self_type
            method module_type : module_type -> 'self_type
            method class_expr : class_expr -> 'self_type
            method class_type : class_type -> 'self_type
            method class_sig_item : class_sig_item -> 'self_type
            method class_str_item : class_str_item -> 'self_type
            method with_constr : with_constr -> 'self_type
            method binding : binding -> 'self_type
            method rec_binding : rec_binding -> 'self_type
            method module_binding : module_binding -> 'self_type
            method match_case : match_case -> 'self_type
            method ident : ident -> 'self_type
            method rec_flag : rec_flag -> 'self_type
            method direction_flag : direction_flag -> 'self_type
            method mutable_flag : mutable_flag -> 'self_type
            method private_flag : private_flag -> 'self_type
            method virtual_flag : virtual_flag -> 'self_type
            method row_var_flag : row_var_flag -> 'self_type
            method override_flag : override_flag -> 'self_type
            method unknown : 'a. 'a -> 'self_type
          end
          
      end
      
    (** Signature for OCaml syntax trees. *)
    (*
    This signature is an extension of {!Ast}
    It provides:
      - Types for all kinds of structure.
      - Map: A base class for map traversals.
      - Map classes and functions for common kinds.

    == Core language ==
    ctyp               :: Representaion of types
    patt               :: The type of patterns
    expr               :: The type of expressions
    match_case         :: The type of cases for match/function/try constructions
    ident              :: The type of identifiers (including path like Foo(X).Bar.y)
    binding            :: The type of let bindings
    rec_binding        :: The type of record definitions

    == Modules ==
    module_type        :: The type of module types
    sig_item           :: The type of signature items
    str_item           :: The type of structure items
    module_expr        :: The type of module expressions
    module_binding     :: The type of recursive module definitions
    with_constr        :: The type of `with' constraints

    == Classes ==
    class_type         :: The type of class types
    class_sig_item     :: The type of class signature items
    class_expr         :: The type of class expressions
    class_str_item     :: The type of class structure items
 *)
    module type Camlp4Ast =
      sig
        (** The inner module for locations *)
        module Loc : Loc
          
        (****************************************************************************)
        (*                                                                          *)
        (*                                   OCaml                                  *)
        (*                                                                          *)
        (*                            INRIA Rocquencourt                            *)
        (*                                                                          *)
        (*  Copyright  2007   Institut National de Recherche  en  Informatique et   *)
        (*  en Automatique.  All rights reserved.  This file is distributed under   *)
        (*  the terms of the GNU Library General Public License, with the special   *)
        (*  exception on linking described in LICENSE at the top of the Camlp4      *)
        (*  source tree.                                                            *)
        (*                                                                          *)
        (****************************************************************************)
        (* Note: when you modify these types you must increment
   ast magic numbers defined in Camlp4_config.ml. *)
        type loc =
          Loc.
          t
          and meta_bool =
          | BTrue | BFalse | BAnt of string
          and rec_flag =
          | ReRecursive | ReNonrecursive | ReNil | ReAnt of string
          and direction_flag =
          | DiTo | DiDownto | DiAnt of string
          and mutable_flag =
          | MuMutable | MuNil | MuAnt of string
          and private_flag =
          | PrPrivate | PrNil | PrAnt of string
          and virtual_flag =
          | ViVirtual | ViNil | ViAnt of string
          and override_flag =
          | OvOverride | OvNil | OvAnt of string
          and row_var_flag =
          | RvRowVar | RvNil | RvAnt of string
          and 'a meta_option =
          | ONone | OSome of 'a | OAnt of string
          and 'a meta_list =
          | LNil | LCons of 'a * 'a meta_list | LAnt of string
          and ident =
          | IdAcc of loc * ident * ident
          | (* i . i *)
          IdApp of loc * ident * ident
          | (* i i *)
          IdLid of loc * string
          | (* foo *)
          IdUid of loc * string
          | (* Bar *)
          IdAnt of loc * string
          and (* $s$ *)
          ctyp =
          | TyNil of loc
          | TyAli of loc * ctyp * ctyp
          | (* t as t *)
          (* list 'a as 'a *)
          TyAny of loc
          | (* _ *)
          TyApp of loc * ctyp * ctyp
          | (* t t *)
          (* list 'a *)
          TyArr of loc * ctyp * ctyp
          | (* t -> t *)
          (* int -> string *)
          TyCls of loc * ident
          | (* #i *)
          (* #point *)
          TyLab of loc * string * ctyp
          | (* ~s:t *)
          TyId of loc * ident
          | (* i *)
          (* Lazy.t *)
          TyMan of loc * ctyp * ctyp
          | (* t == t *)
          (* type t = [ A | B ] == Foo.t *)
          (* type t 'a 'b 'c = t constraint t = t constraint t = t *)
          TyDcl of loc * string * ctyp list * ctyp * (ctyp * ctyp) list
          | (* type t 'a 'b 'c += A *)
          TyExt of loc * ident * ctyp list * ctyp
          | (* < (t)? (..)? > *)
          (* < move : int -> 'a .. > as 'a  *)
          TyObj of loc * ctyp * row_var_flag
          | TyOlb of loc * string * ctyp
          | (* ?s:t *)
          TyPol of loc * ctyp * ctyp
          | (* ! t . t *)
          (* ! 'a . list 'a -> 'a *)
          TyTypePol of loc * ctyp * ctyp
          | (* type t . t *)
          (* type a . list a -> a *)
          TyQuo of loc * string
          | (* 's *)
          TyQuP of loc * string
          | (* +'s *)
          TyQuM of loc * string
          | (* -'s *)
          TyAnP of loc
          | (* +_ *)
          TyAnM of loc
          | (* -_ *)
          TyVrn of loc * string
          | (* `s *)
          TyRec of loc * ctyp
          | (* { t } *)
          (* { foo : int ; bar : mutable string } *)
          TyCol of loc * ctyp * ctyp
          | (* t : t *)
          TySem of loc * ctyp * ctyp
          | (* t; t *)
          TyCom of loc * ctyp * ctyp
          | (* t, t *)
          TySum of loc * ctyp
          | (* [ t ] *)
          (* [ A of int and string | B ] *)
          TyOf of loc * ctyp * ctyp
          | (* t of t *)
          (* A of int *)
          TyAnd of loc * ctyp * ctyp
          | (* t and t *)
          TyOr of loc * ctyp * ctyp
          | (* t | t *)
          TyPrv of loc * ctyp
          | (* private t *)
          TyMut of loc * ctyp
          | (* mutable t *)
          TyTup of loc * ctyp
          | (* ( t ) *)
          (* (int * string) *)
          TySta of loc * ctyp * ctyp
          | (* t * t *)
          TyVrnEq of loc * ctyp
          | (* [ = t ] *)
          TyVrnSup of loc * ctyp
          | (* [ > t ] *)
          TyVrnInf of loc * ctyp
          | (* [ < t ] *)
          TyVrnInfSup of loc * ctyp * ctyp
          | (* [ < t > t ] *)
          TyAmp of loc * ctyp * ctyp
          | (* t & t *)
          TyOfAmp of loc * ctyp * ctyp
          | (* t of & t *)
          TyPkg of loc * module_type
          | (* (module S) *)
          TyOpn of loc
          | (* .. *)
          TyAtt of loc * string * str_item * ctyp
          | (* .. [@attr] *)
          TyAnt of loc * string
          and (* $s$ *)
          patt =
          | PaNil of loc
          | PaId of loc * ident
          | (* i *)
          PaAli of loc * patt * patt
          | (* p as p *)
          (* (Node x y as n) *)
          PaAnt of loc * string
          | (* $s$ *)
          PaAny of loc
          | (* _ *)
          PaApp of loc * patt * patt
          | (* p p *)
          (* fun x y -> *)
          PaArr of loc * patt
          | (* [| p |] *)
          PaCom of loc * patt * patt
          | (* p, p *)
          PaSem of loc * patt * patt
          | (* p; p *)
          PaChr of loc * string
          | (* c *)
          (* 'x' *)
          PaInt of loc * string
          | PaInt32 of loc * string
          | PaInt64 of loc * string
          | PaNativeInt of loc * string
          | PaFlo of loc * string
          | PaLab of loc * string * patt
          | (* ~s or ~s:(p) *)
          (* ?s or ?s:(p) *)
          PaOlb of loc * string * patt
          | (* ?s:(p = e) or ?(p = e) *)
          PaOlbi of loc * string * patt * expr
          | PaOrp of loc * patt * patt
          | (* p | p *)
          PaRng of loc * patt * patt
          | (* p .. p *)
          PaRec of loc * patt
          | (* { p } *)
          PaEq of loc * ident * patt
          | (* i = p *)
          PaStr of loc * string
          | (* s *)
          PaTup of loc * patt
          | (* ( p ) *)
          PaTyc of loc * patt * ctyp
          | (* (p : t) *)
          PaTyp of loc * ident
          | (* #i *)
          PaVrn of loc * string
          | (* `s *)
          PaLaz of loc * patt
          | (* lazy p *)
          PaAtt of loc * string * str_item * patt
          | (* .. [@attr] *)
          PaMod of loc * string
          | (* (module M) *)
          PaExc of loc * patt
          and (* exception p *)
          expr =
          | ExNil of loc
          | ExId of loc * ident
          | (* i *)
          ExAcc of loc * expr * expr
          | (* e.e *)
          ExAnt of loc * string
          | (* $s$ *)
          ExApp of loc * expr * expr
          | (* e e *)
          ExAre of loc * expr * expr
          | (* e.(e) *)
          ExArr of loc * expr
          | (* [| e |] *)
          ExSem of loc * expr * expr
          | (* e; e *)
          ExAsf of loc
          | (* assert False *)
          ExAsr of loc * expr
          | (* assert e *)
          ExAss of loc * expr * expr
          | (* e := e *)
          ExChr of loc * string
          | (* 'c' *)
          ExCoe of loc * expr * ctyp * ctyp
          | (* (e : t) or (e : t :> t) *)
          ExFlo of loc * string
          | (* 3.14 *)
          (* for s = e to/downto e do { e } *)
          ExFor of loc * patt * expr * expr * direction_flag * expr
          | ExFun of loc * match_case
          | (* fun [ mc ] *)
          ExIfe of loc * expr * expr * expr
          | (* if e then e else e *)
          ExInt of loc * string
          | (* 42 *)
          ExInt32 of loc * string
          | ExInt64 of loc * string
          | ExNativeInt of loc * string
          | ExLab of loc * string * expr
          | (* ~s or ~s:e *)
          ExLaz of loc * expr
          | (* lazy e *)
          (* let b in e or let rec b in e *)
          ExLet of loc * rec_flag * binding * expr
          | (* let module s = me in e *)
          ExLmd of loc * string * module_expr * expr
          | (* match e with [ mc ] *)
          ExMat of loc * expr * match_case
          | (* new i *)
          ExNew of loc * ident
          | (* object ((p))? (cst)? end *)
          ExObj of loc * patt * class_str_item
          | (* ?s or ?s:e *)
          ExOlb of loc * string * expr
          | (* {< rb >} *)
          ExOvr of loc * rec_binding
          | (* { rb } or { (e) with rb } *)
          ExRec of loc * rec_binding * expr
          | (* do { e } *)
          ExSeq of loc * expr
          | (* e#s *)
          ExSnd of loc * expr * string
          | (* e.[e] *)
          ExSte of loc * expr * expr
          | (* s *)
          (* "foo" *)
          ExStr of loc * string
          | (* try e with [ mc ] *)
          ExTry of loc * expr * match_case
          | (* (e) *)
          ExTup of loc * expr
          | (* e, e *)
          ExCom of loc * expr * expr
          | (* (e : t) *)
          ExTyc of loc * expr * ctyp
          | (* `s *)
          ExVrn of loc * string
          | (* while e do { e } *)
          ExWhi of loc * expr * expr
          | (* let open i in e *)
          ExOpI of loc * ident * override_flag * expr
          | (* fun (type t) -> e *)
          (* let f x (type t) y z = e *)
          ExFUN of loc * string * expr
          | (* (module ME : S) which is represented as (module (ME : S)) *)
          ExPkg of loc * module_expr
          | (* e [@attr] *)
          ExAtt of loc * string * str_item * expr
          and module_type =
          | MtNil of loc
          | (* i *)
          (* A.B.C *)
          MtId of loc * ident
          | (* (module ident) *)
          MtAlias of loc * ident
          | (* functor (s : mt) -> mt *)
          MtFun of loc * string * module_type * module_type
          | (* 's *)
          MtQuo of loc * string
          | (* sig sg end *)
          MtSig of loc * sig_item
          | (* mt with wc *)
          MtWit of loc * module_type * with_constr
          | (* module type of m *)
          MtOf of loc * module_expr
          | MtAtt of loc * string * str_item * module_type
          | (* .. [@attr] *)
          MtAnt of loc * string
          and (* $s$ *)
          sig_item =
          | SgNil of loc
          | (* class cict *)
          SgCls of loc * class_type
          | (* class type cict *)
          SgClt of loc * class_type
          | (* sg ; sg *)
          SgSem of loc * sig_item * sig_item
          | (* # s or # s e *)
          SgDir of loc * string * expr
          | (* exception t *)
          SgExc of loc * ctyp
          | (* external s : t = s ... s *)
          SgExt of loc * string * ctyp * string meta_list
          | (* include mt *)
          SgInc of loc * module_type
          | (* module s : mt *)
          SgMod of loc * string * module_type
          | (* module rec mb *)
          SgRecMod of loc * module_binding
          | (* module type s = mt *)
          SgMty of loc * string * module_type
          | (* open i *)
          SgOpn of loc * override_flag * ident
          | (* type t *)
          SgTyp of loc * rec_flag * ctyp
          | (* value s : t *)
          SgVal of loc * string * ctyp
          | SgAnt of loc * string
          and (* $s$ *)
          with_constr =
          | WcNil of loc
          | (* type t = t *)
          WcTyp of loc * ctyp * ctyp
          | (* module i = i *)
          WcMod of loc * ident * ident
          | (* type t := t *)
          WcTyS of loc * ctyp * ctyp
          | (* module i := i *)
          WcMoS of loc * ident * ident
          | (* wc and wc *)
          WcAnd of loc * with_constr * with_constr
          | WcAnt of loc * string
          and (* $s$ *)
          binding =
          | BiNil of loc
          | (* bi and bi *)
          (* let a = 42 and c = 43 *)
          BiAnd of loc * binding * binding
          | (* p = e *)
          (* let patt = expr *)
          BiEq of loc * patt * expr
          | BiAnt of loc * string
          and (* $s$ *)
          rec_binding =
          | RbNil of loc
          | (* rb ; rb *)
          RbSem of loc * rec_binding * rec_binding
          | (* i = e *)
          RbEq of loc * ident * expr
          | RbAnt of loc * string
          and (* $s$ *)
          module_binding =
          | MbNil of loc
          | (* mb and mb *)
          (* module rec (s : mt) = me and (s : mt) = me *)
          MbAnd of loc * module_binding * module_binding
          | (* s : mt = me *)
          MbColEq of loc * string * module_type * module_expr
          | (* s : mt *)
          MbCol of loc * string * module_type
          | MbAnt of loc * string
          and (* $s$ *)
          match_case =
          | McNil of loc
          | (* a | a *)
          McOr of loc * match_case * match_case
          | (* p (when e)? -> e *)
          McArr of loc * patt * expr * expr
          | McAnt of loc * string
          and (* $s$ *)
          module_expr =
          | MeNil of loc
          | (* i *)
          MeId of loc * ident
          | (* me me *)
          MeApp of loc * module_expr * module_expr
          | (* functor (s : mt) -> me *)
          MeFun of loc * string * module_type * module_expr
          | (* struct st end *)
          MeStr of loc * str_item
          | (* (me : mt) *)
          MeTyc of loc * module_expr * module_type
          | (* (value e) *)
          (* (value e : S) which is represented as (value (e : S)) *)
          MePkg of loc * expr
          | MeAtt of loc * string * str_item * module_expr
          | (* .. [@attr] *)
          MeAnt of loc * string
          and (* $s$ *)
          str_item =
          | StNil of loc
          | (* class cice *)
          StCls of loc * class_expr
          | (* class type cict *)
          StClt of loc * class_type
          | (* st ; st *)
          StSem of loc * str_item * str_item
          | (* # s or # s e *)
          StDir of loc * string * expr
          | (* exception t or exception t = i *)
          StExc of loc * ctyp * (*FIXME*) ident meta_option
          | (* e *)
          StExp of loc * expr
          | (* external s : t = s ... s *)
          StExt of loc * string * ctyp * string meta_list
          | (* include me *)
          StInc of loc * module_expr
          | (* module s = me *)
          StMod of loc * string * module_expr
          | (* module rec mb *)
          StRecMod of loc * module_binding
          | (* module type s = mt *)
          StMty of loc * string * module_type
          | (* open i *)
          StOpn of loc * override_flag * ident
          | (* type t *)
          StTyp of loc * rec_flag * ctyp
          | (* value (rec)? bi *)
          StVal of loc * rec_flag * binding
          | StAnt of loc * string
          and (* $s$ *)
          class_type =
          | CtNil of loc
          | (* (virtual)? i ([ t ])? *)
          CtCon of loc * virtual_flag * ident * ctyp
          | (* [t] -> ct *)
          CtFun of loc * ctyp * class_type
          | (* object ((t))? (csg)? end *)
          CtSig of loc * ctyp * class_sig_item
          | (* ct and ct *)
          CtAnd of loc * class_type * class_type
          | (* ct : ct *)
          CtCol of loc * class_type * class_type
          | (* ct = ct *)
          CtEq of loc * class_type * class_type
          | (* $s$ *)
          CtAtt of loc * string * str_item * class_type
          | (* .. [@attr] *)
          CtAnt of loc * string
          and class_sig_item =
          | CgNil of loc
          | (* type t = t *)
          CgCtr of loc * ctyp * ctyp
          | (* csg ; csg *)
          CgSem of loc * class_sig_item * class_sig_item
          | (* inherit ct *)
          CgInh of loc * class_type
          | (* method s : t or method private s : t *)
          CgMth of loc * string * private_flag * ctyp
          | (* value (virtual)? (mutable)? s : t *)
          CgVal of loc * string * mutable_flag * virtual_flag * ctyp
          | (* method virtual (private)? s : t *)
          CgVir of loc * string * private_flag * ctyp
          | CgAnt of loc * string
          and (* $s$ *)
          class_expr =
          | CeNil of loc
          | (* ce e *)
          CeApp of loc * class_expr * expr
          | (* (virtual)? i ([ t ])? *)
          CeCon of loc * virtual_flag * ident * ctyp
          | (* fun p -> ce *)
          CeFun of loc * patt * class_expr
          | (* let (rec)? bi in ce *)
          CeLet of loc * rec_flag * binding * class_expr
          | (* object ((p))? (cst)? end *)
          CeStr of loc * patt * class_str_item
          | (* ce : ct *)
          CeTyc of loc * class_expr * class_type
          | (* ce and ce *)
          CeAnd of loc * class_expr * class_expr
          | (* ce = ce *)
          CeEq of loc * class_expr * class_expr
          | (* $s$ *)
          CeAtt of loc * string * str_item * class_expr
          | (* .. [@attr] *)
          CeAnt of loc * string
          and class_str_item =
          | CrNil of loc
          | (* cst ; cst *)
          CrSem of loc * class_str_item * class_str_item
          | (* type t = t *)
          CrCtr of loc * ctyp * ctyp
          | (* inherit(!)? ce (as s)? *)
          CrInh of loc * override_flag * class_expr * string
          | (* initializer e *)
          CrIni of loc * expr
          | (* method(!)? (private)? s : t = e or method(!)? (private)? s = e *)
          CrMth of loc * string * override_flag * private_flag * expr * ctyp
          | (* value(!)? (mutable)? s = e *)
          CrVal of loc * string * override_flag * mutable_flag * expr
          | (* method virtual (private)? s : t *)
          CrVir of loc * string * private_flag * ctyp
          | (* value virtual (mutable)? s : t *)
          CrVvr of loc * string * mutable_flag * ctyp
          | CrAnt of loc * string
        
        val loc_of_ctyp : ctyp -> loc
          
        val loc_of_patt : patt -> loc
          
        val loc_of_expr : expr -> loc
          
        val loc_of_module_type : module_type -> loc
          
        val loc_of_module_expr : module_expr -> loc
          
        val loc_of_sig_item : sig_item -> loc
          
        val loc_of_str_item : str_item -> loc
          
        val loc_of_class_type : class_type -> loc
          
        val loc_of_class_sig_item : class_sig_item -> loc
          
        val loc_of_class_expr : class_expr -> loc
          
        val loc_of_class_str_item : class_str_item -> loc
          
        val loc_of_with_constr : with_constr -> loc
          
        val loc_of_binding : binding -> loc
          
        val loc_of_rec_binding : rec_binding -> loc
          
        val loc_of_module_binding : module_binding -> loc
          
        val loc_of_match_case : match_case -> loc
          
        val loc_of_ident : ident -> loc
          
        module Meta :
          sig
            module type META_LOC =
              sig
                val meta_loc_patt : loc -> loc -> patt
                  
                val meta_loc_expr : loc -> loc -> expr
                  
              end
              
            module MetaLoc :
              sig
                val meta_loc_patt : loc -> loc -> patt
                  
                val meta_loc_expr : loc -> loc -> expr
                  
              end
              
            module MetaGhostLoc :
              sig
                val meta_loc_patt : loc -> 'a -> patt
                  
                val meta_loc_expr : loc -> 'a -> expr
                  
              end
              
            module MetaLocVar :
              sig
                val meta_loc_patt : loc -> 'a -> patt
                  
                val meta_loc_expr : loc -> 'a -> expr
                  
              end
              
            module Make (MetaLoc : META_LOC) :
              sig
                module Expr :
                  sig
                    val meta_string : loc -> string -> expr
                      
                    val meta_int : loc -> string -> expr
                      
                    val meta_float : loc -> string -> expr
                      
                    val meta_char : loc -> string -> expr
                      
                    val meta_bool : loc -> bool -> expr
                      
                    val meta_list :
                      (loc -> 'a -> expr) -> loc -> 'a list -> expr
                      
                    val meta_binding : loc -> binding -> expr
                      
                    val meta_rec_binding : loc -> rec_binding -> expr
                      
                    val meta_class_expr : loc -> class_expr -> expr
                      
                    val meta_class_sig_item : loc -> class_sig_item -> expr
                      
                    val meta_class_str_item : loc -> class_str_item -> expr
                      
                    val meta_class_type : loc -> class_type -> expr
                      
                    val meta_ctyp : loc -> ctyp -> expr
                      
                    val meta_expr : loc -> expr -> expr
                      
                    val meta_ident : loc -> ident -> expr
                      
                    val meta_match_case : loc -> match_case -> expr
                      
                    val meta_module_binding : loc -> module_binding -> expr
                      
                    val meta_module_expr : loc -> module_expr -> expr
                      
                    val meta_module_type : loc -> module_type -> expr
                      
                    val meta_patt : loc -> patt -> expr
                      
                    val meta_sig_item : loc -> sig_item -> expr
                      
                    val meta_str_item : loc -> str_item -> expr
                      
                    val meta_with_constr : loc -> with_constr -> expr
                      
                    val meta_rec_flag : loc -> rec_flag -> expr
                      
                    val meta_mutable_flag : loc -> mutable_flag -> expr
                      
                    val meta_virtual_flag : loc -> virtual_flag -> expr
                      
                    val meta_private_flag : loc -> private_flag -> expr
                      
                    val meta_row_var_flag : loc -> row_var_flag -> expr
                      
                    val meta_override_flag : loc -> override_flag -> expr
                      
                    val meta_direction_flag : loc -> direction_flag -> expr
                      
                  end
                  
                module Patt :
                  sig
                    val meta_string : loc -> string -> patt
                      
                    val meta_int : loc -> string -> patt
                      
                    val meta_float : loc -> string -> patt
                      
                    val meta_char : loc -> string -> patt
                      
                    val meta_bool : loc -> bool -> patt
                      
                    val meta_list :
                      (loc -> 'a -> patt) -> loc -> 'a list -> patt
                      
                    val meta_binding : loc -> binding -> patt
                      
                    val meta_rec_binding : loc -> rec_binding -> patt
                      
                    val meta_class_expr : loc -> class_expr -> patt
                      
                    val meta_class_sig_item : loc -> class_sig_item -> patt
                      
                    val meta_class_str_item : loc -> class_str_item -> patt
                      
                    val meta_class_type : loc -> class_type -> patt
                      
                    val meta_ctyp : loc -> ctyp -> patt
                      
                    val meta_expr : loc -> expr -> patt
                      
                    val meta_ident : loc -> ident -> patt
                      
                    val meta_match_case : loc -> match_case -> patt
                      
                    val meta_module_binding : loc -> module_binding -> patt
                      
                    val meta_module_expr : loc -> module_expr -> patt
                      
                    val meta_module_type : loc -> module_type -> patt
                      
                    val meta_patt : loc -> patt -> patt
                      
                    val meta_sig_item : loc -> sig_item -> patt
                      
                    val meta_str_item : loc -> str_item -> patt
                      
                    val meta_with_constr : loc -> with_constr -> patt
                      
                    val meta_rec_flag : loc -> rec_flag -> patt
                      
                    val meta_mutable_flag : loc -> mutable_flag -> patt
                      
                    val meta_virtual_flag : loc -> virtual_flag -> patt
                      
                    val meta_private_flag : loc -> private_flag -> patt
                      
                    val meta_row_var_flag : loc -> row_var_flag -> patt
                      
                    val meta_override_flag : loc -> override_flag -> patt
                      
                    val meta_direction_flag : loc -> direction_flag -> patt
                      
                  end
                  
              end
              
          end
          
        class map :
          object ('self_type)
            method string : string -> string
            method list :
              'a 'b. ('self_type -> 'a -> 'b) -> 'a list -> 'b list
            method meta_bool : meta_bool -> meta_bool
            method meta_option :
              'a 'b.
                ('self_type -> 'a -> 'b) -> 'a meta_option -> 'b meta_option
            method meta_list :
              'a 'b. ('self_type -> 'a -> 'b) -> 'a meta_list -> 'b meta_list
            method loc : loc -> loc
            method expr : expr -> expr
            method patt : patt -> patt
            method ctyp : ctyp -> ctyp
            method str_item : str_item -> str_item
            method sig_item : sig_item -> sig_item
            method module_expr : module_expr -> module_expr
            method module_type : module_type -> module_type
            method class_expr : class_expr -> class_expr
            method class_type : class_type -> class_type
            method class_sig_item : class_sig_item -> class_sig_item
            method class_str_item : class_str_item -> class_str_item
            method with_constr : with_constr -> with_constr
            method binding : binding -> binding
            method rec_binding : rec_binding -> rec_binding
            method module_binding : module_binding -> module_binding
            method match_case : match_case -> match_case
            method ident : ident -> ident
            method mutable_flag : mutable_flag -> mutable_flag
            method private_flag : private_flag -> private_flag
            method virtual_flag : virtual_flag -> virtual_flag
            method direction_flag : direction_flag -> direction_flag
            method rec_flag : rec_flag -> rec_flag
            method row_var_flag : row_var_flag -> row_var_flag
            method override_flag : override_flag -> override_flag
            method unknown : 'a. 'a -> 'a
          end
          
        class fold :
          object ('self_type)
            method string : string -> 'self_type
            method list :
              'a. ('self_type -> 'a -> 'self_type) -> 'a list -> 'self_type
            method meta_bool : meta_bool -> 'self_type
            method meta_option :
              'a.
                ('self_type -> 'a -> 'self_type) ->
                  'a meta_option -> 'self_type
            method meta_list :
              'a.
                ('self_type -> 'a -> 'self_type) ->
                  'a meta_list -> 'self_type
            method loc : loc -> 'self_type
            method expr : expr -> 'self_type
            method patt : patt -> 'self_type
            method ctyp : ctyp -> 'self_type
            method str_item : str_item -> 'self_type
            method sig_item : sig_item -> 'self_type
            method module_expr : module_expr -> 'self_type
            method module_type : module_type -> 'self_type
            method class_expr : class_expr -> 'self_type
            method class_type : class_type -> 'self_type
            method class_sig_item : class_sig_item -> 'self_type
            method class_str_item : class_str_item -> 'self_type
            method with_constr : with_constr -> 'self_type
            method binding : binding -> 'self_type
            method rec_binding : rec_binding -> 'self_type
            method module_binding : module_binding -> 'self_type
            method match_case : match_case -> 'self_type
            method ident : ident -> 'self_type
            method rec_flag : rec_flag -> 'self_type
            method direction_flag : direction_flag -> 'self_type
            method mutable_flag : mutable_flag -> 'self_type
            method private_flag : private_flag -> 'self_type
            method virtual_flag : virtual_flag -> 'self_type
            method row_var_flag : row_var_flag -> 'self_type
            method override_flag : override_flag -> 'self_type
            method unknown : 'a. 'a -> 'self_type
          end
          
        val map_expr : (expr -> expr) -> map
          
        val map_patt : (patt -> patt) -> map
          
        val map_ctyp : (ctyp -> ctyp) -> map
          
        val map_str_item : (str_item -> str_item) -> map
          
        val map_sig_item : (sig_item -> sig_item) -> map
          
        val map_loc : (loc -> loc) -> map
          
        val ident_of_expr : expr -> ident
          
        val ident_of_patt : patt -> ident
          
        val ident_of_ctyp : ctyp -> ident
          
        val biAnd_of_list : binding list -> binding
          
        val rbSem_of_list : rec_binding list -> rec_binding
          
        val paSem_of_list : patt list -> patt
          
        val paCom_of_list : patt list -> patt
          
        val tyOr_of_list : ctyp list -> ctyp
          
        val tyAnd_of_list : ctyp list -> ctyp
          
        val tyAmp_of_list : ctyp list -> ctyp
          
        val tySem_of_list : ctyp list -> ctyp
          
        val tyCom_of_list : ctyp list -> ctyp
          
        val tySta_of_list : ctyp list -> ctyp
          
        val stSem_of_list : str_item list -> str_item
          
        val sgSem_of_list : sig_item list -> sig_item
          
        val crSem_of_list : class_str_item list -> class_str_item
          
        val cgSem_of_list : class_sig_item list -> class_sig_item
          
        val ctAnd_of_list : class_type list -> class_type
          
        val ceAnd_of_list : class_expr list -> class_expr
          
        val wcAnd_of_list : with_constr list -> with_constr
          
        val meApp_of_list : module_expr list -> module_expr
          
        val mbAnd_of_list : module_binding list -> module_binding
          
        val mcOr_of_list : match_case list -> match_case
          
        val idAcc_of_list : ident list -> ident
          
        val idApp_of_list : ident list -> ident
          
        val exSem_of_list : expr list -> expr
          
        val exCom_of_list : expr list -> expr
          
        val list_of_ctyp : ctyp -> ctyp list -> ctyp list
          
        val list_of_binding : binding -> binding list -> binding list
          
        val list_of_rec_binding :
          rec_binding -> rec_binding list -> rec_binding list
          
        val list_of_with_constr :
          with_constr -> with_constr list -> with_constr list
          
        val list_of_patt : patt -> patt list -> patt list
          
        val list_of_expr : expr -> expr list -> expr list
          
        val list_of_str_item : str_item -> str_item list -> str_item list
          
        val list_of_sig_item : sig_item -> sig_item list -> sig_item list
          
        val list_of_class_sig_item :
          class_sig_item -> class_sig_item list -> class_sig_item list
          
        val list_of_class_str_item :
          class_str_item -> class_str_item list -> class_str_item list
          
        val list_of_class_type :
          class_type -> class_type list -> class_type list
          
        val list_of_class_expr :
          class_expr -> class_expr list -> class_expr list
          
        val list_of_module_expr :
          module_expr -> module_expr list -> module_expr list
          
        val list_of_module_binding :
          module_binding -> module_binding list -> module_binding list
          
        val list_of_match_case :
          match_case -> match_case list -> match_case list
          
        val list_of_ident : ident -> ident list -> ident list
          
        val safe_string_escaped : string -> string
          
        val is_irrefut_patt : patt -> bool
          
        val is_constructor : ident -> bool
          
        val is_patt_constructor : patt -> bool
          
        val is_expr_constructor : expr -> bool
          
        val ty_of_stl : (Loc.t * string * (ctyp list)) -> ctyp
          
        val ty_of_sbt : (Loc.t * string * bool * ctyp) -> ctyp
          
        val bi_of_pe : (patt * expr) -> binding
          
        val pel_of_binding : binding -> (patt * expr) list
          
        val binding_of_pel : (patt * expr) list -> binding
          
        val sum_type_of_list : (Loc.t * string * (ctyp list)) list -> ctyp
          
        val record_type_of_list : (Loc.t * string * bool * ctyp) list -> ctyp
          
      end
      
    module Camlp4AstToAst (M : Camlp4Ast) : Ast with type loc = M.loc
      and type meta_bool = M.meta_bool
      and type 'a meta_option = 'a M.meta_option
      and type 'a meta_list = 'a M.meta_list and type ctyp = M.ctyp
      and type patt = M.patt and type expr = M.expr
      and type module_type = M.module_type and type sig_item = M.sig_item
      and type with_constr = M.with_constr
      and type module_expr = M.module_expr and type str_item = M.str_item
      and type class_type = M.class_type
      and type class_sig_item = M.class_sig_item
      and type class_expr = M.class_expr
      and type class_str_item = M.class_str_item and type binding = M.binding
      and type rec_binding = M.rec_binding
      and type module_binding = M.module_binding
      and type match_case = M.match_case and type ident = M.ident
      and type rec_flag = M.rec_flag
      and type direction_flag = M.direction_flag
      and type mutable_flag = M.mutable_flag
      and type private_flag = M.private_flag
      and type virtual_flag = M.virtual_flag
      and type row_var_flag = M.row_var_flag
      and type override_flag = M.override_flag = M
      
    module MakeCamlp4Ast (Loc : Type) =
      struct
        type loc =
          Loc.
          t
          and meta_bool =
          | BTrue | BFalse | BAnt of string
          and rec_flag =
          | ReRecursive | ReNonrecursive | ReNil | ReAnt of string
          and direction_flag =
          | DiTo | DiDownto | DiAnt of string
          and mutable_flag =
          | MuMutable | MuNil | MuAnt of string
          and private_flag =
          | PrPrivate | PrNil | PrAnt of string
          and virtual_flag =
          | ViVirtual | ViNil | ViAnt of string
          and override_flag =
          | OvOverride | OvNil | OvAnt of string
          and row_var_flag =
          | RvRowVar | RvNil | RvAnt of string
          and 'a meta_option =
          | ONone | OSome of 'a | OAnt of string
          and 'a meta_list =
          | LNil | LCons of 'a * 'a meta_list | LAnt of string
          and ident =
          | IdAcc of loc * ident * ident
          | IdApp of loc * ident * ident
          | IdLid of loc * string
          | IdUid of loc * string
          | IdAnt of loc * string
          and ctyp =
          | TyNil of loc
          | TyAli of loc * ctyp * ctyp
          | TyAny of loc
          | TyApp of loc * ctyp * ctyp
          | TyArr of loc * ctyp * ctyp
          | TyCls of loc * ident
          | TyLab of loc * string * ctyp
          | TyId of loc * ident
          | TyMan of loc * ctyp * ctyp
          | TyDcl of loc * string * ctyp list * ctyp * (ctyp * ctyp) list
          | TyExt of loc * ident * ctyp list * ctyp
          | TyObj of loc * ctyp * row_var_flag
          | TyOlb of loc * string * ctyp
          | TyPol of loc * ctyp * ctyp
          | TyTypePol of loc * ctyp * ctyp
          | TyQuo of loc * string
          | TyQuP of loc * string
          | TyQuM of loc * string
          | TyAnP of loc
          | TyAnM of loc
          | TyVrn of loc * string
          | TyRec of loc * ctyp
          | TyCol of loc * ctyp * ctyp
          | TySem of loc * ctyp * ctyp
          | TyCom of loc * ctyp * ctyp
          | TySum of loc * ctyp
          | TyOf of loc * ctyp * ctyp
          | TyAnd of loc * ctyp * ctyp
          | TyOr of loc * ctyp * ctyp
          | TyPrv of loc * ctyp
          | TyMut of loc * ctyp
          | TyTup of loc * ctyp
          | TySta of loc * ctyp * ctyp
          | TyVrnEq of loc * ctyp
          | TyVrnSup of loc * ctyp
          | TyVrnInf of loc * ctyp
          | TyVrnInfSup of loc * ctyp * ctyp
          | TyAmp of loc * ctyp * ctyp
          | TyOfAmp of loc * ctyp * ctyp
          | TyPkg of loc * module_type
          | TyOpn of loc
          | TyAtt of loc * string * str_item * ctyp
          | TyAnt of loc * string
          and patt =
          | PaNil of loc
          | PaId of loc * ident
          | PaAli of loc * patt * patt
          | PaAnt of loc * string
          | PaAny of loc
          | PaApp of loc * patt * patt
          | PaArr of loc * patt
          | PaCom of loc * patt * patt
          | PaSem of loc * patt * patt
          | PaChr of loc * string
          | PaInt of loc * string
          | PaInt32 of loc * string
          | PaInt64 of loc * string
          | PaNativeInt of loc * string
          | PaFlo of loc * string
          | PaLab of loc * string * patt
          | PaOlb of loc * string * patt
          | PaOlbi of loc * string * patt * expr
          | PaOrp of loc * patt * patt
          | PaRng of loc * patt * patt
          | PaRec of loc * patt
          | PaEq of loc * ident * patt
          | PaStr of loc * string
          | PaTup of loc * patt
          | PaTyc of loc * patt * ctyp
          | PaTyp of loc * ident
          | PaVrn of loc * string
          | PaLaz of loc * patt
          | PaAtt of loc * string * str_item * patt
          | PaMod of loc * string
          | PaExc of loc * patt
          and expr =
          | ExNil of loc
          | ExId of loc * ident
          | ExAcc of loc * expr * expr
          | ExAnt of loc * string
          | ExApp of loc * expr * expr
          | ExAre of loc * expr * expr
          | ExArr of loc * expr
          | ExSem of loc * expr * expr
          | ExAsf of loc
          | ExAsr of loc * expr
          | ExAss of loc * expr * expr
          | ExChr of loc * string
          | ExCoe of loc * expr * ctyp * ctyp
          | ExFlo of loc * string
          | ExFor of loc * patt * expr * expr * direction_flag * expr
          | ExFun of loc * match_case
          | ExIfe of loc * expr * expr * expr
          | ExInt of loc * string
          | ExInt32 of loc * string
          | ExInt64 of loc * string
          | ExNativeInt of loc * string
          | ExLab of loc * string * expr
          | ExLaz of loc * expr
          | ExLet of loc * rec_flag * binding * expr
          | ExLmd of loc * string * module_expr * expr
          | ExMat of loc * expr * match_case
          | ExNew of loc * ident
          | ExObj of loc * patt * class_str_item
          | ExOlb of loc * string * expr
          | ExOvr of loc * rec_binding
          | ExRec of loc * rec_binding * expr
          | ExSeq of loc * expr
          | ExSnd of loc * expr * string
          | ExSte of loc * expr * expr
          | ExStr of loc * string
          | ExTry of loc * expr * match_case
          | ExTup of loc * expr
          | ExCom of loc * expr * expr
          | ExTyc of loc * expr * ctyp
          | ExVrn of loc * string
          | ExWhi of loc * expr * expr
          | ExOpI of loc * ident * override_flag * expr
          | ExFUN of loc * string * expr
          | ExPkg of loc * module_expr
          | ExAtt of loc * string * str_item * expr
          and module_type =
          | MtNil of loc
          | MtId of loc * ident
          | MtAlias of loc * ident
          | MtFun of loc * string * module_type * module_type
          | MtQuo of loc * string
          | MtSig of loc * sig_item
          | MtWit of loc * module_type * with_constr
          | MtOf of loc * module_expr
          | MtAtt of loc * string * str_item * module_type
          | MtAnt of loc * string
          and sig_item =
          | SgNil of loc
          | SgCls of loc * class_type
          | SgClt of loc * class_type
          | SgSem of loc * sig_item * sig_item
          | SgDir of loc * string * expr
          | SgExc of loc * ctyp
          | SgExt of loc * string * ctyp * string meta_list
          | SgInc of loc * module_type
          | SgMod of loc * string * module_type
          | SgRecMod of loc * module_binding
          | SgMty of loc * string * module_type
          | SgOpn of loc * override_flag * ident
          | SgTyp of loc * rec_flag * ctyp
          | SgVal of loc * string * ctyp
          | SgAnt of loc * string
          and with_constr =
          | WcNil of loc
          | WcTyp of loc * ctyp * ctyp
          | WcMod of loc * ident * ident
          | WcTyS of loc * ctyp * ctyp
          | WcMoS of loc * ident * ident
          | WcAnd of loc * with_constr * with_constr
          | WcAnt of loc * string
          and binding =
          | BiNil of loc
          | BiAnd of loc * binding * binding
          | BiEq of loc * patt * expr
          | BiAnt of loc * string
          and rec_binding =
          | RbNil of loc
          | RbSem of loc * rec_binding * rec_binding
          | RbEq of loc * ident * expr
          | RbAnt of loc * string
          and module_binding =
          | MbNil of loc
          | MbAnd of loc * module_binding * module_binding
          | MbColEq of loc * string * module_type * module_expr
          | MbCol of loc * string * module_type
          | MbAnt of loc * string
          and match_case =
          | McNil of loc
          | McOr of loc * match_case * match_case
          | McArr of loc * patt * expr * expr
          | McAnt of loc * string
          and module_expr =
          | MeNil of loc
          | MeId of loc * ident
          | MeApp of loc * module_expr * module_expr
          | MeFun of loc * string * module_type * module_expr
          | MeStr of loc * str_item
          | MeTyc of loc * module_expr * module_type
          | MePkg of loc * expr
          | MeAtt of loc * string * str_item * module_expr
          | MeAnt of loc * string
          and str_item =
          | StNil of loc
          | StCls of loc * class_expr
          | StClt of loc * class_type
          | StSem of loc * str_item * str_item
          | StDir of loc * string * expr
          | StExc of loc * ctyp * ident meta_option
          | StExp of loc * expr
          | StExt of loc * string * ctyp * string meta_list
          | StInc of loc * module_expr
          | StMod of loc * string * module_expr
          | StRecMod of loc * module_binding
          | StMty of loc * string * module_type
          | StOpn of loc * override_flag * ident
          | StTyp of loc * rec_flag * ctyp
          | StVal of loc * rec_flag * binding
          | StAnt of loc * string
          and class_type =
          | CtNil of loc
          | CtCon of loc * virtual_flag * ident * ctyp
          | CtFun of loc * ctyp * class_type
          | CtSig of loc * ctyp * class_sig_item
          | CtAnd of loc * class_type * class_type
          | CtCol of loc * class_type * class_type
          | CtEq of loc * class_type * class_type
          | CtAtt of loc * string * str_item * class_type
          | CtAnt of loc * string
          and class_sig_item =
          | CgNil of loc
          | CgCtr of loc * ctyp * ctyp
          | CgSem of loc * class_sig_item * class_sig_item
          | CgInh of loc * class_type
          | CgMth of loc * string * private_flag * ctyp
          | CgVal of loc * string * mutable_flag * virtual_flag * ctyp
          | CgVir of loc * string * private_flag * ctyp
          | CgAnt of loc * string
          and class_expr =
          | CeNil of loc
          | CeApp of loc * class_expr * expr
          | CeCon of loc * virtual_flag * ident * ctyp
          | CeFun of loc * patt * class_expr
          | CeLet of loc * rec_flag * binding * class_expr
          | CeStr of loc * patt * class_str_item
          | CeTyc of loc * class_expr * class_type
          | CeAnd of loc * class_expr * class_expr
          | CeEq of loc * class_expr * class_expr
          | CeAtt of loc * string * str_item * class_expr
          | CeAnt of loc * string
          and class_str_item =
          | CrNil of loc
          | CrSem of loc * class_str_item * class_str_item
          | CrCtr of loc * ctyp * ctyp
          | CrInh of loc * override_flag * class_expr * string
          | CrIni of loc * expr
          | CrMth of loc * string * override_flag * private_flag * expr
                     * ctyp
          | CrVal of loc * string * override_flag * mutable_flag * expr
          | CrVir of loc * string * private_flag * ctyp
          | CrVvr of loc * string * mutable_flag * ctyp
          | CrAnt of loc * string
        
      end
      
    type ('a, 'loc) stream_filter =
      ('a * 'loc) Stream.t -> ('a * 'loc) Stream.t
    
    module type AstFilters =
      sig
        module Ast : Camlp4Ast
          
        type 'a filter = 'a -> 'a
        
        val register_sig_item_filter : Ast.sig_item filter -> unit
          
        val register_str_item_filter : Ast.str_item filter -> unit
          
        val register_topphrase_filter : Ast.str_item filter -> unit
          
        val fold_interf_filters :
          ('a -> Ast.sig_item filter -> 'a) -> 'a -> 'a
          
        val fold_implem_filters :
          ('a -> Ast.str_item filter -> 'a) -> 'a -> 'a
          
        val fold_topphrase_filters :
          ('a -> Ast.str_item filter -> 'a) -> 'a -> 'a
          
      end
      
    module type DynAst =
      sig
        module Ast : Ast
          
        type 'a tag
        
        val ctyp_tag : Ast.ctyp tag
          
        val patt_tag : Ast.patt tag
          
        val expr_tag : Ast.expr tag
          
        val module_type_tag : Ast.module_type tag
          
        val sig_item_tag : Ast.sig_item tag
          
        val with_constr_tag : Ast.with_constr tag
          
        val module_expr_tag : Ast.module_expr tag
          
        val str_item_tag : Ast.str_item tag
          
        val class_type_tag : Ast.class_type tag
          
        val class_sig_item_tag : Ast.class_sig_item tag
          
        val class_expr_tag : Ast.class_expr tag
          
        val class_str_item_tag : Ast.class_str_item tag
          
        val match_case_tag : Ast.match_case tag
          
        val ident_tag : Ast.ident tag
          
        val binding_tag : Ast.binding tag
          
        val rec_binding_tag : Ast.rec_binding tag
          
        val module_binding_tag : Ast.module_binding tag
          
        val string_of_tag : 'a tag -> string
          
        module Pack (X : sig type 'a t
                              end) :
          sig
            type pack
            
            val pack : 'a tag -> 'a X.t -> pack
              
            val unpack : 'a tag -> pack -> 'a X.t
              
            val print_tag : Format.formatter -> pack -> unit
              
          end
          
      end
      
    type quotation =
      { q_name : string; q_loc : string; q_shift : int; q_contents : string
      }
    
    module type Quotation =
      sig
        module Ast : Ast
          
        module DynAst : DynAst with module Ast = Ast
          
        open Ast
          
        type 'a expand_fun = loc -> string option -> string -> 'a
        
        val add : string -> 'a DynAst.tag -> 'a expand_fun -> unit
          
        val find : string -> 'a DynAst.tag -> 'a expand_fun
          
        val default : string ref
          
        val parse_quotation_result :
          (loc -> string -> 'a) -> loc -> quotation -> string -> string -> 'a
          
        val translate : (string -> string) ref
          
        val expand : loc -> quotation -> 'a DynAst.tag -> 'a
          
        val dump_file : (string option) ref
          
        module Error : Error
          
      end
      
    module type Token =
      sig
        module Loc : Loc
          
        type t
        
        val to_string : t -> string
          
        val print : Format.formatter -> t -> unit
          
        val match_keyword : string -> t -> bool
          
        val extract_string : t -> string
          
        module Filter :
          sig
            type token_filter = (t, Loc.t) stream_filter
            
            type t
            
            val mk : (string -> bool) -> t
              
            val define_filter : t -> (token_filter -> token_filter) -> unit
              
            val filter : t -> token_filter
              
            val keyword_added : t -> string -> bool -> unit
              
            val keyword_removed : t -> string -> unit
              
          end
          
        module Error : Error
          
      end
      
    type camlp4_token =
      | KEYWORD of string
      | SYMBOL of string
      | LIDENT of string
      | UIDENT of string
      | ESCAPED_IDENT of string
      | INT of int * string
      | INT32 of int32 * string
      | INT64 of int64 * string
      | NATIVEINT of nativeint * string
      | FLOAT of float * string
      | CHAR of char * string
      | STRING of string * string
      | LABEL of string
      | OPTLABEL of string
      | QUOTATION of quotation
      | ANTIQUOT of string * string
      | COMMENT of string
      | BLANKS of string
      | NEWLINE
      | LINE_DIRECTIVE of int * string option
      | EOI
    
    module type Camlp4Token = Token with type t = camlp4_token
      
    module type DynLoader =
      sig
        type t
        
        exception Error of string * string
          
        val mk : ?ocaml_stdlib: bool -> ?camlp4_stdlib: bool -> unit -> t
          
        val fold_load_path : t -> (string -> 'a -> 'a) -> 'a -> 'a
          
        val load : t -> string -> unit
          
        val include_dir : t -> string -> unit
          
        val find_in_path : t -> string -> string
          
        val is_native : bool
          
      end
      
    module Grammar =
      struct
        module type Action =
          sig
            type t
            
            val mk : 'a -> t
              
            val get : t -> 'a
              
            val getf : t -> 'a -> 'b
              
            val getf2 : t -> 'a -> 'b -> 'c
              
          end
          
        type assoc = | NonA | RightA | LeftA
        
        type position =
          | First
          | Last
          | Before of string
          | After of string
          | Level of string
        
        module type Structure =
          sig
            module Loc : Loc
              
            module Action : Action
              
            module Token : Token with module Loc = Loc
              
            type gram
            
            type internal_entry
            
            type tree
            
            type token_pattern = ((Token.t -> bool) * string)
            
            type token_info
            
            type token_stream = (Token.t * token_info) Stream.t
            
            val token_location : token_info -> Loc.t
              
            type symbol =
              | Smeta of string * symbol list * Action.t
              | Snterm of internal_entry
              | Snterml of internal_entry * string
              | Slist0 of symbol
              | Slist0sep of symbol * symbol
              | Slist1 of symbol
              | Slist1sep of symbol * symbol
              | Sopt of symbol
              | Stry of symbol
              | Sself
              | Snext
              | Stoken of token_pattern
              | Skeyword of string
              | Stree of tree
            
            type production_rule = ((symbol list) * Action.t)
            
            type single_extend_statment =
              ((string option) * (assoc option) * (production_rule list))
            
            type extend_statment =
              ((position option) * (single_extend_statment list))
            
            type delete_statment = symbol list
            
            type ('a, 'b, 'c) fold =
              internal_entry ->
                symbol list -> ('a Stream.t -> 'b) -> 'a Stream.t -> 'c
            
            type ('a, 'b, 'c) foldsep =
              internal_entry ->
                symbol list ->
                  ('a Stream.t -> 'b) ->
                    ('a Stream.t -> unit) -> 'a Stream.t -> 'c
            
          end
          
        module type Dynamic =
          sig
            include Structure
              
            val mk : unit -> gram
              
            module Entry :
              sig
                type 'a t
                
                val mk : gram -> string -> 'a t
                  
                val of_parser :
                  gram -> string -> (token_stream -> 'a) -> 'a t
                  
                val setup_parser : 'a t -> (token_stream -> 'a) -> unit
                  
                val name : 'a t -> string
                  
                val print : Format.formatter -> 'a t -> unit
                  
                val dump : Format.formatter -> 'a t -> unit
                  
                val obj : 'a t -> internal_entry
                  
                val clear : 'a t -> unit
                  
              end
              
            val get_filter : gram -> Token.Filter.t
              
            type 'a not_filtered
            
            val extend : 'a Entry.t -> extend_statment -> unit
              
            val delete_rule : 'a Entry.t -> delete_statment -> unit
              
            val srules :
              'a Entry.t -> ((symbol list) * Action.t) list -> symbol
              
            val sfold0 : ('a -> 'b -> 'b) -> 'b -> (_, 'a, 'b) fold
              
            val sfold1 : ('a -> 'b -> 'b) -> 'b -> (_, 'a, 'b) fold
              
            val sfold0sep : ('a -> 'b -> 'b) -> 'b -> (_, 'a, 'b) foldsep
              
            val lex :
              gram ->
                Loc.t ->
                  char Stream.t -> ((Token.t * Loc.t) Stream.t) not_filtered
              
            val lex_string :
              gram ->
                Loc.t -> string -> ((Token.t * Loc.t) Stream.t) not_filtered
              
            val filter :
              gram ->
                ((Token.t * Loc.t) Stream.t) not_filtered -> token_stream
              
            val parse : 'a Entry.t -> Loc.t -> char Stream.t -> 'a
              
            val parse_string : 'a Entry.t -> Loc.t -> string -> 'a
              
            val parse_tokens_before_filter :
              'a Entry.t -> ((Token.t * Loc.t) Stream.t) not_filtered -> 'a
              
            val parse_tokens_after_filter : 'a Entry.t -> token_stream -> 'a
              
          end
          
        module type Static =
          sig
            include Structure
              
            module Entry :
              sig
                type 'a t
                
                val mk : string -> 'a t
                  
                val of_parser : string -> (token_stream -> 'a) -> 'a t
                  
                val setup_parser : 'a t -> (token_stream -> 'a) -> unit
                  
                val name : 'a t -> string
                  
                val print : Format.formatter -> 'a t -> unit
                  
                val dump : Format.formatter -> 'a t -> unit
                  
                val obj : 'a t -> internal_entry
                  
                val clear : 'a t -> unit
                  
              end
              
            val get_filter : unit -> Token.Filter.t
              
            type 'a not_filtered
            
            val extend : 'a Entry.t -> extend_statment -> unit
              
            val delete_rule : 'a Entry.t -> delete_statment -> unit
              
            val srules :
              'a Entry.t -> ((symbol list) * Action.t) list -> symbol
              
            val sfold0 : ('a -> 'b -> 'b) -> 'b -> (_, 'a, 'b) fold
              
            val sfold1 : ('a -> 'b -> 'b) -> 'b -> (_, 'a, 'b) fold
              
            val sfold0sep : ('a -> 'b -> 'b) -> 'b -> (_, 'a, 'b) foldsep
              
            val lex :
              Loc.t ->
                char Stream.t -> ((Token.t * Loc.t) Stream.t) not_filtered
              
            val lex_string :
              Loc.t -> string -> ((Token.t * Loc.t) Stream.t) not_filtered
              
            val filter :
              ((Token.t * Loc.t) Stream.t) not_filtered -> token_stream
              
            val parse : 'a Entry.t -> Loc.t -> char Stream.t -> 'a
              
            val parse_string : 'a Entry.t -> Loc.t -> string -> 'a
              
            val parse_tokens_before_filter :
              'a Entry.t -> ((Token.t * Loc.t) Stream.t) not_filtered -> 'a
              
            val parse_tokens_after_filter : 'a Entry.t -> token_stream -> 'a
              
          end
          
      end
      
    module type Lexer =
      sig
        module Loc : Loc
          
        module Token : Token with module Loc = Loc
          
        module Error : Error
          
        val mk : unit -> Loc.t -> char Stream.t -> (Token.t * Loc.t) Stream.t
          
      end
      
    module Parser (Ast : Ast) =
      struct
        module type SIMPLE =
          sig
            val parse_expr : Ast.loc -> string -> Ast.expr
              
            val parse_patt : Ast.loc -> string -> Ast.patt
              
          end
          
        module type S =
          sig
            val parse_implem :
              ?directive_handler: (Ast.str_item -> Ast.str_item option) ->
                Ast.loc -> char Stream.t -> Ast.str_item
              
            val parse_interf :
              ?directive_handler: (Ast.sig_item -> Ast.sig_item option) ->
                Ast.loc -> char Stream.t -> Ast.sig_item
              
          end
          
      end
      
    module Printer (Ast : Ast) =
      struct
        module type S =
          sig
            val print_interf :
              ?input_file: string ->
                ?output_file: string -> Ast.sig_item -> unit
              
            val print_implem :
              ?input_file: string ->
                ?output_file: string -> Ast.str_item -> unit
              
          end
          
      end
      
    module type Syntax =
      sig
        module Loc : Loc
          
        module Ast : Ast with type loc = Loc.t
          
        module Token : Token with module Loc = Loc
          
        module Gram : Grammar.Static with module Loc = Loc
          and module Token = Token
          
        module Quotation : Quotation with module Ast = Ast
          
        module AntiquotSyntax : Parser(Ast).SIMPLE
          
        include Warning(Loc).S
          
        include Parser(Ast).S
          
        include Printer(Ast).S
          
      end
      
    module type Camlp4Syntax =
      sig
        module Loc : Loc
          
        module Ast : Camlp4Ast with module Loc = Loc
          
        module Token : Camlp4Token with module Loc = Loc
          
        module Gram : Grammar.Static with module Loc = Loc
          and module Token = Token
          
        module Quotation : Quotation with module Ast = Camlp4AstToAst(Ast)
          
        module AntiquotSyntax : Parser(Ast).SIMPLE
          
        include Warning(Loc).S
          
        include Parser(Ast).S
          
        include Printer(Ast).S
          
        val interf : ((Ast.sig_item list) * (Loc.t option)) Gram.Entry.t
          
        val implem : ((Ast.str_item list) * (Loc.t option)) Gram.Entry.t
          
        val top_phrase : (Ast.str_item option) Gram.Entry.t
          
        val use_file : ((Ast.str_item list) * (Loc.t option)) Gram.Entry.t
          
        val a_CHAR : string Gram.Entry.t
          
        val a_FLOAT : string Gram.Entry.t
          
        val a_INT : string Gram.Entry.t
          
        val a_INT32 : string Gram.Entry.t
          
        val a_INT64 : string Gram.Entry.t
          
        val a_LABEL : string Gram.Entry.t
          
        val a_LIDENT : string Gram.Entry.t
          
        val a_NATIVEINT : string Gram.Entry.t
          
        val a_OPTLABEL : string Gram.Entry.t
          
        val a_STRING : string Gram.Entry.t
          
        val a_UIDENT : string Gram.Entry.t
          
        val a_ident : string Gram.Entry.t
          
        val amp_ctyp : Ast.ctyp Gram.Entry.t
          
        val and_ctyp : Ast.ctyp Gram.Entry.t
          
        val match_case : Ast.match_case Gram.Entry.t
          
        val match_case0 : Ast.match_case Gram.Entry.t
          
        val match_case_quot : Ast.match_case Gram.Entry.t
          
        val binding : Ast.binding Gram.Entry.t
          
        val binding_quot : Ast.binding Gram.Entry.t
          
        val rec_binding_quot : Ast.rec_binding Gram.Entry.t
          
        val class_declaration : Ast.class_expr Gram.Entry.t
          
        val class_description : Ast.class_type Gram.Entry.t
          
        val class_expr : Ast.class_expr Gram.Entry.t
          
        val class_expr_quot : Ast.class_expr Gram.Entry.t
          
        val class_fun_binding : Ast.class_expr Gram.Entry.t
          
        val class_fun_def : Ast.class_expr Gram.Entry.t
          
        val class_info_for_class_expr : Ast.class_expr Gram.Entry.t
          
        val class_info_for_class_type : Ast.class_type Gram.Entry.t
          
        val class_longident : Ast.ident Gram.Entry.t
          
        val class_longident_and_param : Ast.class_expr Gram.Entry.t
          
        val class_name_and_param : (string * Ast.ctyp) Gram.Entry.t
          
        val class_sig_item : Ast.class_sig_item Gram.Entry.t
          
        val class_sig_item_quot : Ast.class_sig_item Gram.Entry.t
          
        val class_signature : Ast.class_sig_item Gram.Entry.t
          
        val class_str_item : Ast.class_str_item Gram.Entry.t
          
        val class_str_item_quot : Ast.class_str_item Gram.Entry.t
          
        val class_structure : Ast.class_str_item Gram.Entry.t
          
        val class_type : Ast.class_type Gram.Entry.t
          
        val class_type_declaration : Ast.class_type Gram.Entry.t
          
        val class_type_longident : Ast.ident Gram.Entry.t
          
        val class_type_longident_and_param : Ast.class_type Gram.Entry.t
          
        val class_type_plus : Ast.class_type Gram.Entry.t
          
        val class_type_quot : Ast.class_type Gram.Entry.t
          
        val comma_ctyp : Ast.ctyp Gram.Entry.t
          
        val comma_expr : Ast.expr Gram.Entry.t
          
        val comma_ipatt : Ast.patt Gram.Entry.t
          
        val comma_patt : Ast.patt Gram.Entry.t
          
        val comma_type_parameter : Ast.ctyp Gram.Entry.t
          
        val constrain : (Ast.ctyp * Ast.ctyp) Gram.Entry.t
          
        val constructor_arg_list : Ast.ctyp Gram.Entry.t
          
        val constructor_declaration : Ast.ctyp Gram.Entry.t
          
        val constructor_declarations : Ast.ctyp Gram.Entry.t
          
        val ctyp : Ast.ctyp Gram.Entry.t
          
        val ctyp_quot : Ast.ctyp Gram.Entry.t
          
        val cvalue_binding : Ast.expr Gram.Entry.t
          
        val direction_flag : Ast.direction_flag Gram.Entry.t
          
        val direction_flag_quot : Ast.direction_flag Gram.Entry.t
          
        val dummy : unit Gram.Entry.t
          
        val eq_expr : (string -> Ast.patt -> Ast.patt) Gram.Entry.t
          
        val expr : Ast.expr Gram.Entry.t
          
        val expr_eoi : Ast.expr Gram.Entry.t
          
        val expr_quot : Ast.expr Gram.Entry.t
          
        val field_expr : Ast.rec_binding Gram.Entry.t
          
        val field_expr_list : Ast.rec_binding Gram.Entry.t
          
        val fun_binding : Ast.expr Gram.Entry.t
          
        val fun_def : Ast.expr Gram.Entry.t
          
        val ident : Ast.ident Gram.Entry.t
          
        val ident_quot : Ast.ident Gram.Entry.t
          
        val ipatt : Ast.patt Gram.Entry.t
          
        val ipatt_tcon : Ast.patt Gram.Entry.t
          
        val label : string Gram.Entry.t
          
        val label_declaration : Ast.ctyp Gram.Entry.t
          
        val label_declaration_list : Ast.ctyp Gram.Entry.t
          
        val label_expr : Ast.rec_binding Gram.Entry.t
          
        val label_expr_list : Ast.rec_binding Gram.Entry.t
          
        val label_ipatt : Ast.patt Gram.Entry.t
          
        val label_ipatt_list : Ast.patt Gram.Entry.t
          
        val label_longident : Ast.ident Gram.Entry.t
          
        val label_patt : Ast.patt Gram.Entry.t
          
        val label_patt_list : Ast.patt Gram.Entry.t
          
        val labeled_ipatt : Ast.patt Gram.Entry.t
          
        val let_binding : Ast.binding Gram.Entry.t
          
        val meth_list : (Ast.ctyp * Ast.row_var_flag) Gram.Entry.t
          
        val meth_decl : Ast.ctyp Gram.Entry.t
          
        val module_binding : Ast.module_binding Gram.Entry.t
          
        val module_binding0 : Ast.module_expr Gram.Entry.t
          
        val module_binding_quot : Ast.module_binding Gram.Entry.t
          
        val module_declaration : Ast.module_type Gram.Entry.t
          
        val module_expr : Ast.module_expr Gram.Entry.t
          
        val module_expr_quot : Ast.module_expr Gram.Entry.t
          
        val module_longident : Ast.ident Gram.Entry.t
          
        val module_longident_with_app : Ast.ident Gram.Entry.t
          
        val module_rec_declaration : Ast.module_binding Gram.Entry.t
          
        val module_type : Ast.module_type Gram.Entry.t
          
        val package_type : Ast.module_type Gram.Entry.t
          
        val module_type_quot : Ast.module_type Gram.Entry.t
          
        val more_ctyp : Ast.ctyp Gram.Entry.t
          
        val name_tags : Ast.ctyp Gram.Entry.t
          
        val opt_as_lident : string Gram.Entry.t
          
        val opt_class_self_patt : Ast.patt Gram.Entry.t
          
        val opt_class_self_type : Ast.ctyp Gram.Entry.t
          
        val opt_comma_ctyp : Ast.ctyp Gram.Entry.t
          
        val opt_dot_dot : Ast.row_var_flag Gram.Entry.t
          
        val row_var_flag_quot : Ast.row_var_flag Gram.Entry.t
          
        val opt_eq_ctyp : Ast.ctyp Gram.Entry.t
          
        val opt_expr : Ast.expr Gram.Entry.t
          
        val opt_meth_list : Ast.ctyp Gram.Entry.t
          
        val opt_mutable : Ast.mutable_flag Gram.Entry.t
          
        val mutable_flag_quot : Ast.mutable_flag Gram.Entry.t
          
        val opt_override : Ast.override_flag Gram.Entry.t
          
        val override_flag_quot : Ast.override_flag Gram.Entry.t
          
        val opt_polyt : Ast.ctyp Gram.Entry.t
          
        val opt_private : Ast.private_flag Gram.Entry.t
          
        val private_flag_quot : Ast.private_flag Gram.Entry.t
          
        val opt_rec : Ast.rec_flag Gram.Entry.t
          
        val opt_nonrec : Ast.rec_flag Gram.Entry.t
          
        val rec_flag_quot : Ast.rec_flag Gram.Entry.t
          
        val opt_virtual : Ast.virtual_flag Gram.Entry.t
          
        val virtual_flag_quot : Ast.virtual_flag Gram.Entry.t
          
        val opt_when_expr : Ast.expr Gram.Entry.t
          
        val patt : Ast.patt Gram.Entry.t
          
        val patt_as_patt_opt : Ast.patt Gram.Entry.t
          
        val patt_eoi : Ast.patt Gram.Entry.t
          
        val patt_quot : Ast.patt Gram.Entry.t
          
        val patt_tcon : Ast.patt Gram.Entry.t
          
        val phrase : Ast.str_item Gram.Entry.t
          
        val poly_type : Ast.ctyp Gram.Entry.t
          
        val row_field : Ast.ctyp Gram.Entry.t
          
        val sem_expr : Ast.expr Gram.Entry.t
          
        val sem_expr_for_list : (Ast.expr -> Ast.expr) Gram.Entry.t
          
        val sem_patt : Ast.patt Gram.Entry.t
          
        val sem_patt_for_list : (Ast.patt -> Ast.patt) Gram.Entry.t
          
        val semi : unit Gram.Entry.t
          
        val sequence : Ast.expr Gram.Entry.t
          
        val do_sequence : Ast.expr Gram.Entry.t
          
        val sig_item : Ast.sig_item Gram.Entry.t
          
        val sig_item_quot : Ast.sig_item Gram.Entry.t
          
        val sig_items : Ast.sig_item Gram.Entry.t
          
        val star_ctyp : Ast.ctyp Gram.Entry.t
          
        val str_item : Ast.str_item Gram.Entry.t
          
        val str_item_quot : Ast.str_item Gram.Entry.t
          
        val str_items : Ast.str_item Gram.Entry.t
          
        val type_constraint : unit Gram.Entry.t
          
        val type_declaration : Ast.ctyp Gram.Entry.t
          
        val type_ident_and_parameters :
          (Ast.ident * (Ast.ctyp list)) Gram.Entry.t
          
        val type_kind : Ast.ctyp Gram.Entry.t
          
        val type_longident : Ast.ident Gram.Entry.t
          
        val type_longident_and_parameters : Ast.ctyp Gram.Entry.t
          
        val type_parameter : Ast.ctyp Gram.Entry.t
          
        val type_parameters : (Ast.ctyp -> Ast.ctyp) Gram.Entry.t
          
        val typevars : Ast.ctyp Gram.Entry.t
          
        val val_longident : Ast.ident Gram.Entry.t
          
        val value_let : unit Gram.Entry.t
          
        val value_val : unit Gram.Entry.t
          
        val with_constr : Ast.with_constr Gram.Entry.t
          
        val with_constr_quot : Ast.with_constr Gram.Entry.t
          
        val prefixop : Ast.expr Gram.Entry.t
          
        val infixop0 : Ast.expr Gram.Entry.t
          
        val infixop1 : Ast.expr Gram.Entry.t
          
        val infixop2 : Ast.expr Gram.Entry.t
          
        val infixop3 : Ast.expr Gram.Entry.t
          
        val infixop4 : Ast.expr Gram.Entry.t
          
      end
      
    module type SyntaxExtension =
      functor (Syn : Syntax) -> Syntax with module Loc = Syn.Loc
        and module Ast = Syn.Ast and module Token = Syn.Token
        and module Gram = Syn.Gram and module Quotation = Syn.Quotation
      
  end
  
module ErrorHandler :
  sig
    val print : Format.formatter -> exn -> unit
      
    val try_print : Format.formatter -> exn -> unit
      
    val to_string : exn -> string
      
    val try_to_string : exn -> string
      
    val register : (Format.formatter -> exn -> unit) -> unit
      
    module Register (Error : Sig.Error) : sig  end
      
    module ObjTools :
      sig
        val print : Format.formatter -> Obj.t -> unit
          
        val print_desc : Format.formatter -> Obj.t -> unit
          
        val to_string : Obj.t -> string
          
        val desc : Obj.t -> string
          
      end
      
  end =
  struct
    open Format
      
    module ObjTools =
      struct
        let desc obj =
          if Obj.is_block obj
          then "tag = " ^ (string_of_int (Obj.tag obj))
          else "int_val = " ^ (string_of_int (Obj.obj obj))
          
        let rec to_string r =
          if Obj.is_int r
          then
            (let i : int = Obj.magic r
             in (string_of_int i) ^ (" | CstTag" ^ (string_of_int (i + 1))))
          else
            (let rec get_fields acc =
               function
               | 0 -> acc
               | n -> let n = n - 1 in get_fields ((Obj.field r n) :: acc) n in
             let rec is_list r =
               if Obj.is_int r
               then r = (Obj.repr 0)
               else
                 (let s = Obj.size r
                  and t = Obj.tag r
                  in (t = 0) && ((s = 2) && (is_list (Obj.field r 1)))) in
             let rec get_list r =
               if Obj.is_int r
               then []
               else
                 (let h = Obj.field r 0
                  and t = get_list (Obj.field r 1)
                  in h :: t) in
             let opaque name = "<" ^ (name ^ ">") in
             let s = Obj.size r
             and t = Obj.tag r
             in
               match t with
               | _ when is_list r ->
                   let fields = get_list r
                   in
                     "[" ^
                       ((String.concat "; " (List.map to_string fields)) ^
                          "]")
               | 0 ->
                   let fields = get_fields [] s
                   in
                     "(" ^
                       ((String.concat ", " (List.map to_string fields)) ^
                          ")")
               | x when x = Obj.lazy_tag -> opaque "lazy"
               | x when x = Obj.closure_tag -> opaque "closure"
               | x when x = Obj.object_tag ->
                   let fields = get_fields [] s in
                   let (_class, id, slots) =
                     (match fields with
                      | h :: h' :: t -> (h, h', t)
                      | _ -> assert false)
                   in
                     "Object #" ^
                       ((to_string id) ^
                          (" (" ^
                             ((String.concat ", " (List.map to_string slots))
                                ^ ")")))
               | x when x = Obj.infix_tag -> opaque "infix"
               | x when x = Obj.forward_tag -> opaque "forward"
               | x when x < Obj.no_scan_tag ->
                   let fields = get_fields [] s
                   in
                     "Tag" ^
                       ((string_of_int t) ^
                          (" (" ^
                             ((String.concat ", " (List.map to_string fields))
                                ^ ")")))
               | x when x = Obj.string_tag ->
                   "\"" ^ ((String.escaped (Obj.magic r : string)) ^ "\"")
               | x when x = Obj.double_tag ->
                   Utils.float_repres (Obj.magic r : float)
               | x when x = Obj.abstract_tag -> opaque "abstract"
               | x when x = Obj.custom_tag -> opaque "custom"
               | _ ->
                   failwith
                     ("ObjTools.to_string: unknown tag (" ^
                        ((string_of_int t) ^ ")")))
          
        let print ppf x = fprintf ppf "%s" (to_string x)
          
        let print_desc ppf x = fprintf ppf "%s" (desc x)
          
      end
      
    let default_handler ppf x =
      let x = Obj.repr x
      in
        (if (Obj.tag x) <> 0
         then
           fprintf ppf "Camlp4: Uncaught exception: %s"
             (Obj.obj (Obj.field x 0) : string)
         else
           (fprintf ppf "Camlp4: Uncaught exception: %s"
              (Obj.obj (Obj.field (Obj.field x 0) 0) : string);
            if (Obj.size x) > 1
            then
              (pp_print_string ppf " (";
               for i = 1 to (Obj.size x) - 1 do
                 if i > 1 then pp_print_string ppf ", " else ();
                 ObjTools.print ppf (Obj.field x i)
               done;
               pp_print_char ppf ')')
            else ());
         fprintf ppf "@.")
      
    let handler =
      ref (fun ppf default_handler exn -> default_handler ppf exn)
      
    let register f =
      let current_handler = !handler
      in
        handler :=
          fun ppf default_handler exn ->
            try f ppf exn
            with | exn -> current_handler ppf default_handler exn
      
    module Register (Error : Sig.Error) =
      struct
        let _ =
          let current_handler = !handler
          in
            handler :=
              fun ppf default_handler ->
                function
                | Error.E x -> Error.print ppf x
                | x -> current_handler ppf default_handler x
          
      end
      
    let gen_print ppf default_handler =
      function
      | Out_of_memory -> fprintf ppf "Out of memory"
      | Assert_failure ((file, line, char)) ->
          fprintf ppf "Assertion failed, file %S, line %d, char %d" file line
            char
      | Match_failure ((file, line, char)) ->
          fprintf ppf "Pattern matching failed, file %S, line %d, char %d"
            file line char
      | Failure str -> fprintf ppf "Failure: %S" str
      | Invalid_argument str -> fprintf ppf "Invalid argument: %S" str
      | Sys_error str -> fprintf ppf "I/O error: %S" str
      | Stream.Failure -> fprintf ppf "Parse failure"
      | Stream.Error str -> fprintf ppf "Parse error: %s" str
      | x -> !handler ppf default_handler x
      
    let print ppf = gen_print ppf default_handler
      
    let try_print ppf = gen_print ppf (fun _ -> raise)
      
    let to_string exn = Format.asprintf "%a" print exn
      
    let try_to_string exn = Format.asprintf "%a" try_print exn
      
  end
  
module Struct =
  struct
    module Loc : sig include Sig.Loc
                        end =
      struct
        open Format
          
        type pos = { line : int; bol : int; off : int }
        
        type t =
          { file_name : string; start : pos; stop : pos; ghost : bool
          }
        
        let dump_sel f x =
          let s =
            match x with
            | `start -> "`start"
            | `stop -> "`stop"
            | `both -> "`both"
            | _ -> "<not-printable>"
          in pp_print_string f s
          
        let dump_pos f x =
          fprintf f "@[<hov 2>{ line = %d ;@ bol = %d ;@ off = %d } : pos@]"
            x.line x.bol x.off
          
        let dump_long f x =
          fprintf f
            "@[<hov 2>{ file_name = %s ;@ start = %a (%d-%d);@ stop = %a (%d);@ ghost = %b@ } : Loc.t@]"
            x.file_name dump_pos x.start (x.start.off - x.start.bol)
            (x.stop.off - x.start.bol) dump_pos x.stop
            (x.stop.off - x.stop.bol) x.ghost
          
        let dump f x =
          fprintf f "[%S: %d:%d-%d %d:%d%t]" x.file_name x.start.line
            (x.start.off - x.start.bol) (x.stop.off - x.start.bol)
            x.stop.line (x.stop.off - x.stop.bol)
            (fun o -> if x.ghost then fprintf o " (ghost)" else ())
          
        let start_pos = { line = 1; bol = 0; off = 0; }
          
        let ghost =
          {
            file_name = "ghost-location";
            start = start_pos;
            stop = start_pos;
            ghost = true;
          }
          
        let mk file_name =
          {
            file_name = file_name;
            start = start_pos;
            stop = start_pos;
            ghost = false;
          }
          
        let of_tuple (file_name, start_line, start_bol, start_off, stop_line,
                      stop_bol, stop_off, ghost)
                     =
          {
            file_name = file_name;
            start = { line = start_line; bol = start_bol; off = start_off; };
            stop = { line = stop_line; bol = stop_bol; off = stop_off; };
            ghost = ghost;
          }
          
        let to_tuple {
                       file_name = file_name;
                       start =
                         {
                           line = start_line;
                           bol = start_bol;
                           off = start_off
                         };
                       stop =
                         { line = stop_line; bol = stop_bol; off = stop_off };
                       ghost = ghost
                     } =
          (file_name, start_line, start_bol, start_off, stop_line, stop_bol,
           stop_off, ghost)
          
        let pos_of_lexing_position p =
          let pos =
            {
              line = p.Lexing.pos_lnum;
              bol = p.Lexing.pos_bol;
              off = p.Lexing.pos_cnum;
            }
          in pos
          
        let pos_to_lexing_position p file_name =
          {
            Lexing.pos_fname = file_name;
            pos_lnum = p.line;
            pos_bol = p.bol;
            pos_cnum = p.off;
          }
          
        let better_file_name a b =
          match (a, b) with
          | ("", "") -> a
          | ("", x) -> x
          | (x, "") -> x
          | ("-", x) -> x
          | (x, "-") -> x
          | (x, _) -> x
          
        let of_lexbuf lb =
          let start = Lexing.lexeme_start_p lb
          and stop = Lexing.lexeme_end_p lb in
          let loc =
            {
              file_name =
                better_file_name start.Lexing.pos_fname stop.Lexing.pos_fname;
              start = pos_of_lexing_position start;
              stop = pos_of_lexing_position stop;
              ghost = false;
            }
          in loc
          
        let of_lexing_position pos =
          let loc =
            {
              file_name = pos.Lexing.pos_fname;
              start = pos_of_lexing_position pos;
              stop = pos_of_lexing_position pos;
              ghost = false;
            }
          in loc
          
        let to_ocaml_location x =
          {
            Location.loc_start = pos_to_lexing_position x.start x.file_name;
            loc_end = pos_to_lexing_position x.stop x.file_name;
            loc_ghost = x.ghost;
          }
          
        let of_ocaml_location {
                                Location.loc_start = a;
                                loc_end = b;
                                loc_ghost = g
                              } =
          let res =
            {
              file_name =
                better_file_name a.Lexing.pos_fname b.Lexing.pos_fname;
              start = pos_of_lexing_position a;
              stop = pos_of_lexing_position b;
              ghost = g;
            }
          in res
          
        let start_pos x = pos_to_lexing_position x.start x.file_name
          
        let stop_pos x = pos_to_lexing_position x.stop x.file_name
          
        let merge a b =
          if a == b
          then a
          else
            (let r =
               match ((a.ghost), (b.ghost)) with
               | (false, false) -> { (a) with stop = b.stop; }
               | (true, true) -> { (a) with stop = b.stop; }
               | (true, _) -> { (a) with stop = b.stop; }
               | (_, true) -> { (b) with start = a.start; }
             in r)
          
        let join x = { (x) with stop = x.start; }
          
        let map f start_stop_both x =
          match start_stop_both with
          | `start -> { (x) with start = f x.start; }
          | `stop -> { (x) with stop = f x.stop; }
          | `both -> { (x) with start = f x.start; stop = f x.stop; }
          
        let move_pos chars x = { (x) with off = x.off + chars; }
          
        let move s chars x = map (move_pos chars) s x
          
        let move_line lines x =
          let move_line_pos x =
            { (x) with line = x.line + lines; bol = x.off; }
          in map move_line_pos `both x
          
        let shift width x =
          { (x) with start = x.stop; stop = move_pos width x.stop; }
          
        let file_name x = x.file_name
          
        let start_line x = x.start.line
          
        let stop_line x = x.stop.line
          
        let start_bol x = x.start.bol
          
        let stop_bol x = x.stop.bol
          
        let start_off x = x.start.off
          
        let stop_off x = x.stop.off
          
        let is_ghost x = x.ghost
          
        let set_file_name s x = { (x) with file_name = s; }
          
        let ghostify x = { (x) with ghost = true; }
          
        let make_absolute x =
          let pwd = Sys.getcwd ()
          in
            if Filename.is_relative x.file_name
            then { (x) with file_name = Filename.concat pwd x.file_name; }
            else x
          
        let strictly_before x y =
          let b = (x.stop.off < y.start.off) && (x.file_name = y.file_name)
          in b
          
        let to_string x =
          let (a, b) = ((x.start), (x.stop)) in
          let res =
            sprintf "File \"%s\", line %d, characters %d-%d" x.file_name
              a.line (a.off - a.bol) (b.off - a.bol)
          in
            if x.start.line <> x.stop.line
            then
              sprintf "%s (end at line %d, character %d)" res x.stop.line
                (b.off - b.bol)
            else res
          
        let print out x = pp_print_string out (to_string x)
          
        let check x msg =
          if
            ((start_line x) > (stop_line x)) ||
              (((start_bol x) > (stop_bol x)) ||
                 (((start_off x) > (stop_off x)) ||
                    (((start_line x) < 0) ||
                       (((stop_line x) < 0) ||
                          (((start_bol x) < 0) ||
                             (((stop_bol x) < 0) ||
                                (((start_off x) < 0) || ((stop_off x) < 0))))))))
          then
            (eprintf "*** Warning: (%s) strange positions ***\n%a@\n" msg
               print x;
             false)
          else true
          
        exception Exc_located of t * exn
          
        let _ =
          ErrorHandler.register
            (fun ppf ->
               function
               | Exc_located (loc, exn) ->
                   fprintf ppf "%a:@\n%a" print loc ErrorHandler.print exn
               | exn -> raise exn)
          
        let name = ref "_loc"
          
        let raise loc exc =
          match exc with
          | Exc_located (_, _) -> raise exc
          | _ -> raise (Exc_located (loc, exc))
          
      end
      
    module Token :
      sig
        module Make (Loc : Sig.Loc) : Sig.Camlp4Token with module Loc = Loc
          
        module Eval :
          sig
            val char : string -> char
              
            val string : ?strict: unit -> string -> string
              
          end
          
      end =
      struct
        open Format
          
        module Make (Loc : Sig.Loc) : Sig.Camlp4Token with module Loc = Loc =
          struct
            module Loc = Loc
              
            open Sig
              
            type t = camlp4_token
            
            type token = t
            
            let to_string =
              function
              | KEYWORD s -> sprintf "KEYWORD %S" s
              | SYMBOL s -> sprintf "SYMBOL %S" s
              | LIDENT s -> sprintf "LIDENT %S" s
              | UIDENT s -> sprintf "UIDENT %S" s
              | INT (_, s) -> sprintf "INT %s" s
              | INT32 (_, s) -> sprintf "INT32 %sd" s
              | INT64 (_, s) -> sprintf "INT64 %sd" s
              | NATIVEINT (_, s) -> sprintf "NATIVEINT %sd" s
              | FLOAT (_, s) -> sprintf "FLOAT %s" s
              | CHAR (_, s) -> sprintf "CHAR '%s'" s
              | STRING (_, s) -> sprintf "STRING \"%s\"" s
              | LABEL s -> sprintf "LABEL %S" s
              | OPTLABEL s -> sprintf "OPTLABEL %S" s
              | ANTIQUOT (n, s) -> sprintf "ANTIQUOT %s: %S" n s
              | QUOTATION x ->
                  sprintf
                    "QUOTATION { q_name=%S; q_loc=%S; q_shift=%d; q_contents=%S }"
                    x.q_name x.q_loc x.q_shift x.q_contents
              | COMMENT s -> sprintf "COMMENT %S" s
              | BLANKS s -> sprintf "BLANKS %S" s
              | NEWLINE -> sprintf "NEWLINE"
              | EOI -> sprintf "EOI"
              | ESCAPED_IDENT s -> sprintf "ESCAPED_IDENT %S" s
              | LINE_DIRECTIVE (i, None) -> sprintf "LINE_DIRECTIVE %d" i
              | LINE_DIRECTIVE (i, (Some s)) ->
                  sprintf "LINE_DIRECTIVE %d %S" i s
              
            let print ppf x = pp_print_string ppf (to_string x)
              
            let match_keyword kwd =
              function | KEYWORD kwd' when kwd = kwd' -> true | _ -> false
              
            let extract_string =
              function
              | KEYWORD s | SYMBOL s | LIDENT s | UIDENT s | INT (_, s) |
                  INT32 (_, s) | INT64 (_, s) | NATIVEINT (_, s) |
                  FLOAT (_, s) | CHAR (_, s) | STRING (_, s) | LABEL s |
                  OPTLABEL s | COMMENT s | BLANKS s | ESCAPED_IDENT s -> s
              | tok ->
                  invalid_arg
                    ("Cannot extract a string from this token: " ^
                       (to_string tok))
              
            module Error =
              struct
                type t =
                  | Illegal_token of string
                  | Keyword_as_label of string
                  | Illegal_token_pattern of string * string
                  | Illegal_constructor of string
                
                exception E of t
                  
                let print ppf =
                  function
                  | Illegal_token s -> fprintf ppf "Illegal token (%s)" s
                  | Keyword_as_label kwd ->
                      fprintf ppf
                        "`%s' is a keyword, it cannot be used as label name"
                        kwd
                  | Illegal_token_pattern (p_con, p_prm) ->
                      fprintf ppf "Illegal token pattern: %s %S" p_con p_prm
                  | Illegal_constructor con ->
                      fprintf ppf "Illegal constructor %S" con
                  
                let to_string x = Format.asprintf "%a" print x
                  
              end
              
            let _ = let module M = ErrorHandler.Register(Error) in ()
              
            module Filter =
              struct
                type token_filter = (t, Loc.t) stream_filter
                
                type t =
                  { is_kwd : string -> bool; mutable filter : token_filter
                  }
                
                let err error loc =
                  raise (Loc.Exc_located (loc, (Error.E error)))
                  
                let keyword_conversion tok is_kwd =
                  match tok with
                  | SYMBOL s | LIDENT s | UIDENT s when is_kwd s -> KEYWORD s
                  | ESCAPED_IDENT s -> LIDENT s
                  | _ -> tok
                  
                let check_keyword_as_label tok loc is_kwd =
                  let s =
                    match tok with | LABEL s -> s | OPTLABEL s -> s | _ -> ""
                  in
                    if (s <> "") && (is_kwd s)
                    then err (Error.Keyword_as_label s) loc
                    else ()
                  
                let check_unknown_keywords tok loc =
                  match tok with
                  | SYMBOL s -> err (Error.Illegal_token s) loc
                  | _ -> ()
                  
                let error_no_respect_rules p_con p_prm =
                  raise
                    (Error.E (Error.Illegal_token_pattern (p_con, p_prm)))
                  
                let check_keyword _ = true
                  
                let error_on_unknown_keywords = ref false
                  
                let rec ignore_layout (__strm : _ Stream.t) =
                  match Stream.peek __strm with
                  | Some
                      (((COMMENT _ | BLANKS _ | NEWLINE |
                           LINE_DIRECTIVE (_, _)),
                        _))
                      -> (Stream.junk __strm; ignore_layout __strm)
                  | Some x ->
                      (Stream.junk __strm;
                       let s = __strm
                       in
                         Stream.icons x
                           (Stream.slazy (fun _ -> ignore_layout s)))
                  | _ -> Stream.sempty
                  
                let mk is_kwd = { is_kwd = is_kwd; filter = ignore_layout; }
                  
                let filter x =
                  let f tok loc =
                    let tok = keyword_conversion tok x.is_kwd
                    in
                      (check_keyword_as_label tok loc x.is_kwd;
                       if !error_on_unknown_keywords
                       then check_unknown_keywords tok loc
                       else ();
                       (tok, loc)) in
                  let rec filter (__strm : _ Stream.t) =
                    match Stream.peek __strm with
                    | Some ((tok, loc)) ->
                        (Stream.junk __strm;
                         let s = __strm
                         in
                           Stream.lcons (fun _ -> f tok loc)
                             (Stream.slazy (fun _ -> filter s)))
                    | _ -> Stream.sempty in
                  let rec tracer (__strm : _ Stream.t) =
                    match Stream.peek __strm with
                    | Some (((_tok, _loc) as x)) ->
                        (Stream.junk __strm;
                         let xs = __strm
                         in
                           Stream.icons x (Stream.slazy (fun _ -> tracer xs)))
                    | _ -> Stream.sempty
                  in fun strm -> tracer (x.filter (filter strm))
                  
                let define_filter x f = x.filter <- f x.filter
                  
                let keyword_added _ _ _ = ()
                  
                let keyword_removed _ _ = ()
                  
              end
              
          end
          
        module Eval =
          struct
            let valch x = (Char.code x) - (Char.code '0')
              
            let valch_hex x =
              let d = Char.code x
              in
                if d >= 97
                then d - 87
                else if d >= 65 then d - 55 else d - 48
              
            let rec skip_indent (__strm : _ Stream.t) =
              match Stream.peek __strm with
              | Some (' ' | '\t') -> (Stream.junk __strm; skip_indent __strm)
              | _ -> ()
              
            let skip_opt_linefeed (__strm : _ Stream.t) =
              match Stream.peek __strm with
              | Some '\n' -> (Stream.junk __strm; ())
              | _ -> ()
              
            let chr c =
              if (c < 0) || (c > 255)
              then failwith "invalid char token"
              else Char.chr c
              
            let rec backslash (__strm : _ Stream.t) =
              match Stream.peek __strm with
              | Some '\n' -> (Stream.junk __strm; '\n')
              | Some '\r' -> (Stream.junk __strm; '\r')
              | Some 'n' -> (Stream.junk __strm; '\n')
              | Some 'r' -> (Stream.junk __strm; '\r')
              | Some 't' -> (Stream.junk __strm; '\t')
              | Some 'b' -> (Stream.junk __strm; '\b')
              | Some '\\' -> (Stream.junk __strm; '\\')
              | Some '"' -> (Stream.junk __strm; '"')
              | Some '\'' -> (Stream.junk __strm; '\'')
              | Some ' ' -> (Stream.junk __strm; ' ')
              | Some (('0' .. '9' as c1)) ->
                  (Stream.junk __strm;
                   (match Stream.peek __strm with
                    | Some (('0' .. '9' as c2)) ->
                        (Stream.junk __strm;
                         (match Stream.peek __strm with
                          | Some (('0' .. '9' as c3)) ->
                              (Stream.junk __strm;
                               chr
                                 (((100 * (valch c1)) + (10 * (valch c2))) +
                                    (valch c3)))
                          | _ -> raise (Stream.Error "")))
                    | _ -> raise (Stream.Error "")))
              | Some 'x' ->
                  (Stream.junk __strm;
                   (match Stream.peek __strm with
                    | Some (('0' .. '9' | 'a' .. 'f' | 'A' .. 'F' as c1)) ->
                        (Stream.junk __strm;
                         (match Stream.peek __strm with
                          | Some
                              (('0' .. '9' | 'a' .. 'f' | 'A' .. 'F' as c2))
                              ->
                              (Stream.junk __strm;
                               chr ((16 * (valch_hex c1)) + (valch_hex c2)))
                          | _ -> raise (Stream.Error "")))
                    | _ -> raise (Stream.Error "")))
              | _ -> raise Stream.Failure
              
            let rec backslash_in_string strict store (__strm : _ Stream.t) =
              match Stream.peek __strm with
              | Some '\n' -> (Stream.junk __strm; skip_indent __strm)
              | Some '\r' ->
                  (Stream.junk __strm;
                   let s = __strm in (skip_opt_linefeed s; skip_indent s))
              | _ ->
                  (match try Some (backslash __strm)
                         with | Stream.Failure -> None
                   with
                   | Some x -> store x
                   | _ ->
                       (match Stream.peek __strm with
                        | Some c when not strict ->
                            (Stream.junk __strm; store '\\'; store c)
                        | _ -> failwith "invalid string token"))
              
            let char s =
              if (String.length s) = 1
              then s.[0]
              else
                if (String.length s) = 0
                then failwith "invalid char token"
                else
                  (let (__strm : _ Stream.t) = Stream.of_string s
                   in
                     match Stream.peek __strm with
                     | Some '\\' ->
                         (Stream.junk __strm;
                          (try backslash __strm
                           with | Stream.Failure -> raise (Stream.Error "")))
                     | _ -> failwith "invalid char token")
              
            let string ?strict s =
              let buf = Buffer.create 23 in
              let store = Buffer.add_char buf in
              let rec parse (__strm : _ Stream.t) =
                match Stream.peek __strm with
                | Some '\\' ->
                    (Stream.junk __strm;
                     let _ =
                       (try backslash_in_string (strict <> None) store __strm
                        with | Stream.Failure -> raise (Stream.Error ""))
                     in parse __strm)
                | Some c ->
                    (Stream.junk __strm;
                     let s = __strm in (store c; parse s))
                | _ -> Buffer.contents buf
              in parse (Stream.of_string s)
              
          end
          
      end
      
    module Lexer =
      struct
        module TokenEval = Token.Eval
          
        module Make (Token : Sig.Camlp4Token) =
          struct
            module Loc = Token.Loc
              
            module Token = Token
              
            open Lexing
              
            open Sig
              
            module Error =
              struct
                type t =
                  | Illegal_character of char
                  | Illegal_escape of string
                  | Unterminated_comment
                  | Unterminated_string
                  | Unterminated_quotation
                  | Unterminated_antiquot
                  | Unterminated_string_in_comment
                  | Comment_start
                  | Comment_not_end
                  | Literal_overflow of string
                  | Invalid_literal of string
                
                exception E of t
                  
                open Format
                  
                let print ppf =
                  function
                  | Illegal_character c ->
                      fprintf ppf "Illegal character (%s)" (Char.escaped c)
                  | Illegal_escape s ->
                      fprintf ppf
                        "Illegal backslash escape in string or character (%s)"
                        s
                  | Unterminated_comment ->
                      fprintf ppf "Comment not terminated"
                  | Unterminated_string ->
                      fprintf ppf "String literal not terminated"
                  | Unterminated_string_in_comment ->
                      fprintf ppf
                        "This comment contains an unterminated string literal"
                  | Unterminated_quotation ->
                      fprintf ppf "Quotation not terminated"
                  | Unterminated_antiquot ->
                      fprintf ppf "Antiquotation not terminated"
                  | Literal_overflow ty ->
                      fprintf ppf
                        "Integer literal exceeds the range of representable integers of type %s"
                        ty
                  | Comment_start ->
                      fprintf ppf "this is the start of a comment"
                  | Comment_not_end ->
                      fprintf ppf "this is not the end of a comment"
                  | Invalid_literal s -> fprintf ppf "Invalid literal %s" s
                  
                let to_string x =
                  let b = Buffer.create 50 in
                  let () = bprintf b "%a" print x in Buffer.contents b
                  
              end
              
            let _ = let module M = ErrorHandler.Register(Error) in ()
              
            open Error
              
            type context =
              { loc : Loc.t; in_comment : bool; quotations : bool;
                antiquots : bool; lexbuf : lexbuf; buffer : Buffer.t
              }
            
            let default_context lb =
              {
                loc = Loc.ghost;
                in_comment = false;
                quotations = true;
                antiquots = false;
                lexbuf = lb;
                buffer = Buffer.create 256;
              }
              
            let store c = Buffer.add_string c.buffer (Lexing.lexeme c.lexbuf)
              
            let istore_char c i =
              Buffer.add_char c.buffer (Lexing.lexeme_char c.lexbuf i)
              
            let buff_contents c =
              let contents = Buffer.contents c.buffer
              in (Buffer.reset c.buffer; contents)
              
            let loc c = Loc.merge c.loc (Loc.of_lexbuf c.lexbuf)
              
            let quotations c = c.quotations
              
            let antiquots c = c.antiquots
              
            let is_in_comment c = c.in_comment
              
            let in_comment c = { (c) with in_comment = true; }
              
            let set_start_p c = c.lexbuf.lex_start_p <- Loc.start_pos c.loc
              
            let move_start_p shift c =
              let p = c.lexbuf.lex_start_p
              in
                c.lexbuf.lex_start_p <-
                  { (p) with pos_cnum = p.pos_cnum + shift; }
              
            let update_loc c = { (c) with loc = Loc.of_lexbuf c.lexbuf; }
              
            let with_curr_loc f c = f (update_loc c) c.lexbuf
              
            let parse_nested f c =
              (with_curr_loc f c; set_start_p c; buff_contents c)
              
            let shift n c = { (c) with loc = Loc.move `both n c.loc; }
              
            let store_parse f c = (store c; f c c.lexbuf)
              
            let parse f c = f c c.lexbuf
              
            let mk_quotation quotation c name loc shift =
              let s = parse_nested quotation (update_loc c) in
              let contents = String.sub s 0 ((String.length s) - 2)
              in
                QUOTATION
                  {
                    q_name = name;
                    q_loc = loc;
                    q_shift = shift;
                    q_contents = contents;
                  }
              
            let update_loc c file line absolute chars =
              let lexbuf = c.lexbuf in
              let pos = lexbuf.lex_curr_p in
              let new_file =
                match file with | None -> pos.pos_fname | Some s -> s
              in
                lexbuf.lex_curr_p <-
                  {
                    (pos)
                    with
                    pos_fname = new_file;
                    pos_lnum = if absolute then line else pos.pos_lnum + line;
                    pos_bol = pos.pos_cnum - chars;
                  }
              
            let cvt_int_literal s = - (int_of_string ("-" ^ s))
              
            let cvt_int32_literal s = Int32.neg (Int32.of_string ("-" ^ s))
              
            let cvt_int64_literal s = Int64.neg (Int64.of_string ("-" ^ s))
              
            let cvt_nativeint_literal s =
              Nativeint.neg (Nativeint.of_string ("-" ^ s))
              
            let err error loc =
              raise (Loc.Exc_located (loc, (Error.E error)))
              
            let warn error loc =
              Format.eprintf "Warning: %a: %a@." Loc.print loc Error.print
                error
              
            let __ocaml_lex_tables =
              {
                Lexing.lex_base =
                  "\000\000\222\255\223\255\224\000\225\255\253\000\035\001\072\001\
    \109\001\146\001\091\000\183\001\068\000\190\001\218\001\226\255\
    \122\000\002\002\071\002\110\002\176\000\243\255\172\002\133\003\
    \085\004\037\005\004\006\096\006\124\000\001\000\255\255\048\007\
    \253\255\000\008\252\255\208\008\160\009\112\010\064\011\025\012\
    \233\012\185\013\137\014\089\015\041\016\249\016\201\017\153\018\
    \091\000\121\019\091\000\122\000\095\000\242\255\241\255\240\255\
    \247\002\224\000\096\000\253\000\208\003\154\019\221\019\004\020\
    \071\020\110\020\109\000\238\255\129\020\150\000\213\020\252\020\
    \063\021\231\255\230\255\229\255\102\021\169\021\208\021\019\022\
    \058\022\024\001\227\255\228\255\237\255\093\022\130\022\167\022\
    \204\022\241\022\022\023\059\023\094\023\131\023\168\023\205\023\
    \242\023\023\024\060\024\097\024\199\004\020\002\004\000\232\255\
    \007\000\153\000\054\001\008\000\005\000\232\255\134\024\141\024\
    \178\024\215\024\252\024\003\025\176\004\128\005\040\025\139\007\
    \077\025\112\025\149\025\186\025\248\025\241\255\006\000\242\255\
    \243\255\072\002\251\255\210\026\100\000\116\000\123\000\255\255\
    \254\255\253\255\018\027\209\027\170\028\122\029\090\030\128\000\
    \027\001\129\000\130\000\249\255\248\255\247\255\212\006\127\002\
    \141\000\246\255\137\002\142\000\245\255\075\008\144\000\244\255\
    \068\002\247\255\248\255\009\000\249\255\117\030\255\255\250\255\
    \027\009\025\003\253\255\074\001\070\001\046\003\252\255\235\009\
    \251\255\156\030\223\030\190\031\220\031\255\255\187\032\154\033\
    \187\033\251\033\255\255\203\034\254\255\201\001\251\255\012\000\
    \252\255\253\255\122\000\043\001\255\255\011\035\202\035\163\036\
    \115\037\254\255\115\038\253\255\254\255\054\001\068\039\035\040\
    \255\255\091\001\243\040\206\001\251\255\080\001\013\000\253\255\
    \254\255\255\255\252\255\051\041\242\041\203\042\155\043";
                Lexing.lex_backtrk =
                  "\255\255\255\255\255\255\031\000\255\255\029\000\031\000\031\000\
    \031\000\031\000\029\000\029\000\029\000\029\000\029\000\255\255\
    \029\000\031\000\031\000\029\000\029\000\255\255\006\000\006\000\
    \005\000\004\000\031\000\031\000\001\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\011\000\010\000\009\000\008\000\011\000\
    \007\000\011\000\011\000\011\000\006\000\006\000\006\000\007\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \015\000\015\000\015\000\255\255\255\255\255\255\255\255\255\255\
    \029\000\029\000\016\000\255\255\029\000\255\255\255\255\029\000\
    \255\255\255\255\255\255\255\255\029\000\029\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\022\000\
    \021\000\019\000\031\000\019\000\019\000\019\000\019\000\029\000\
    \019\000\255\255\020\000\031\000\255\255\255\255\023\000\255\255\
    \255\255\255\255\255\255\255\255\023\000\255\255\255\255\255\255\
    \255\255\029\000\255\255\029\000\255\255\029\000\029\000\029\000\
    \029\000\031\000\031\000\031\000\255\255\255\255\013\000\255\255\
    \255\255\014\000\255\255\003\000\014\000\014\000\014\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\006\000\255\255\008\000\255\255\255\255\
    \005\000\005\000\255\255\001\000\001\000\255\255\255\255\255\255\
    \255\255\000\000\001\000\001\000\255\255\255\255\002\000\002\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\003\000\
    \255\255\255\255\004\000\004\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\002\000\002\000\002\000\
    \255\255\255\255\255\255\255\255\255\255\004\000\002\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255";
                Lexing.lex_default =
                  "\001\000\000\000\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\050\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\055\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\104\000\255\255\255\255\000\000\
    \104\000\105\000\104\000\107\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\125\000\000\000\255\255\000\000\
    \000\000\143\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\255\255\255\255\000\000\
    \161\000\000\000\000\000\255\255\000\000\167\000\000\000\000\000\
    \255\255\255\255\000\000\255\255\255\255\255\255\000\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\190\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\206\000\000\000\000\000\255\255\255\255\255\255\
    \000\000\255\255\255\255\212\000\000\000\255\255\255\255\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255";
                Lexing.lex_trans =
                  "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\028\000\030\000\030\000\028\000\029\000\103\000\109\000\
    \127\000\103\000\109\000\164\000\102\000\108\000\192\000\215\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \028\000\003\000\021\000\016\000\004\000\009\000\009\000\020\000\
    \019\000\005\000\018\000\003\000\015\000\003\000\014\000\009\000\
    \023\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\013\000\012\000\017\000\006\000\007\000\026\000\
    \009\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\011\000\003\000\005\000\009\000\025\000\
    \015\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\010\000\008\000\005\000\027\000\015\000\
    \118\000\118\000\054\000\101\000\052\000\028\000\053\000\054\000\
    \028\000\116\000\118\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\119\000\067\000\117\000\
    \116\000\116\000\101\000\118\000\028\000\136\000\139\000\069\000\
    \137\000\053\000\069\000\103\000\138\000\135\000\102\000\149\000\
    \148\000\147\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\153\000\156\000\069\000\159\000\
    \201\000\118\000\052\000\107\000\000\000\051\000\000\000\075\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\119\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\000\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \002\000\003\000\000\000\000\000\003\000\003\000\003\000\255\255\
    \255\255\000\000\003\000\003\000\049\000\003\000\003\000\003\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\081\000\097\000\097\000\081\000\145\000\085\000\000\000\
    \000\000\000\000\000\000\085\000\097\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\085\000\
    \081\000\085\000\085\000\085\000\003\000\097\000\003\000\106\000\
    \103\000\082\000\148\000\102\000\003\000\000\000\000\000\003\000\
    \009\000\009\000\000\000\000\000\085\000\003\000\003\000\172\000\
    \003\000\006\000\009\000\172\000\172\000\000\000\106\000\209\000\
    \105\000\000\000\085\000\097\000\003\000\086\000\003\000\006\000\
    \006\000\006\000\003\000\009\000\209\000\198\000\172\000\196\000\
    \000\000\003\000\172\000\197\000\003\000\122\000\122\000\000\000\
    \204\000\085\000\003\000\003\000\000\000\003\000\007\000\122\000\
    \000\000\085\000\085\000\000\000\209\000\000\000\000\000\003\000\
    \085\000\009\000\121\000\000\000\007\000\007\000\007\000\003\000\
    \122\000\209\000\220\000\000\000\218\000\000\000\003\000\000\000\
    \219\000\003\000\009\000\009\000\000\000\204\000\005\000\003\000\
    \003\000\255\255\003\000\006\000\009\000\000\000\000\000\086\000\
    \085\000\003\000\000\000\000\000\003\000\005\000\122\000\086\000\
    \000\000\006\000\006\000\006\000\003\000\009\000\000\000\000\000\
    \255\255\000\000\000\000\003\000\000\000\000\000\003\000\009\000\
    \009\000\000\000\000\000\095\000\003\000\003\000\000\000\003\000\
    \009\000\009\000\000\000\000\000\121\000\005\000\003\000\000\000\
    \000\000\003\000\005\000\009\000\099\000\000\000\009\000\009\000\
    \009\000\003\000\009\000\192\000\000\000\000\000\191\000\000\000\
    \215\000\000\000\000\000\214\000\118\000\118\000\000\000\000\000\
    \000\000\000\000\000\000\112\000\112\000\116\000\118\000\005\000\
    \000\000\086\000\005\000\003\000\110\000\112\000\003\000\095\000\
    \009\000\117\000\217\000\117\000\116\000\116\000\000\000\118\000\
    \115\000\000\000\110\000\113\000\113\000\000\000\112\000\112\000\
    \112\000\000\000\000\000\085\000\000\000\195\000\000\000\194\000\
    \113\000\112\000\213\000\000\000\000\000\000\000\099\000\095\000\
    \003\000\000\000\000\000\000\000\111\000\118\000\110\000\110\000\
    \110\000\000\000\112\000\005\000\112\000\101\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\000\000\003\000\009\000\
    \009\000\000\000\000\000\085\000\003\000\003\000\000\000\003\000\
    \006\000\009\000\000\000\117\000\101\000\000\000\255\255\085\000\
    \112\000\000\000\111\000\005\000\087\000\000\000\089\000\006\000\
    \006\000\003\000\088\000\000\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\164\000\000\000\
    \000\000\163\000\145\000\000\000\000\000\144\000\111\000\085\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\085\000\
    \009\000\000\000\000\000\000\000\000\000\000\000\166\000\000\000\
    \003\000\000\000\000\000\003\000\003\000\003\000\000\000\146\000\
    \084\000\003\000\003\000\000\000\003\000\003\000\003\000\061\000\
    \000\000\000\000\061\000\000\000\000\000\000\000\086\000\085\000\
    \003\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\061\000\062\000\
    \000\000\000\000\062\000\065\000\065\000\000\000\000\000\000\000\
    \066\000\062\000\000\000\062\000\063\000\065\000\000\000\000\000\
    \165\000\000\000\000\000\003\000\142\000\003\000\000\000\000\000\
    \064\000\000\000\063\000\063\000\063\000\062\000\065\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\003\000\000\000\003\000\000\000\000\000\
    \000\000\193\000\062\000\000\000\065\000\000\000\216\000\000\000\
    \000\000\000\000\000\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\040\000\000\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\000\000\000\000\
    \000\000\000\000\064\000\000\000\062\000\035\000\035\000\035\000\
    \035\000\039\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \037\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \000\000\000\000\000\000\022\000\000\000\035\000\035\000\035\000\
    \035\000\039\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \038\000\035\000\036\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\162\000\000\000\000\000\000\000\
    \255\255\173\000\173\000\173\000\173\000\173\000\173\000\173\000\
    \173\000\173\000\173\000\000\000\000\000\000\000\000\000\000\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\174\000\174\000\
    \174\000\174\000\174\000\174\000\174\000\174\000\174\000\174\000\
    \000\000\000\000\000\000\000\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\040\000\000\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\035\000\041\000\
    \035\000\035\000\039\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\037\000\035\000\035\000\042\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\043\000\035\000\035\000\
    \000\000\000\000\000\000\000\000\022\000\000\000\035\000\041\000\
    \035\000\035\000\039\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\038\000\035\000\036\000\042\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\043\000\035\000\035\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\050\000\050\000\050\000\050\000\050\000\050\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\050\000\050\000\050\000\050\000\050\000\050\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\024\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \000\000\000\000\000\000\000\000\024\000\000\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \106\000\103\000\000\000\000\000\102\000\118\000\118\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\116\000\118\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\106\000\
    \000\000\105\000\116\000\000\000\117\000\116\000\116\000\000\000\
    \118\000\000\000\000\000\000\000\000\000\000\000\000\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\118\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\116\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\025\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \000\000\000\000\000\000\000\000\025\000\000\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \000\000\000\000\000\000\000\000\000\000\118\000\118\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\116\000\118\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\116\000\000\000\117\000\116\000\116\000\000\000\
    \118\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\118\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\116\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\000\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\003\000\000\000\000\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\003\000\003\000\
    \000\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\033\000\000\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \003\000\003\000\003\000\000\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\003\000\003\000\000\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\031\000\
    \000\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\000\000\003\000\000\000\003\000\000\000\
    \000\000\000\000\000\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\000\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\000\000\000\000\000\000\000\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\032\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\000\000\000\000\000\000\000\000\031\000\
    \000\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\000\000\000\000\000\000\000\000\000\000\
    \120\000\120\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\119\000\120\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\119\000\000\000\119\000\
    \119\000\119\000\000\000\120\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\120\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\119\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\033\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\034\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\000\000\000\000\000\000\000\000\033\000\
    \000\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\035\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\175\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\175\000\175\000\175\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\176\000\176\000\176\000\176\000\176\000\
    \176\000\176\000\176\000\176\000\176\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\176\000\176\000\176\000\176\000\
    \176\000\176\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\176\000\176\000\176\000\176\000\
    \176\000\176\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\048\000\000\000\048\000\000\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\039\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \040\000\000\000\035\000\035\000\035\000\035\000\039\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\046\000\046\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \035\000\035\000\035\000\035\000\035\000\037\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \044\000\000\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \035\000\035\000\035\000\035\000\035\000\038\000\035\000\036\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\037\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \045\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\038\000\035\000\036\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\046\000\046\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\037\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \046\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\038\000\035\000\036\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \047\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\058\000\000\000\058\000\000\000\000\000\000\000\000\000\
    \058\000\000\000\000\000\061\000\000\000\000\000\061\000\000\000\
    \000\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\061\000\079\000\000\000\000\000\079\000\079\000\
    \079\000\000\000\000\000\000\000\080\000\079\000\000\000\079\000\
    \079\000\079\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\079\000\058\000\079\000\079\000\
    \079\000\079\000\079\000\058\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\069\000\058\000\
    \000\000\069\000\000\000\058\000\000\000\058\000\000\000\000\000\
    \000\000\056\000\000\000\000\000\000\000\000\000\079\000\000\000\
    \079\000\000\000\000\000\000\000\000\000\069\000\070\000\000\000\
    \000\000\070\000\070\000\070\000\000\000\000\000\073\000\072\000\
    \070\000\000\000\070\000\070\000\070\000\069\000\000\000\000\000\
    \069\000\000\000\000\000\000\000\000\000\000\000\079\000\070\000\
    \079\000\070\000\070\000\070\000\070\000\070\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\069\000\070\000\000\000\000\000\
    \070\000\071\000\071\000\000\000\000\000\073\000\072\000\070\000\
    \000\000\070\000\078\000\071\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\000\000\070\000\000\000\000\000\078\000\000\000\
    \078\000\078\000\078\000\070\000\071\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \069\000\000\000\000\000\069\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\000\000\070\000\000\000\000\000\000\000\000\000\
    \070\000\000\000\071\000\000\000\000\000\000\000\000\000\069\000\
    \070\000\000\000\000\000\070\000\077\000\077\000\000\000\000\000\
    \073\000\072\000\070\000\000\000\070\000\076\000\077\000\069\000\
    \000\000\255\255\069\000\000\000\000\000\000\000\000\000\000\000\
    \078\000\076\000\070\000\076\000\076\000\076\000\070\000\077\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\069\000\070\000\
    \000\000\000\000\070\000\071\000\071\000\000\000\068\000\073\000\
    \072\000\070\000\000\000\070\000\071\000\071\000\000\000\000\000\
    \000\000\000\000\000\000\070\000\000\000\077\000\068\000\068\000\
    \071\000\068\000\071\000\071\000\071\000\070\000\071\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\068\000\000\000\068\000\068\000\068\000\
    \000\000\068\000\000\000\076\000\000\000\070\000\000\000\000\000\
    \000\000\068\000\070\000\000\000\071\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\069\000\068\000\
    \000\000\069\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\068\000\071\000\000\000\070\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\069\000\070\000\000\000\
    \000\000\070\000\070\000\070\000\068\000\068\000\074\000\072\000\
    \070\000\000\000\070\000\070\000\070\000\069\000\000\000\000\000\
    \069\000\000\000\000\000\000\000\000\000\000\000\000\000\070\000\
    \000\000\070\000\070\000\070\000\070\000\070\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\069\000\070\000\000\000\000\000\
    \070\000\071\000\071\000\000\000\068\000\074\000\072\000\070\000\
    \000\000\070\000\071\000\071\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\000\000\070\000\000\000\000\000\071\000\000\000\
    \071\000\071\000\071\000\070\000\071\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \069\000\000\000\000\000\069\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\000\000\070\000\000\000\000\000\000\000\068\000\
    \070\000\000\000\071\000\000\000\000\000\000\000\000\000\069\000\
    \070\000\000\000\000\000\070\000\070\000\070\000\000\000\000\000\
    \000\000\072\000\070\000\000\000\070\000\070\000\070\000\069\000\
    \000\000\000\000\069\000\000\000\000\000\000\000\000\000\068\000\
    \071\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\069\000\070\000\
    \000\000\000\000\070\000\077\000\077\000\000\000\000\000\074\000\
    \072\000\070\000\000\000\070\000\076\000\077\000\000\000\000\000\
    \000\000\000\000\000\000\070\000\000\000\070\000\000\000\000\000\
    \076\000\000\000\076\000\076\000\076\000\070\000\077\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\069\000\000\000\000\000\069\000\000\000\000\000\
    \000\000\000\000\000\000\070\000\000\000\070\000\000\000\000\000\
    \000\000\000\000\070\000\000\000\077\000\000\000\000\000\000\000\
    \000\000\069\000\070\000\000\000\000\000\070\000\077\000\077\000\
    \000\000\068\000\074\000\072\000\070\000\000\000\070\000\077\000\
    \077\000\069\000\000\000\000\000\069\000\000\000\000\000\000\000\
    \000\000\000\000\076\000\077\000\070\000\077\000\077\000\077\000\
    \070\000\077\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \069\000\070\000\000\000\000\000\070\000\071\000\071\000\000\000\
    \000\000\074\000\072\000\070\000\000\000\070\000\078\000\071\000\
    \000\000\000\000\000\000\000\000\068\000\070\000\000\000\077\000\
    \000\000\000\000\078\000\000\000\078\000\078\000\078\000\070\000\
    \071\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\081\000\000\000\000\000\081\000\
    \000\000\000\000\000\000\000\000\068\000\077\000\000\000\070\000\
    \000\000\000\000\000\000\000\000\070\000\000\000\071\000\000\000\
    \000\000\000\000\000\000\081\000\079\000\000\000\000\000\079\000\
    \079\000\079\000\000\000\000\000\083\000\080\000\079\000\000\000\
    \079\000\079\000\079\000\081\000\000\000\000\000\081\000\000\000\
    \000\000\000\000\000\000\000\000\078\000\079\000\070\000\079\000\
    \079\000\079\000\079\000\079\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\081\000\079\000\000\000\000\000\079\000\079\000\
    \079\000\000\000\000\000\000\000\080\000\079\000\000\000\079\000\
    \079\000\079\000\000\000\000\000\000\000\000\000\000\000\079\000\
    \000\000\079\000\000\000\000\000\079\000\000\000\079\000\079\000\
    \079\000\079\000\079\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\097\000\097\000\000\000\000\000\085\000\000\000\
    \000\000\000\000\000\000\085\000\097\000\000\000\000\000\079\000\
    \000\000\079\000\000\000\000\000\000\000\000\000\079\000\085\000\
    \079\000\085\000\085\000\085\000\000\000\097\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\000\000\003\000\009\000\
    \009\000\000\000\000\000\005\000\003\000\003\000\000\000\003\000\
    \006\000\009\000\000\000\000\000\000\000\000\000\079\000\000\000\
    \079\000\000\000\085\000\097\000\086\000\000\000\006\000\006\000\
    \006\000\003\000\009\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\000\000\003\000\009\000\009\000\000\000\000\000\
    \005\000\003\000\003\000\000\000\003\000\006\000\009\000\000\000\
    \000\000\085\000\085\000\000\000\000\000\000\000\003\000\085\000\
    \009\000\086\000\000\000\006\000\006\000\006\000\003\000\009\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\000\000\000\000\
    \003\000\009\000\009\000\000\000\000\000\095\000\003\000\003\000\
    \000\000\003\000\009\000\009\000\000\000\000\000\086\000\005\000\
    \003\000\000\000\000\000\003\000\085\000\009\000\099\000\000\000\
    \009\000\009\000\009\000\003\000\009\000\000\000\000\000\000\000\
    \000\000\000\000\091\000\000\000\000\000\003\000\094\000\094\000\
    \000\000\000\000\085\000\091\000\091\000\000\000\091\000\092\000\
    \094\000\000\000\000\000\086\000\005\000\003\000\000\000\000\000\
    \003\000\095\000\009\000\093\000\000\000\006\000\092\000\090\000\
    \091\000\094\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \000\000\000\000\003\000\009\000\009\000\000\000\000\000\085\000\
    \003\000\003\000\000\000\003\000\006\000\009\000\000\000\000\000\
    \099\000\095\000\003\000\000\000\000\000\091\000\085\000\094\000\
    \086\000\000\000\006\000\006\000\098\000\003\000\009\000\000\000\
    \000\000\000\000\000\000\000\000\091\000\000\000\000\000\003\000\
    \091\000\091\000\000\000\000\000\000\000\091\000\091\000\000\000\
    \091\000\091\000\091\000\000\000\000\000\093\000\085\000\091\000\
    \000\000\000\000\003\000\085\000\009\000\091\000\000\000\003\000\
    \091\000\003\000\091\000\091\000\000\000\000\000\000\000\091\000\
    \000\000\000\000\003\000\094\000\094\000\000\000\000\000\085\000\
    \091\000\091\000\000\000\091\000\092\000\094\000\000\000\000\000\
    \000\000\000\000\086\000\085\000\003\000\000\000\000\000\091\000\
    \093\000\091\000\006\000\092\000\006\000\091\000\094\000\000\000\
    \000\000\000\000\000\000\000\000\091\000\000\000\000\000\003\000\
    \094\000\094\000\000\000\000\000\005\000\091\000\091\000\000\000\
    \091\000\092\000\094\000\000\000\000\000\000\000\000\000\091\000\
    \000\000\091\000\091\000\085\000\094\000\093\000\000\000\006\000\
    \092\000\006\000\091\000\094\000\000\000\000\000\000\000\000\000\
    \000\000\091\000\000\000\000\000\003\000\094\000\094\000\000\000\
    \000\000\095\000\091\000\091\000\000\000\091\000\094\000\094\000\
    \000\000\000\000\093\000\085\000\091\000\000\000\000\000\091\000\
    \085\000\094\000\096\000\000\000\009\000\094\000\009\000\091\000\
    \094\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\097\000\097\000\000\000\000\000\095\000\000\000\
    \000\000\000\000\000\000\097\000\097\000\000\000\000\000\093\000\
    \005\000\091\000\000\000\000\000\091\000\095\000\094\000\097\000\
    \000\000\097\000\097\000\097\000\000\000\097\000\000\000\000\000\
    \000\000\000\000\000\000\091\000\000\000\000\000\003\000\094\000\
    \094\000\000\000\000\000\095\000\091\000\091\000\000\000\091\000\
    \094\000\094\000\000\000\000\000\096\000\095\000\091\000\000\000\
    \000\000\000\000\095\000\097\000\096\000\000\000\009\000\094\000\
    \009\000\091\000\094\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\097\000\097\000\000\000\000\000\
    \095\000\000\000\000\000\000\000\000\000\097\000\097\000\000\000\
    \000\000\097\000\095\000\000\000\000\000\000\000\091\000\095\000\
    \094\000\097\000\000\000\097\000\097\000\097\000\000\000\097\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\000\000\000\000\
    \003\000\009\000\009\000\000\000\000\000\085\000\003\000\003\000\
    \000\000\003\000\006\000\009\000\000\000\000\000\096\000\095\000\
    \091\000\000\000\000\000\000\000\095\000\097\000\086\000\000\000\
    \006\000\006\000\006\000\003\000\009\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\000\000\003\000\009\000\009\000\
    \000\000\000\000\095\000\003\000\003\000\000\000\003\000\009\000\
    \009\000\000\000\000\000\097\000\095\000\000\000\000\000\000\000\
    \003\000\085\000\009\000\099\000\000\000\009\000\009\000\009\000\
    \003\000\009\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\112\000\112\000\000\000\000\000\085\000\
    \000\000\000\000\112\000\112\000\110\000\112\000\005\000\000\000\
    \086\000\085\000\003\000\110\000\112\000\003\000\095\000\009\000\
    \111\000\000\000\110\000\110\000\110\000\000\000\112\000\111\000\
    \000\000\110\000\110\000\110\000\000\000\112\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\112\000\
    \112\000\000\000\000\000\095\000\000\000\099\000\095\000\003\000\
    \112\000\112\000\000\000\085\000\112\000\000\000\000\000\000\000\
    \000\000\000\000\085\000\112\000\114\000\000\000\112\000\112\000\
    \112\000\000\000\112\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\112\000\112\000\000\000\000\000\
    \085\000\000\000\111\000\085\000\000\000\110\000\112\000\000\000\
    \000\000\111\000\005\000\000\000\000\000\000\000\000\000\095\000\
    \112\000\111\000\000\000\110\000\110\000\110\000\000\000\112\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\112\000\112\000\000\000\000\000\095\000\000\000\000\000\
    \112\000\112\000\112\000\112\000\005\000\000\000\114\000\095\000\
    \000\000\110\000\112\000\000\000\085\000\112\000\114\000\000\000\
    \112\000\112\000\112\000\000\000\112\000\111\000\000\000\110\000\
    \110\000\110\000\000\000\112\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\118\000\118\000\000\000\
    \068\000\000\000\000\000\111\000\085\000\000\000\118\000\118\000\
    \000\000\095\000\112\000\000\000\000\000\000\000\000\000\000\000\
    \085\000\112\000\118\000\000\000\118\000\118\000\118\000\000\000\
    \118\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\120\000\120\000\000\000\068\000\000\000\000\000\
    \114\000\095\000\000\000\120\000\120\000\000\000\000\000\111\000\
    \005\000\000\000\000\000\068\000\000\000\000\000\118\000\120\000\
    \000\000\120\000\120\000\120\000\000\000\120\000\000\000\000\000\
    \000\000\003\000\000\000\000\000\003\000\122\000\122\000\000\000\
    \000\000\005\000\003\000\003\000\000\000\003\000\007\000\122\000\
    \000\000\000\000\000\000\068\000\118\000\000\000\000\000\000\000\
    \068\000\000\000\121\000\120\000\007\000\007\000\007\000\003\000\
    \122\000\000\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \000\000\003\000\122\000\122\000\000\000\000\000\095\000\003\000\
    \003\000\000\000\003\000\122\000\122\000\000\000\000\000\000\000\
    \068\000\120\000\000\000\000\000\003\000\005\000\122\000\123\000\
    \000\000\122\000\122\000\122\000\003\000\122\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\000\000\003\000\122\000\
    \122\000\000\000\000\000\095\000\003\000\003\000\000\000\003\000\
    \122\000\122\000\000\000\000\000\121\000\005\000\003\000\000\000\
    \000\000\003\000\095\000\122\000\123\000\000\000\122\000\122\000\
    \122\000\003\000\122\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\127\000\000\000\000\000\126\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\123\000\095\000\003\000\000\000\000\000\003\000\095\000\
    \122\000\000\000\130\000\000\000\000\000\000\000\000\000\129\000\
    \134\000\000\000\133\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\132\000\000\000\123\000\095\000\
    \003\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\000\000\000\000\000\000\000\000\131\000\
    \000\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\000\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\000\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \128\000\131\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\000\000\000\000\000\000\
    \000\000\131\000\000\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\000\000\000\000\000\000\
    \000\000\141\000\000\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\000\000\000\000\000\000\
    \000\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\000\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\000\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\000\000\000\000\000\000\000\000\
    \140\000\000\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \000\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \000\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \140\000\000\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\000\000\000\000\137\000\000\000\
    \000\000\000\000\138\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\000\000\000\000\000\000\
    \000\000\140\000\000\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\000\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\141\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\000\000\000\000\137\000\000\000\
    \000\000\000\000\000\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\000\000\000\000\000\000\
    \000\000\141\000\000\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\000\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\000\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\152\000\000\000\152\000\000\000\000\000\172\000\
    \000\000\152\000\171\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\000\000\170\000\000\000\170\000\
    \000\000\000\000\000\000\000\000\170\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\152\000\000\000\
    \000\000\000\000\000\000\000\000\152\000\177\000\000\000\000\000\
    \177\000\177\000\177\000\000\000\000\000\000\000\177\000\177\000\
    \152\000\177\000\177\000\177\000\152\000\000\000\152\000\000\000\
    \000\000\170\000\150\000\000\000\000\000\000\000\177\000\170\000\
    \177\000\177\000\177\000\177\000\177\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\170\000\000\000\000\000\000\000\170\000\
    \000\000\170\000\000\000\000\000\000\000\168\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \177\000\000\000\177\000\000\000\000\000\000\000\000\000\000\000\
    \179\000\000\000\000\000\179\000\179\000\179\000\000\000\000\000\
    \000\000\179\000\179\000\000\000\179\000\179\000\179\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \177\000\179\000\177\000\179\000\179\000\179\000\179\000\179\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\000\000\179\000\000\000\179\000\180\000\000\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\000\000\179\000\000\000\179\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\000\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\000\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\179\000\
    \000\000\000\000\179\000\179\000\179\000\000\000\000\000\000\000\
    \179\000\179\000\000\000\179\000\179\000\179\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \179\000\000\000\179\000\179\000\179\000\179\000\179\000\000\000\
    \000\000\000\000\000\000\180\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\000\000\000\000\
    \181\000\000\000\179\000\000\000\179\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\000\000\
    \000\000\000\000\179\000\180\000\179\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\000\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\000\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\183\000\000\000\000\000\183\000\
    \183\000\183\000\000\000\000\000\000\000\183\000\183\000\000\000\
    \183\000\183\000\183\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\183\000\000\000\183\000\
    \183\000\183\000\183\000\183\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\000\000\183\000\
    \000\000\183\000\184\000\000\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\000\000\183\000\
    \000\000\183\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\183\000\000\000\000\000\183\000\183\000\
    \183\000\000\000\000\000\000\000\183\000\183\000\000\000\183\000\
    \183\000\183\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\183\000\000\000\183\000\183\000\
    \183\000\183\000\183\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\184\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\184\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\000\000\183\000\186\000\
    \183\000\000\000\000\000\185\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\183\000\000\000\
    \183\000\000\000\184\000\000\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\000\000\000\000\
    \000\000\000\000\187\000\000\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\000\000\000\000\
    \000\000\000\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\000\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\000\000\000\000\188\000\
    \000\000\000\000\000\000\000\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\000\000\000\000\
    \000\000\000\000\187\000\000\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\000\000\000\000\
    \000\000\000\000\200\000\000\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\000\000\000\000\
    \000\000\000\000\000\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\000\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\000\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\000\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\000\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\000\000\000\000\000\000\
    \000\000\199\000\000\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\000\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\000\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\199\000\000\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\000\000\000\000\196\000\
    \000\000\000\000\000\000\197\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\000\000\000\000\
    \000\000\000\000\199\000\000\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\000\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\200\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\000\000\000\000\196\000\
    \000\000\000\000\000\000\000\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\000\000\000\000\
    \000\000\000\000\200\000\000\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\000\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\000\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\205\000\203\000\203\000\208\000\
    \203\000\203\000\000\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\205\000\203\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\204\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\203\000\203\000\
    \203\000\203\000\000\000\207\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\203\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\203\000\210\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\204\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\000\000\
    \000\000\000\000\000\000\210\000\000\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\209\000\000\000\000\000\000\000\
    \000\000\000\000\210\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\209\000\000\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\204\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\000\000\000\000\
    \000\000\000\000\210\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\000\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\204\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\000\000\000\000\
    \000\000\000\000\210\000\000\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\000\000\000\000\
    \000\000\000\000\222\000\000\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\000\000\000\000\
    \000\000\000\000\000\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\000\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\000\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\000\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\000\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\000\000\000\000\000\000\
    \000\000\221\000\000\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\000\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\000\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\221\000\000\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\000\000\000\000\218\000\
    \000\000\000\000\000\000\219\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\000\000\000\000\
    \000\000\000\000\221\000\000\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\000\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\222\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\000\000\000\000\218\000\
    \000\000\000\000\000\000\000\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\000\000\000\000\
    \000\000\000\000\222\000\000\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\000\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\000\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\000\000";
                Lexing.lex_check =
                  "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\029\000\000\000\000\000\102\000\108\000\
    \126\000\104\000\107\000\163\000\104\000\107\000\191\000\214\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\012\000\
    \010\000\010\000\050\000\016\000\051\000\028\000\052\000\058\000\
    \028\000\010\000\010\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\010\000\066\000\010\000\
    \010\000\010\000\016\000\010\000\028\000\133\000\132\000\069\000\
    \132\000\051\000\069\000\105\000\132\000\134\000\105\000\143\000\
    \145\000\146\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\152\000\155\000\069\000\158\000\
    \194\000\010\000\020\000\105\000\255\255\020\000\255\255\069\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\010\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\255\255\255\255\003\000\003\000\003\000\104\000\
    \107\000\255\255\003\000\003\000\020\000\003\000\003\000\003\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\003\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\081\000\005\000\005\000\081\000\144\000\005\000\255\255\
    \255\255\255\255\255\255\005\000\005\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\005\000\
    \081\000\005\000\005\000\005\000\003\000\005\000\003\000\106\000\
    \106\000\081\000\144\000\106\000\006\000\255\255\255\255\006\000\
    \006\000\006\000\255\255\255\255\006\000\006\000\006\000\172\000\
    \006\000\006\000\006\000\171\000\171\000\255\255\106\000\205\000\
    \106\000\255\255\005\000\005\000\003\000\006\000\003\000\006\000\
    \006\000\006\000\006\000\006\000\205\000\195\000\172\000\195\000\
    \255\255\007\000\171\000\195\000\007\000\007\000\007\000\255\255\
    \205\000\007\000\007\000\007\000\255\255\007\000\007\000\007\000\
    \255\255\005\000\005\000\255\255\209\000\255\255\255\255\006\000\
    \006\000\006\000\007\000\255\255\007\000\007\000\007\000\007\000\
    \007\000\209\000\213\000\255\255\213\000\255\255\008\000\255\255\
    \213\000\008\000\008\000\008\000\255\255\209\000\008\000\008\000\
    \008\000\105\000\008\000\008\000\008\000\255\255\255\255\006\000\
    \006\000\006\000\255\255\255\255\007\000\007\000\007\000\008\000\
    \255\255\008\000\008\000\008\000\008\000\008\000\255\255\255\255\
    \020\000\255\255\255\255\009\000\255\255\255\255\009\000\009\000\
    \009\000\255\255\255\255\009\000\009\000\009\000\255\255\009\000\
    \009\000\009\000\255\255\255\255\007\000\007\000\007\000\255\255\
    \255\255\008\000\008\000\008\000\009\000\255\255\009\000\009\000\
    \009\000\009\000\009\000\189\000\255\255\255\255\189\000\255\255\
    \211\000\255\255\255\255\211\000\011\000\011\000\255\255\255\255\
    \255\255\255\255\255\255\013\000\013\000\011\000\011\000\013\000\
    \255\255\008\000\008\000\008\000\013\000\013\000\009\000\009\000\
    \009\000\011\000\211\000\011\000\011\000\011\000\255\255\011\000\
    \013\000\255\255\013\000\013\000\013\000\255\255\013\000\014\000\
    \014\000\255\255\255\255\014\000\255\255\189\000\255\255\189\000\
    \014\000\014\000\211\000\255\255\255\255\255\255\009\000\009\000\
    \009\000\255\255\255\255\255\255\014\000\011\000\014\000\014\000\
    \014\000\255\255\014\000\013\000\013\000\101\000\255\255\255\255\
    \255\255\255\255\255\255\017\000\255\255\255\255\017\000\017\000\
    \017\000\255\255\255\255\017\000\017\000\017\000\255\255\017\000\
    \017\000\017\000\255\255\011\000\101\000\255\255\106\000\014\000\
    \014\000\255\255\013\000\013\000\017\000\255\255\017\000\017\000\
    \017\000\017\000\017\000\255\255\101\000\101\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\160\000\255\255\
    \255\255\160\000\129\000\255\255\255\255\129\000\014\000\014\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\017\000\017\000\
    \017\000\255\255\255\255\255\255\255\255\255\255\160\000\255\255\
    \018\000\255\255\255\255\018\000\018\000\018\000\255\255\129\000\
    \018\000\018\000\018\000\255\255\018\000\018\000\018\000\019\000\
    \255\255\255\255\019\000\255\255\255\255\255\255\017\000\017\000\
    \017\000\018\000\255\255\018\000\018\000\018\000\018\000\018\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\019\000\019\000\
    \255\255\255\255\019\000\019\000\019\000\255\255\255\255\255\255\
    \019\000\019\000\255\255\019\000\019\000\019\000\255\255\255\255\
    \160\000\255\255\255\255\018\000\129\000\018\000\255\255\255\255\
    \019\000\255\255\019\000\019\000\019\000\019\000\019\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\018\000\255\255\018\000\255\255\255\255\
    \255\255\189\000\019\000\255\255\019\000\255\255\211\000\255\255\
    \255\255\255\255\255\255\022\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\022\000\255\255\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\255\255\255\255\
    \255\255\255\255\019\000\255\255\019\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\255\255\
    \255\255\255\255\255\255\022\000\255\255\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\160\000\255\255\255\255\255\255\
    \129\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\255\255\255\255\255\255\255\255\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\173\000\173\000\
    \173\000\173\000\173\000\173\000\173\000\173\000\173\000\173\000\
    \255\255\255\255\255\255\255\255\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\255\255\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\255\255\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\023\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\023\000\255\255\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \255\255\255\255\255\255\255\255\023\000\255\255\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\060\000\060\000\060\000\060\000\060\000\060\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\060\000\060\000\060\000\060\000\060\000\060\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\255\255\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\024\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \255\255\255\255\255\255\255\255\024\000\255\255\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \100\000\100\000\255\255\255\255\100\000\116\000\116\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\116\000\116\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\100\000\
    \255\255\100\000\116\000\255\255\116\000\116\000\116\000\255\255\
    \116\000\255\255\255\255\255\255\255\255\255\255\255\255\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\116\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\116\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\025\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \255\255\255\255\255\255\255\255\025\000\255\255\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \255\255\255\255\255\255\255\255\255\255\117\000\117\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\117\000\117\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\117\000\255\255\117\000\117\000\117\000\255\255\
    \117\000\255\255\255\255\255\255\255\255\255\255\255\255\100\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\117\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\117\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\255\255\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\026\000\255\255\255\255\
    \026\000\026\000\026\000\255\255\255\255\255\255\026\000\026\000\
    \255\255\026\000\026\000\026\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\026\000\255\255\
    \026\000\026\000\026\000\026\000\026\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \026\000\255\255\026\000\026\000\255\255\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \026\000\027\000\026\000\255\255\027\000\027\000\027\000\255\255\
    \255\255\255\255\027\000\027\000\255\255\027\000\027\000\027\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\027\000\255\255\027\000\027\000\027\000\027\000\
    \027\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\027\000\255\255\027\000\027\000\
    \255\255\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\255\255\027\000\255\255\027\000\255\255\
    \255\255\255\255\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\150\000\150\000\150\000\
    \150\000\150\000\150\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\150\000\150\000\150\000\
    \150\000\150\000\150\000\255\255\255\255\255\255\255\255\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\031\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\031\000\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \119\000\119\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\119\000\119\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\119\000\255\255\119\000\
    \119\000\119\000\255\255\119\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\119\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\119\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\033\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\255\255\255\255\255\255\033\000\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\157\000\157\000\157\000\157\000\
    \157\000\157\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\157\000\157\000\157\000\157\000\
    \157\000\157\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\035\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\255\255\255\255\255\255\255\255\035\000\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\168\000\168\000\168\000\168\000\
    \168\000\168\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\168\000\168\000\168\000\168\000\
    \168\000\168\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\036\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\255\255\255\255\255\255\255\255\036\000\
    \255\255\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\175\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\175\000\175\000\175\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\175\000\175\000\175\000\175\000\
    \175\000\175\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\175\000\175\000\175\000\175\000\
    \175\000\175\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\255\255\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\037\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\255\255\255\255\255\255\255\255\037\000\
    \255\255\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\255\255\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\038\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\255\255\255\255\255\255\255\255\038\000\
    \255\255\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\255\255\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\255\255\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \039\000\255\255\255\255\255\255\039\000\255\255\039\000\255\255\
    \255\255\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\255\255\255\255\255\255\255\255\
    \039\000\255\255\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \255\255\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \040\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\255\255\255\255\255\255\255\255\
    \040\000\255\255\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \255\255\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \041\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\255\255\255\255\255\255\255\255\
    \041\000\255\255\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \255\255\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \042\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\255\255\
    \042\000\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \255\255\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \043\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\255\255\255\255\255\255\
    \043\000\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \044\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\255\255\255\255\255\255\255\255\
    \044\000\255\255\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \255\255\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \045\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\255\255\255\255\255\255\255\255\
    \045\000\255\255\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \255\255\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \046\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\255\255\255\255\255\255\255\255\
    \046\000\255\255\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \255\255\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \047\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\255\255\255\255\255\255\255\255\
    \047\000\255\255\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \255\255\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \255\255\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\049\000\255\255\049\000\255\255\255\255\255\255\255\255\
    \049\000\255\255\255\255\061\000\255\255\255\255\061\000\255\255\
    \255\255\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\061\000\061\000\255\255\255\255\061\000\061\000\
    \061\000\255\255\255\255\255\255\061\000\061\000\255\255\061\000\
    \061\000\061\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\061\000\049\000\061\000\061\000\
    \061\000\061\000\061\000\049\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\062\000\049\000\
    \255\255\062\000\255\255\049\000\255\255\049\000\255\255\255\255\
    \255\255\049\000\255\255\255\255\255\255\255\255\061\000\255\255\
    \061\000\255\255\255\255\255\255\255\255\062\000\062\000\255\255\
    \255\255\062\000\062\000\062\000\255\255\255\255\062\000\062\000\
    \062\000\255\255\062\000\062\000\062\000\063\000\255\255\255\255\
    \063\000\255\255\255\255\255\255\255\255\255\255\061\000\062\000\
    \061\000\062\000\062\000\062\000\062\000\062\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\063\000\063\000\255\255\255\255\
    \063\000\063\000\063\000\255\255\255\255\063\000\063\000\063\000\
    \255\255\063\000\063\000\063\000\255\255\255\255\255\255\255\255\
    \255\255\062\000\255\255\062\000\255\255\255\255\063\000\255\255\
    \063\000\063\000\063\000\063\000\063\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \064\000\255\255\255\255\064\000\255\255\255\255\255\255\255\255\
    \255\255\062\000\255\255\062\000\255\255\255\255\255\255\255\255\
    \063\000\255\255\063\000\255\255\255\255\255\255\255\255\064\000\
    \064\000\255\255\255\255\064\000\064\000\064\000\255\255\255\255\
    \064\000\064\000\064\000\255\255\064\000\064\000\064\000\065\000\
    \255\255\049\000\065\000\255\255\255\255\255\255\255\255\255\255\
    \063\000\064\000\063\000\064\000\064\000\064\000\064\000\064\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\065\000\065\000\
    \255\255\255\255\065\000\065\000\065\000\255\255\065\000\065\000\
    \065\000\065\000\255\255\065\000\065\000\065\000\255\255\255\255\
    \255\255\255\255\255\255\064\000\255\255\064\000\068\000\068\000\
    \065\000\068\000\065\000\065\000\065\000\065\000\065\000\068\000\
    \068\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\068\000\255\255\068\000\068\000\068\000\
    \255\255\068\000\255\255\064\000\255\255\064\000\255\255\255\255\
    \255\255\065\000\065\000\255\255\065\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\068\000\255\255\070\000\068\000\
    \255\255\070\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\065\000\065\000\255\255\065\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\070\000\070\000\255\255\
    \255\255\070\000\070\000\070\000\068\000\068\000\070\000\070\000\
    \070\000\255\255\070\000\070\000\070\000\071\000\255\255\255\255\
    \071\000\255\255\255\255\255\255\255\255\255\255\255\255\070\000\
    \255\255\070\000\070\000\070\000\070\000\070\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\071\000\071\000\255\255\255\255\
    \071\000\071\000\071\000\255\255\071\000\071\000\071\000\071\000\
    \255\255\071\000\071\000\071\000\255\255\255\255\255\255\255\255\
    \255\255\070\000\255\255\070\000\255\255\255\255\071\000\255\255\
    \071\000\071\000\071\000\071\000\071\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \072\000\255\255\255\255\072\000\255\255\255\255\255\255\255\255\
    \255\255\070\000\255\255\070\000\255\255\255\255\255\255\071\000\
    \071\000\255\255\071\000\255\255\255\255\255\255\255\255\072\000\
    \072\000\255\255\255\255\072\000\072\000\072\000\255\255\255\255\
    \255\255\072\000\072\000\255\255\072\000\072\000\072\000\076\000\
    \255\255\255\255\076\000\255\255\255\255\255\255\255\255\071\000\
    \071\000\072\000\071\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\076\000\076\000\
    \255\255\255\255\076\000\076\000\076\000\255\255\255\255\076\000\
    \076\000\076\000\255\255\076\000\076\000\076\000\255\255\255\255\
    \255\255\255\255\255\255\072\000\255\255\072\000\255\255\255\255\
    \076\000\255\255\076\000\076\000\076\000\076\000\076\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\077\000\255\255\255\255\077\000\255\255\255\255\
    \255\255\255\255\255\255\072\000\255\255\072\000\255\255\255\255\
    \255\255\255\255\076\000\255\255\076\000\255\255\255\255\255\255\
    \255\255\077\000\077\000\255\255\255\255\077\000\077\000\077\000\
    \255\255\077\000\077\000\077\000\077\000\255\255\077\000\077\000\
    \077\000\078\000\255\255\255\255\078\000\255\255\255\255\255\255\
    \255\255\255\255\076\000\077\000\076\000\077\000\077\000\077\000\
    \077\000\077\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \078\000\078\000\255\255\255\255\078\000\078\000\078\000\255\255\
    \255\255\078\000\078\000\078\000\255\255\078\000\078\000\078\000\
    \255\255\255\255\255\255\255\255\077\000\077\000\255\255\077\000\
    \255\255\255\255\078\000\255\255\078\000\078\000\078\000\078\000\
    \078\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\079\000\255\255\255\255\079\000\
    \255\255\255\255\255\255\255\255\077\000\077\000\255\255\077\000\
    \255\255\255\255\255\255\255\255\078\000\255\255\078\000\255\255\
    \255\255\255\255\255\255\079\000\079\000\255\255\255\255\079\000\
    \079\000\079\000\255\255\255\255\079\000\079\000\079\000\255\255\
    \079\000\079\000\079\000\080\000\255\255\255\255\080\000\255\255\
    \255\255\255\255\255\255\255\255\078\000\079\000\078\000\079\000\
    \079\000\079\000\079\000\079\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\000\080\000\255\255\255\255\080\000\080\000\
    \080\000\255\255\255\255\255\255\080\000\080\000\255\255\080\000\
    \080\000\080\000\255\255\255\255\255\255\255\255\255\255\079\000\
    \255\255\079\000\255\255\255\255\080\000\255\255\080\000\080\000\
    \080\000\080\000\080\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\085\000\085\000\255\255\255\255\085\000\255\255\
    \255\255\255\255\255\255\085\000\085\000\255\255\255\255\079\000\
    \255\255\079\000\255\255\255\255\255\255\255\255\080\000\085\000\
    \080\000\085\000\085\000\085\000\255\255\085\000\255\255\255\255\
    \255\255\255\255\255\255\086\000\255\255\255\255\086\000\086\000\
    \086\000\255\255\255\255\086\000\086\000\086\000\255\255\086\000\
    \086\000\086\000\255\255\255\255\255\255\255\255\080\000\255\255\
    \080\000\255\255\085\000\085\000\086\000\255\255\086\000\086\000\
    \086\000\086\000\086\000\255\255\255\255\255\255\255\255\255\255\
    \087\000\255\255\255\255\087\000\087\000\087\000\255\255\255\255\
    \087\000\087\000\087\000\255\255\087\000\087\000\087\000\255\255\
    \255\255\085\000\085\000\255\255\255\255\255\255\086\000\086\000\
    \086\000\087\000\255\255\087\000\087\000\087\000\087\000\087\000\
    \255\255\255\255\255\255\255\255\255\255\088\000\255\255\255\255\
    \088\000\088\000\088\000\255\255\255\255\088\000\088\000\088\000\
    \255\255\088\000\088\000\088\000\255\255\255\255\086\000\086\000\
    \086\000\255\255\255\255\087\000\087\000\087\000\088\000\255\255\
    \088\000\088\000\088\000\088\000\088\000\255\255\255\255\255\255\
    \255\255\255\255\089\000\255\255\255\255\089\000\089\000\089\000\
    \255\255\255\255\089\000\089\000\089\000\255\255\089\000\089\000\
    \089\000\255\255\255\255\087\000\087\000\087\000\255\255\255\255\
    \088\000\088\000\088\000\089\000\255\255\089\000\089\000\089\000\
    \089\000\089\000\255\255\255\255\255\255\255\255\255\255\090\000\
    \255\255\255\255\090\000\090\000\090\000\255\255\255\255\090\000\
    \090\000\090\000\255\255\090\000\090\000\090\000\255\255\255\255\
    \088\000\088\000\088\000\255\255\255\255\089\000\089\000\089\000\
    \090\000\255\255\090\000\090\000\090\000\090\000\090\000\255\255\
    \255\255\255\255\255\255\255\255\091\000\255\255\255\255\091\000\
    \091\000\091\000\255\255\255\255\255\255\091\000\091\000\255\255\
    \091\000\091\000\091\000\255\255\255\255\089\000\089\000\089\000\
    \255\255\255\255\090\000\090\000\090\000\091\000\255\255\091\000\
    \091\000\091\000\091\000\091\000\255\255\255\255\255\255\092\000\
    \255\255\255\255\092\000\092\000\092\000\255\255\255\255\092\000\
    \092\000\092\000\255\255\092\000\092\000\092\000\255\255\255\255\
    \255\255\255\255\090\000\090\000\090\000\255\255\255\255\091\000\
    \092\000\091\000\092\000\092\000\092\000\092\000\092\000\255\255\
    \255\255\255\255\255\255\255\255\093\000\255\255\255\255\093\000\
    \093\000\093\000\255\255\255\255\093\000\093\000\093\000\255\255\
    \093\000\093\000\093\000\255\255\255\255\255\255\255\255\091\000\
    \255\255\091\000\092\000\092\000\092\000\093\000\255\255\093\000\
    \093\000\093\000\093\000\093\000\255\255\255\255\255\255\255\255\
    \255\255\094\000\255\255\255\255\094\000\094\000\094\000\255\255\
    \255\255\094\000\094\000\094\000\255\255\094\000\094\000\094\000\
    \255\255\255\255\092\000\092\000\092\000\255\255\255\255\093\000\
    \093\000\093\000\094\000\255\255\094\000\094\000\094\000\094\000\
    \094\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\095\000\095\000\255\255\255\255\095\000\255\255\
    \255\255\255\255\255\255\095\000\095\000\255\255\255\255\093\000\
    \093\000\093\000\255\255\255\255\094\000\094\000\094\000\095\000\
    \255\255\095\000\095\000\095\000\255\255\095\000\255\255\255\255\
    \255\255\255\255\255\255\096\000\255\255\255\255\096\000\096\000\
    \096\000\255\255\255\255\096\000\096\000\096\000\255\255\096\000\
    \096\000\096\000\255\255\255\255\094\000\094\000\094\000\255\255\
    \255\255\255\255\095\000\095\000\096\000\255\255\096\000\096\000\
    \096\000\096\000\096\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\097\000\097\000\255\255\255\255\
    \097\000\255\255\255\255\255\255\255\255\097\000\097\000\255\255\
    \255\255\095\000\095\000\255\255\255\255\255\255\096\000\096\000\
    \096\000\097\000\255\255\097\000\097\000\097\000\255\255\097\000\
    \255\255\255\255\255\255\255\255\255\255\098\000\255\255\255\255\
    \098\000\098\000\098\000\255\255\255\255\098\000\098\000\098\000\
    \255\255\098\000\098\000\098\000\255\255\255\255\096\000\096\000\
    \096\000\255\255\255\255\255\255\097\000\097\000\098\000\255\255\
    \098\000\098\000\098\000\098\000\098\000\255\255\255\255\255\255\
    \255\255\255\255\099\000\255\255\255\255\099\000\099\000\099\000\
    \255\255\255\255\099\000\099\000\099\000\255\255\099\000\099\000\
    \099\000\255\255\255\255\097\000\097\000\255\255\255\255\255\255\
    \098\000\098\000\098\000\099\000\255\255\099\000\099\000\099\000\
    \099\000\099\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\110\000\110\000\255\255\255\255\110\000\
    \255\255\255\255\111\000\111\000\110\000\110\000\111\000\255\255\
    \098\000\098\000\098\000\111\000\111\000\099\000\099\000\099\000\
    \110\000\255\255\110\000\110\000\110\000\255\255\110\000\111\000\
    \255\255\111\000\111\000\111\000\255\255\111\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\112\000\
    \112\000\255\255\255\255\112\000\255\255\099\000\099\000\099\000\
    \112\000\112\000\255\255\110\000\110\000\255\255\255\255\255\255\
    \255\255\255\255\111\000\111\000\112\000\255\255\112\000\112\000\
    \112\000\255\255\112\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\113\000\113\000\255\255\255\255\
    \113\000\255\255\110\000\110\000\255\255\113\000\113\000\255\255\
    \255\255\111\000\111\000\255\255\255\255\255\255\255\255\112\000\
    \112\000\113\000\255\255\113\000\113\000\113\000\255\255\113\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\114\000\114\000\255\255\255\255\114\000\255\255\255\255\
    \115\000\115\000\114\000\114\000\115\000\255\255\112\000\112\000\
    \255\255\115\000\115\000\255\255\113\000\113\000\114\000\255\255\
    \114\000\114\000\114\000\255\255\114\000\115\000\255\255\115\000\
    \115\000\115\000\255\255\115\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\118\000\118\000\255\255\
    \118\000\255\255\255\255\113\000\113\000\255\255\118\000\118\000\
    \255\255\114\000\114\000\255\255\255\255\255\255\255\255\255\255\
    \115\000\115\000\118\000\255\255\118\000\118\000\118\000\255\255\
    \118\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\120\000\120\000\255\255\120\000\255\255\255\255\
    \114\000\114\000\255\255\120\000\120\000\255\255\255\255\115\000\
    \115\000\255\255\255\255\118\000\255\255\255\255\118\000\120\000\
    \255\255\120\000\120\000\120\000\255\255\120\000\255\255\255\255\
    \255\255\121\000\255\255\255\255\121\000\121\000\121\000\255\255\
    \255\255\121\000\121\000\121\000\255\255\121\000\121\000\121\000\
    \255\255\255\255\255\255\118\000\118\000\255\255\255\255\255\255\
    \120\000\255\255\121\000\120\000\121\000\121\000\121\000\121\000\
    \121\000\255\255\255\255\255\255\255\255\255\255\122\000\255\255\
    \255\255\122\000\122\000\122\000\255\255\255\255\122\000\122\000\
    \122\000\255\255\122\000\122\000\122\000\255\255\255\255\255\255\
    \120\000\120\000\255\255\255\255\121\000\121\000\121\000\122\000\
    \255\255\122\000\122\000\122\000\122\000\122\000\255\255\255\255\
    \255\255\255\255\255\255\123\000\255\255\255\255\123\000\123\000\
    \123\000\255\255\255\255\123\000\123\000\123\000\255\255\123\000\
    \123\000\123\000\255\255\255\255\121\000\121\000\121\000\255\255\
    \255\255\122\000\122\000\122\000\123\000\255\255\123\000\123\000\
    \123\000\123\000\123\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\124\000\255\255\255\255\124\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\122\000\122\000\122\000\255\255\255\255\123\000\123\000\
    \123\000\255\255\124\000\255\255\255\255\255\255\255\255\124\000\
    \124\000\255\255\124\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\124\000\255\255\123\000\123\000\
    \123\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\255\255\255\255\255\255\255\255\124\000\
    \255\255\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\255\255\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\255\255\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\131\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\131\000\255\255\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\255\255\255\255\255\255\
    \255\255\138\000\255\255\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\255\255\255\255\255\255\
    \255\255\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\255\255\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\255\255\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\255\255\255\255\255\255\255\255\
    \139\000\255\255\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \255\255\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \255\255\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\140\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \140\000\255\255\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\255\255\255\255\140\000\255\255\
    \255\255\255\255\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\255\255\255\255\255\255\
    \255\255\140\000\255\255\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\255\255\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\141\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\255\255\255\255\141\000\255\255\
    \255\255\255\255\255\255\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\255\255\255\255\255\255\
    \255\255\141\000\255\255\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\255\255\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\255\255\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\142\000\255\255\142\000\255\255\255\255\165\000\
    \255\255\142\000\165\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\255\255\165\000\255\255\165\000\
    \255\255\255\255\255\255\255\255\165\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\165\000\165\000\165\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\142\000\255\255\
    \255\255\255\255\255\255\255\255\142\000\177\000\255\255\255\255\
    \177\000\177\000\177\000\255\255\255\255\255\255\177\000\177\000\
    \142\000\177\000\177\000\177\000\142\000\255\255\142\000\255\255\
    \255\255\165\000\142\000\255\255\255\255\255\255\177\000\165\000\
    \177\000\177\000\177\000\177\000\177\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\165\000\255\255\255\255\255\255\165\000\
    \255\255\165\000\255\255\255\255\255\255\165\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \177\000\255\255\177\000\255\255\255\255\255\255\255\255\255\255\
    \178\000\255\255\255\255\178\000\178\000\178\000\255\255\255\255\
    \255\255\178\000\178\000\255\255\178\000\178\000\178\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \177\000\178\000\177\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\255\255\178\000\255\255\178\000\178\000\255\255\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\255\255\178\000\255\255\178\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\165\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\255\255\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\255\255\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\179\000\
    \255\255\255\255\179\000\179\000\179\000\255\255\255\255\255\255\
    \179\000\179\000\255\255\179\000\179\000\179\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \179\000\255\255\179\000\179\000\179\000\179\000\179\000\255\255\
    \255\255\255\255\255\255\180\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\255\255\255\255\
    \180\000\255\255\179\000\255\255\179\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\255\255\
    \255\255\255\255\179\000\180\000\179\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\255\255\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\255\255\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\182\000\255\255\255\255\182\000\
    \182\000\182\000\255\255\255\255\255\255\182\000\182\000\255\255\
    \182\000\182\000\182\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\182\000\255\255\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\255\255\182\000\
    \255\255\182\000\182\000\255\255\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\255\255\182\000\
    \255\255\182\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\255\255\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\255\255\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\183\000\255\255\255\255\183\000\183\000\
    \183\000\255\255\255\255\255\255\183\000\183\000\255\255\183\000\
    \183\000\183\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\183\000\255\255\183\000\183\000\
    \183\000\183\000\183\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\184\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\184\000\255\255\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\255\255\183\000\184\000\
    \183\000\255\255\255\255\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\183\000\255\255\
    \183\000\255\255\184\000\255\255\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\255\255\255\255\
    \255\255\255\255\185\000\255\255\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\255\255\255\255\
    \255\255\255\255\255\255\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\255\255\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\255\255\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\255\255\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\187\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\255\255\255\255\187\000\
    \255\255\255\255\255\255\255\255\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\255\255\255\255\
    \255\255\255\255\187\000\255\255\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\255\255\255\255\
    \255\255\255\255\197\000\255\255\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\255\255\255\255\
    \255\255\255\255\255\255\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\255\255\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\255\255\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\255\255\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\255\255\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\255\255\255\255\255\255\
    \255\255\198\000\255\255\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\255\255\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\255\255\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\199\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\199\000\255\255\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\255\255\255\255\199\000\
    \255\255\255\255\255\255\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\255\255\255\255\
    \255\255\255\255\199\000\255\255\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\255\255\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\200\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\255\255\255\255\200\000\
    \255\255\255\255\255\255\255\255\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\255\255\255\255\
    \255\255\255\255\200\000\255\255\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\255\255\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\255\255\200\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\255\255\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\202\000\202\000\
    \202\000\202\000\255\255\202\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\202\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\202\000\206\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\202\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\255\255\
    \255\255\255\255\255\255\206\000\255\255\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\255\255\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\255\255\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\207\000\255\255\255\255\255\255\
    \255\255\255\255\207\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\207\000\255\255\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\255\255\255\255\
    \255\255\255\255\207\000\255\255\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\255\255\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\210\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\255\255\255\255\
    \255\255\255\255\210\000\255\255\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\255\255\255\255\
    \255\255\255\255\219\000\255\255\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\255\255\255\255\
    \255\255\255\255\255\255\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\255\255\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\255\255\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\255\255\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\255\255\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\255\255\255\255\255\255\
    \255\255\220\000\255\255\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\255\255\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\255\255\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\221\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\221\000\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\255\255\255\255\221\000\
    \255\255\255\255\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\255\255\255\255\
    \255\255\255\255\221\000\255\255\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\222\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\255\255\255\255\222\000\
    \255\255\255\255\255\255\255\255\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\255\255\255\255\
    \255\255\255\255\222\000\255\255\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\255\255\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\255\255\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\255\255";
                Lexing.lex_base_code =
                  "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\027\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\066\000\101\000\136\000\
    \171\000\206\000\000\000\000\000\000\000\000\000\241\000\020\001\
    \055\001\000\000\000\000\018\000\090\001\125\001\160\001\195\001\
    \230\001\000\000\021\000\026\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\247\001\040\002\000\000\034\000\
    \000\000\000\000\003\000\000\000\000\000\049\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\001\000\000\000\000\000\
    \000\000\000\000\000\000\002\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\036\002\000\000\
    \253\002\000\000\000\000\000\000\061\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
                Lexing.lex_backtrk_code =
                  "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\034\000\000\000\
    \000\000\000\000\000\000\000\000\049\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
                Lexing.lex_default_code =
                  "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\041\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
                Lexing.lex_trans_code =
                  "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\058\000\058\000\000\000\058\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\007\000\044\000\000\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\007\000\012\000\000\000\000\000\012\000\
    \012\000\012\000\000\000\000\000\000\000\000\000\012\000\000\000\
    \012\000\012\000\012\000\007\000\000\000\000\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\012\000\000\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\007\000\015\000\000\000\000\000\015\000\015\000\
    \015\000\000\000\000\000\000\000\015\000\015\000\000\000\015\000\
    \015\000\015\000\000\000\000\000\000\000\000\000\000\000\012\000\
    \000\000\012\000\000\000\000\000\015\000\000\000\015\000\015\000\
    \015\000\015\000\015\000\000\000\000\000\000\000\012\000\000\000\
    \000\000\012\000\012\000\012\000\000\000\000\000\000\000\012\000\
    \012\000\000\000\012\000\012\000\012\000\000\000\000\000\012\000\
    \000\000\012\000\000\000\000\000\000\000\000\000\015\000\012\000\
    \015\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\012\000\000\000\000\000\012\000\012\000\012\000\000\000\
    \000\000\000\000\012\000\012\000\000\000\012\000\012\000\012\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\015\000\000\000\
    \015\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\000\000\000\000\000\000\012\000\000\000\000\000\012\000\
    \012\000\012\000\000\000\000\000\000\000\012\000\012\000\000\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\012\000\
    \000\000\000\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\000\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\000\000\
    \000\000\000\000\012\000\000\000\000\000\012\000\012\000\012\000\
    \000\000\000\000\000\000\012\000\012\000\000\000\012\000\012\000\
    \012\000\000\000\000\000\000\000\000\000\000\000\000\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\000\000\000\000\012\000\000\000\000\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\012\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \012\000\000\000\000\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\012\000\012\000\000\000\012\000\012\000\012\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\012\000\000\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\000\000\000\000\012\000\000\000\000\000\012\000\012\000\
    \012\000\000\000\000\000\000\000\012\000\012\000\000\000\012\000\
    \012\000\012\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\012\000\000\000\
    \000\000\012\000\012\000\012\000\000\000\000\000\000\000\012\000\
    \012\000\000\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\012\000\000\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\012\000\000\000\000\000\012\000\012\000\012\000\000\000\
    \000\000\000\000\012\000\012\000\000\000\012\000\012\000\012\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\012\000\000\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\000\000\000\000\000\000\015\000\000\000\000\000\015\000\
    \015\000\015\000\000\000\000\000\000\000\015\000\015\000\000\000\
    \015\000\015\000\015\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\000\000\012\000\012\000\015\000\012\000\015\000\
    \015\000\015\000\015\000\015\000\000\000\000\000\000\000\015\000\
    \000\000\000\000\015\000\015\000\015\000\000\000\000\000\000\000\
    \015\000\015\000\000\000\015\000\015\000\015\000\000\000\000\000\
    \000\000\029\000\000\000\000\000\012\000\000\000\012\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\015\000\
    \000\000\015\000\015\000\000\000\015\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\015\000\000\000\015\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\000\000\
    \000\000\000\000\000\000\058\000\000\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\000\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\000\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\058\000\000\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \000\000\000\000\000\000\000\000\058\000\000\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\000\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\000\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\000\000";
                Lexing.lex_check_code =
                  "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\016\000\105\000\165\000\171\000\105\000\165\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \016\000\255\255\105\000\000\000\019\000\106\000\255\255\019\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\255\255\019\000\019\000\255\255\255\255\019\000\
    \019\000\019\000\255\255\255\255\255\255\255\255\019\000\255\255\
    \019\000\019\000\019\000\061\000\255\255\255\255\061\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\019\000\255\255\019\000\
    \019\000\019\000\019\000\019\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\061\000\061\000\255\255\255\255\061\000\061\000\
    \061\000\255\255\255\255\255\255\061\000\061\000\255\255\061\000\
    \061\000\061\000\255\255\255\255\255\255\255\255\255\255\019\000\
    \255\255\019\000\255\255\255\255\061\000\255\255\061\000\061\000\
    \061\000\061\000\061\000\255\255\255\255\255\255\062\000\255\255\
    \255\255\062\000\062\000\062\000\255\255\255\255\255\255\062\000\
    \062\000\255\255\062\000\062\000\062\000\255\255\255\255\019\000\
    \255\255\019\000\255\255\255\255\255\255\255\255\061\000\062\000\
    \061\000\062\000\062\000\062\000\062\000\062\000\255\255\255\255\
    \255\255\063\000\255\255\255\255\063\000\063\000\063\000\255\255\
    \255\255\255\255\063\000\063\000\255\255\063\000\063\000\063\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\061\000\255\255\
    \061\000\062\000\063\000\062\000\063\000\063\000\063\000\063\000\
    \063\000\255\255\255\255\255\255\064\000\255\255\255\255\064\000\
    \064\000\064\000\255\255\255\255\255\255\064\000\064\000\255\255\
    \064\000\064\000\064\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\062\000\255\255\062\000\063\000\064\000\063\000\064\000\
    \064\000\064\000\064\000\064\000\255\255\255\255\255\255\065\000\
    \255\255\255\255\065\000\065\000\065\000\255\255\255\255\255\255\
    \065\000\065\000\255\255\065\000\065\000\065\000\255\255\255\255\
    \105\000\255\255\255\255\255\255\063\000\255\255\063\000\064\000\
    \065\000\064\000\065\000\065\000\065\000\065\000\065\000\255\255\
    \255\255\255\255\070\000\255\255\255\255\070\000\070\000\070\000\
    \255\255\255\255\255\255\070\000\070\000\255\255\070\000\070\000\
    \070\000\255\255\255\255\255\255\255\255\255\255\255\255\064\000\
    \255\255\064\000\065\000\070\000\065\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\255\255\071\000\255\255\255\255\
    \071\000\071\000\071\000\255\255\255\255\255\255\071\000\071\000\
    \255\255\071\000\071\000\071\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\065\000\255\255\065\000\070\000\071\000\070\000\
    \071\000\071\000\071\000\071\000\071\000\255\255\255\255\255\255\
    \072\000\255\255\255\255\072\000\072\000\072\000\255\255\255\255\
    \255\255\072\000\072\000\255\255\072\000\072\000\072\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\070\000\255\255\070\000\
    \071\000\072\000\071\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\076\000\255\255\255\255\076\000\076\000\
    \076\000\255\255\255\255\255\255\076\000\076\000\255\255\076\000\
    \076\000\076\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \071\000\255\255\071\000\072\000\076\000\072\000\076\000\076\000\
    \076\000\076\000\076\000\255\255\255\255\255\255\077\000\255\255\
    \255\255\077\000\077\000\077\000\255\255\255\255\255\255\077\000\
    \077\000\255\255\077\000\077\000\077\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\072\000\255\255\072\000\076\000\077\000\
    \076\000\077\000\077\000\077\000\077\000\077\000\255\255\255\255\
    \255\255\078\000\255\255\255\255\078\000\078\000\078\000\255\255\
    \255\255\255\255\078\000\078\000\255\255\078\000\078\000\078\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\076\000\255\255\
    \076\000\077\000\078\000\077\000\078\000\078\000\078\000\078\000\
    \078\000\255\255\255\255\255\255\079\000\255\255\255\255\079\000\
    \079\000\079\000\255\255\255\255\255\255\079\000\079\000\255\255\
    \079\000\079\000\079\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\077\000\255\255\077\000\078\000\079\000\078\000\079\000\
    \079\000\079\000\079\000\079\000\255\255\255\255\255\255\080\000\
    \255\255\255\255\080\000\080\000\080\000\255\255\255\255\255\255\
    \080\000\080\000\255\255\080\000\080\000\080\000\255\255\255\255\
    \255\255\100\000\255\255\255\255\078\000\255\255\078\000\079\000\
    \080\000\079\000\080\000\080\000\080\000\080\000\080\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\101\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\079\000\
    \255\255\079\000\080\000\255\255\080\000\255\255\255\255\255\255\
    \101\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \101\000\101\000\080\000\255\255\080\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\255\255\
    \255\255\255\255\255\255\182\000\255\255\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\255\255\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\255\255\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\184\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\184\000\255\255\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \255\255\255\255\255\255\255\255\184\000\255\255\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\255\255\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\255\255\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\255\255";
                Lexing.lex_code =
                  "\255\004\255\255\009\255\255\006\255\005\255\255\007\255\255\008\
    \255\255\000\007\255\000\006\001\008\255\000\005\255\011\255\010\
    \255\255\003\255\000\004\001\009\255\011\255\255\010\255\011\255\
    \255\000\004\001\009\003\010\002\011\255\001\255\255\000\001\255\
    ";
              }
              
            let rec token c lexbuf =
              (lexbuf.Lexing.lex_mem <- Array.make 12 (-1);
               __ocaml_lex_token_rec c lexbuf 0)
            and __ocaml_lex_token_rec c lexbuf __ocaml_lex_state =
              match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state
                      lexbuf
              with
              | 0 -> (update_loc c None 1 false 0; NEWLINE)
              | 1 ->
                  let x =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in BLANKS x
              | 2 ->
                  let x =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in LABEL x
              | 3 ->
                  let x =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in OPTLABEL x
              | 4 ->
                  let x =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in LIDENT x
              | 5 ->
                  let x =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in UIDENT x
              | 6 ->
                  let i =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in
                    (try INT ((cvt_int_literal i), i)
                     with
                     | Failure _ ->
                         err (Literal_overflow "int") (Loc.of_lexbuf lexbuf))
              | 7 ->
                  let f =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in
                    (try FLOAT ((float_of_string f), f)
                     with
                     | Failure _ ->
                         err (Literal_overflow "float")
                           (Loc.of_lexbuf lexbuf))
              | 8 ->
                  let i =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    (try INT32 ((cvt_int32_literal i), i)
                     with
                     | Failure _ ->
                         err (Literal_overflow "int32")
                           (Loc.of_lexbuf lexbuf))
              | 9 ->
                  let i =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    (try INT64 ((cvt_int64_literal i), i)
                     with
                     | Failure _ ->
                         err (Literal_overflow "int64")
                           (Loc.of_lexbuf lexbuf))
              | 10 ->
                  let i =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    (try NATIVEINT ((cvt_nativeint_literal i), i)
                     with
                     | Failure _ ->
                         err (Literal_overflow "nativeint")
                           (Loc.of_lexbuf lexbuf))
              | 11 ->
                  err (Invalid_literal (Lexing.lexeme lexbuf))
                    (Loc.of_lexbuf lexbuf)
              | 12 ->
                  (with_curr_loc string c;
                   let s = buff_contents c
                   in STRING ((TokenEval.string s), s))
              | 13 ->
                  let x =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    (update_loc c None 1 false 1;
                     CHAR ((TokenEval.char x), x))
              | 14 ->
                  let x =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in CHAR ((TokenEval.char x), x)
              | 15 ->
                  let c =
                    Lexing.sub_lexeme_char lexbuf
                      (lexbuf.Lexing.lex_start_pos + 2)
                  in
                    err (Illegal_escape (String.make 1 c))
                      (Loc.of_lexbuf lexbuf)
              | 16 ->
                  (store c; COMMENT (parse_nested comment (in_comment c)))
              | 17 ->
                  (warn Comment_start (Loc.of_lexbuf lexbuf);
                   parse comment (in_comment c);
                   COMMENT (buff_contents c))
              | 18 ->
                  (warn Comment_not_end (Loc.of_lexbuf lexbuf);
                   c.lexbuf.lex_curr_pos <- c.lexbuf.lex_curr_pos - 1;
                   SYMBOL "*")
              | 19 ->
                  let beginning =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 2)
                      lexbuf.Lexing.lex_curr_pos
                  in
                    if quotations c
                    then
                      (move_start_p (- (String.length beginning));
                       mk_quotation quotation c "" "" 2)
                    else parse (symbolchar_star ("<<" ^ beginning)) c
              | 20 ->
                  if quotations c
                  then
                    QUOTATION
                      {
                        q_name = "";
                        q_loc = "";
                        q_shift = 2;
                        q_contents = "";
                      }
                  else parse (symbolchar_star "<<>>") c
              | 21 ->
                  if quotations c
                  then with_curr_loc maybe_quotation_at c
                  else parse (symbolchar_star "<@") c
              | 22 ->
                  if quotations c
                  then with_curr_loc maybe_quotation_colon c
                  else parse (symbolchar_star "<:") c
              | 23 ->
                  let num =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0)
                      lexbuf.Lexing.lex_mem.(1)
                  and name =
                    Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3)
                      lexbuf.Lexing.lex_mem.(2) in
                  let inum = int_of_string num
                  in
                    (update_loc c name inum true 0;
                     LINE_DIRECTIVE (inum, name))
              | 24 ->
                  let op =
                    Lexing.sub_lexeme_char lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                  in ESCAPED_IDENT (String.make 1 op)
              | 25 ->
                  let op =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in ESCAPED_IDENT op
              | 26 ->
                  let op =
                    Lexing.sub_lexeme lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                      lexbuf.Lexing.lex_mem.(0)
                  in ESCAPED_IDENT op
              | 27 ->
                  let op =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in ESCAPED_IDENT op
              | 28 ->
                  let op =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0)
                      lexbuf.Lexing.lex_mem.(1)
                  in ESCAPED_IDENT op
              | 29 ->
                  let x =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in SYMBOL x
              | 30 ->
                  if antiquots c
                  then with_curr_loc dollar (shift 1 c)
                  else parse (symbolchar_star "$") c
              | 31 ->
                  let x =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in SYMBOL x
              | 32 ->
                  let pos = lexbuf.lex_curr_p
                  in
                    (lexbuf.lex_curr_p <-
                       {
                         (pos)
                         with
                         pos_bol = pos.pos_bol + 1;
                         pos_cnum = pos.pos_cnum + 1;
                       };
                     EOI)
              | 33 ->
                  let c =
                    Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos
                  in err (Illegal_character c) (Loc.of_lexbuf lexbuf)
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_token_rec c lexbuf __ocaml_lex_state)
            and comment c lexbuf = __ocaml_lex_comment_rec c lexbuf 124
            and __ocaml_lex_comment_rec c lexbuf __ocaml_lex_state =
              match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf
              with
              | 0 -> (store c; with_curr_loc comment c; parse comment c)
              | 1 -> store c
              | 2 ->
                  (store c;
                   if quotations c then with_curr_loc quotation c else ();
                   parse comment c)
              | 3 -> store_parse comment c
              | 4 ->
                  (store c;
                   (try with_curr_loc string c
                    with
                    | Loc.Exc_located (_, (Error.E Unterminated_string)) ->
                        err Unterminated_string_in_comment (loc c));
                   Buffer.add_char c.buffer '"';
                   parse comment c)
              | 5 -> store_parse comment c
              | 6 -> store_parse comment c
              | 7 -> (update_loc c None 1 false 1; store_parse comment c)
              | 8 -> store_parse comment c
              | 9 -> store_parse comment c
              | 10 -> store_parse comment c
              | 11 -> store_parse comment c
              | 12 -> err Unterminated_comment (loc c)
              | 13 -> (update_loc c None 1 false 0; store_parse comment c)
              | 14 -> store_parse comment c
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_comment_rec c lexbuf __ocaml_lex_state)
            and string c lexbuf =
              (lexbuf.Lexing.lex_mem <- Array.make 2 (-1);
               __ocaml_lex_string_rec c lexbuf 160)
            and __ocaml_lex_string_rec c lexbuf __ocaml_lex_state =
              match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state
                      lexbuf
              with
              | 0 -> set_start_p c
              | 1 ->
                  let space =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0)
                      lexbuf.Lexing.lex_curr_pos
                  in
                    (update_loc c None 1 false (String.length space);
                     store_parse string c)
              | 2 -> store_parse string c
              | 3 -> store_parse string c
              | 4 -> store_parse string c
              | 5 ->
                  let x =
                    Lexing.sub_lexeme_char lexbuf
                      (lexbuf.Lexing.lex_start_pos + 1)
                  in
                    if is_in_comment c
                    then store_parse string c
                    else
                      (warn (Illegal_escape (String.make 1 x))
                         (Loc.of_lexbuf lexbuf);
                       store_parse string c)
              | 6 -> (update_loc c None 1 false 0; store_parse string c)
              | 7 -> err Unterminated_string (loc c)
              | 8 -> store_parse string c
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_string_rec c lexbuf __ocaml_lex_state)
            and symbolchar_star beginning c lexbuf =
              __ocaml_lex_symbolchar_star_rec beginning c lexbuf 177
            and
              __ocaml_lex_symbolchar_star_rec beginning c lexbuf
                                              __ocaml_lex_state =
              match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf
              with
              | 0 ->
                  let tok =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in
                    (move_start_p (- (String.length beginning)) c;
                     SYMBOL (beginning ^ tok))
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_symbolchar_star_rec beginning c lexbuf
                     __ocaml_lex_state)
            and maybe_quotation_at c lexbuf =
              __ocaml_lex_maybe_quotation_at_rec c lexbuf 178
            and
              __ocaml_lex_maybe_quotation_at_rec c lexbuf __ocaml_lex_state =
              match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf
              with
              | 0 ->
                  let loc =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    mk_quotation quotation c "" loc (1 + (String.length loc))
              | 1 ->
                  let tok =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in SYMBOL ("<@" ^ tok)
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_maybe_quotation_at_rec c lexbuf
                     __ocaml_lex_state)
            and maybe_quotation_colon c lexbuf =
              (lexbuf.Lexing.lex_mem <- Array.make 2 (-1);
               __ocaml_lex_maybe_quotation_colon_rec c lexbuf 182)
            and
              __ocaml_lex_maybe_quotation_colon_rec c lexbuf
                                                    __ocaml_lex_state =
              match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state
                      lexbuf
              with
              | 0 ->
                  let name =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    mk_quotation quotation c name ""
                      (1 + (String.length name))
              | 1 ->
                  let name =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_mem.(0)
                  and loc =
                    Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_mem.(0) + 1)
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    mk_quotation quotation c name loc
                      ((2 + (String.length loc)) + (String.length name))
              | 2 ->
                  let tok =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      lexbuf.Lexing.lex_curr_pos
                  in SYMBOL ("<:" ^ tok)
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_maybe_quotation_colon_rec c lexbuf
                     __ocaml_lex_state)
            and quotation c lexbuf = __ocaml_lex_quotation_rec c lexbuf 189
            and __ocaml_lex_quotation_rec c lexbuf __ocaml_lex_state =
              match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf
              with
              | 0 -> (store c; with_curr_loc quotation c; parse quotation c)
              | 1 -> store c
              | 2 -> err Unterminated_quotation (loc c)
              | 3 -> (update_loc c None 1 false 0; store_parse quotation c)
              | 4 -> store_parse quotation c
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_quotation_rec c lexbuf __ocaml_lex_state)
            and dollar c lexbuf = __ocaml_lex_dollar_rec c lexbuf 202
            and __ocaml_lex_dollar_rec c lexbuf __ocaml_lex_state =
              match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf
              with
              | 0 -> (set_start_p c; ANTIQUOT ("", ""))
              | 1 ->
                  let name =
                    Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos
                      (lexbuf.Lexing.lex_curr_pos + (-1))
                  in
                    with_curr_loc (antiquot name)
                      (shift (1 + (String.length name)) c)
              | 2 -> store_parse (antiquot "") c
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_dollar_rec c lexbuf __ocaml_lex_state)
            and antiquot name c lexbuf =
              __ocaml_lex_antiquot_rec name c lexbuf 211
            and __ocaml_lex_antiquot_rec name c lexbuf __ocaml_lex_state =
              match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf
              with
              | 0 -> (set_start_p c; ANTIQUOT (name, (buff_contents c)))
              | 1 -> err Unterminated_antiquot (loc c)
              | 2 ->
                  (update_loc c None 1 false 0;
                   store_parse (antiquot name) c)
              | 3 ->
                  (store c;
                   with_curr_loc quotation c;
                   parse (antiquot name) c)
              | 4 -> store_parse (antiquot name) c
              | __ocaml_lex_state ->
                  (lexbuf.Lexing.refill_buff lexbuf;
                   __ocaml_lex_antiquot_rec name c lexbuf __ocaml_lex_state)
              
            let lexing_store s buff max =
              let rec self n s =
                if n >= max
                then n
                else
                  (match Stream.peek s with
                   | Some x -> (Stream.junk s; buff.[n] <- x; succ n)
                   | _ -> n)
              in self 0 s
              
            let from_context c =
              let next _ =
                let tok = with_curr_loc token c in
                let loc = Loc.of_lexbuf c.lexbuf in Some (tok, loc)
              in Stream.from next
              
            let from_lexbuf ?(quotations = true) lb =
              let c =
                {
                  (default_context lb)
                  with
                  loc = Loc.of_lexbuf lb;
                  antiquots = !Camlp4_config.antiquotations;
                  quotations = quotations;
                }
              in from_context c
              
            let setup_loc lb loc =
              let start_pos = Loc.start_pos loc
              in
                (lb.lex_abs_pos <- start_pos.pos_cnum;
                 lb.lex_curr_p <- start_pos)
              
            let from_string ?quotations loc str =
              let lb = Lexing.from_string str
              in (setup_loc lb loc; from_lexbuf ?quotations lb)
              
            let from_stream ?quotations loc strm =
              let lb = Lexing.from_function (lexing_store strm)
              in (setup_loc lb loc; from_lexbuf ?quotations lb)
              
            let mk () loc strm =
              from_stream ~quotations: !Camlp4_config.quotations loc strm
              
          end
          
      end
      
    module Camlp4Ast =
      struct
        module Make (Loc : Sig.Loc) : Sig.Camlp4Ast with module Loc = Loc =
          struct
            module Loc = Loc
              
            module Ast =
              struct
                include Sig.MakeCamlp4Ast(Loc)
                  
                let safe_string_escaped s =
                  if
                    ((String.length s) > 2) &&
                      ((s.[0] = '\\') && (s.[1] = '$'))
                  then s
                  else String.escaped s
                  
              end
              
            include Ast
              
            external loc_of_ctyp : ctyp -> Loc.t = "%field0"
              
            external loc_of_patt : patt -> Loc.t = "%field0"
              
            external loc_of_expr : expr -> Loc.t = "%field0"
              
            external loc_of_module_type : module_type -> Loc.t = "%field0"
              
            external loc_of_module_expr : module_expr -> Loc.t = "%field0"
              
            external loc_of_sig_item : sig_item -> Loc.t = "%field0"
              
            external loc_of_str_item : str_item -> Loc.t = "%field0"
              
            external loc_of_class_type : class_type -> Loc.t = "%field0"
              
            external loc_of_class_sig_item : class_sig_item -> Loc.t =
              "%field0"
              
            external loc_of_class_expr : class_expr -> Loc.t = "%field0"
              
            external loc_of_class_str_item : class_str_item -> Loc.t =
              "%field0"
              
            external loc_of_with_constr : with_constr -> Loc.t = "%field0"
              
            external loc_of_binding : binding -> Loc.t = "%field0"
              
            external loc_of_rec_binding : rec_binding -> Loc.t = "%field0"
              
            external loc_of_module_binding : module_binding -> Loc.t =
              "%field0"
              
            external loc_of_match_case : match_case -> Loc.t = "%field0"
              
            external loc_of_ident : ident -> Loc.t = "%field0"
              
            let ghost = Loc.ghost
              
            let rec is_module_longident =
              function
              | Ast.IdAcc (_, _, i) -> is_module_longident i
              | Ast.IdApp (_, i1, i2) ->
                  (is_module_longident i1) && (is_module_longident i2)
              | Ast.IdUid (_, _) -> true
              | _ -> false
              
            let ident_of_expr =
              let error () =
                invalid_arg
                  "ident_of_expr: this expression is not an identifier" in
              let rec self =
                function
                | Ast.ExApp (_loc, e1, e2) ->
                    Ast.IdApp (_loc, (self e1), (self e2))
                | Ast.ExAcc (_loc, e1, e2) ->
                    Ast.IdAcc (_loc, (self e1), (self e2))
                | Ast.ExId (_, (Ast.IdLid (_, _))) -> error ()
                | Ast.ExId (_, i) ->
                    if is_module_longident i then i else error ()
                | _ -> error ()
              in
                function
                | Ast.ExId (_, i) -> i
                | Ast.ExApp (_, _, _) -> error ()
                | t -> self t
              
            let ident_of_ctyp =
              let error () =
                invalid_arg "ident_of_ctyp: this type is not an identifier" in
              let rec self =
                function
                | Ast.TyApp (_loc, t1, t2) ->
                    Ast.IdApp (_loc, (self t1), (self t2))
                | Ast.TyId (_, (Ast.IdLid (_, _))) -> error ()
                | Ast.TyId (_, i) ->
                    if is_module_longident i then i else error ()
                | _ -> error ()
              in function | Ast.TyId (_, i) -> i | t -> self t
              
            let ident_of_patt =
              let error () =
                invalid_arg
                  "ident_of_patt: this pattern is not an identifier" in
              let rec self =
                function
                | Ast.PaApp (_loc, p1, p2) ->
                    Ast.IdApp (_loc, (self p1), (self p2))
                | Ast.PaId (_, (Ast.IdLid (_, _))) -> error ()
                | Ast.PaId (_, i) ->
                    if is_module_longident i then i else error ()
                | _ -> error ()
              in function | Ast.PaId (_, i) -> i | p -> self p
              
            let rec is_irrefut_patt =
              function
              | Ast.PaId (_, (Ast.IdLid (_, _))) -> true
              | Ast.PaId (_, (Ast.IdUid (_, "()"))) -> true
              | Ast.PaAny _ -> true
              | Ast.PaNil _ -> true
              | Ast.PaAli (_, x, y) ->
                  (is_irrefut_patt x) && (is_irrefut_patt y)
              | Ast.PaRec (_, p) -> is_irrefut_patt p
              | Ast.PaEq (_, _, p) -> is_irrefut_patt p
              | Ast.PaSem (_, p1, p2) ->
                  (is_irrefut_patt p1) && (is_irrefut_patt p2)
              | Ast.PaCom (_, p1, p2) ->
                  (is_irrefut_patt p1) && (is_irrefut_patt p2)
              | Ast.PaOrp (_, p1, p2) ->
                  (is_irrefut_patt p1) && (is_irrefut_patt p2)
              | Ast.PaApp (_, p1, p2) ->
                  (is_irrefut_patt p1) && (is_irrefut_patt p2)
              | Ast.PaTyc (_, p, _) -> is_irrefut_patt p
              | Ast.PaTup (_, pl) -> is_irrefut_patt pl
              | Ast.PaOlb (_, _, (Ast.PaNil _)) -> true
              | Ast.PaOlb (_, _, p) -> is_irrefut_patt p
              | Ast.PaOlbi (_, _, p, _) -> is_irrefut_patt p
              | Ast.PaLab (_, _, (Ast.PaNil _)) -> true
              | Ast.PaLab (_, _, p) -> is_irrefut_patt p
              | Ast.PaLaz (_, p) -> is_irrefut_patt p
              | Ast.PaAtt (_loc, _s, _str, p) -> is_irrefut_patt p
              | Ast.PaId (_, _) -> false
              | Ast.PaMod (_, _) -> true
              | Ast.PaExc (_loc, p) -> is_irrefut_patt p
              | Ast.PaVrn (_, _) | Ast.PaStr (_, _) | Ast.PaRng (_, _, _) |
                  Ast.PaFlo (_, _) | Ast.PaNativeInt (_, _) |
                  Ast.PaInt64 (_, _) | Ast.PaInt32 (_, _) | Ast.PaInt (_, _)
                  | Ast.PaChr (_, _) | Ast.PaTyp (_, _) | Ast.PaArr (_, _) |
                  Ast.PaAnt (_, _) -> false
              
            let rec is_constructor =
              function
              | Ast.IdAcc (_, _, i) -> is_constructor i
              | Ast.IdUid (_, _) -> true
              | Ast.IdLid (_, _) | Ast.IdApp (_, _, _) -> false
              | Ast.IdAnt (_, _) -> assert false
              
            let is_patt_constructor =
              function
              | Ast.PaId (_, i) -> is_constructor i
              | Ast.PaVrn (_, _) -> true
              | _ -> false
              
            let rec is_expr_constructor =
              function
              | Ast.ExId (_, i) -> is_constructor i
              | Ast.ExAcc (_, e1, e2) ->
                  (is_expr_constructor e1) && (is_expr_constructor e2)
              | Ast.ExVrn (_, _) -> true
              | _ -> false
              
            let rec tyOr_of_list =
              function
              | [] -> Ast.TyNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_ctyp t
                  in Ast.TyOr (_loc, t, (tyOr_of_list ts))
              
            let rec tyAnd_of_list =
              function
              | [] -> Ast.TyNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_ctyp t
                  in Ast.TyAnd (_loc, t, (tyAnd_of_list ts))
              
            let rec tySem_of_list =
              function
              | [] -> Ast.TyNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_ctyp t
                  in Ast.TySem (_loc, t, (tySem_of_list ts))
              
            let rec tyCom_of_list =
              function
              | [] -> Ast.TyNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_ctyp t
                  in Ast.TyCom (_loc, t, (tyCom_of_list ts))
              
            let rec tyAmp_of_list =
              function
              | [] -> Ast.TyNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_ctyp t
                  in Ast.TyAmp (_loc, t, (tyAmp_of_list ts))
              
            let rec tySta_of_list =
              function
              | [] -> Ast.TyNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_ctyp t
                  in Ast.TySta (_loc, t, (tySta_of_list ts))
              
            let rec stSem_of_list =
              function
              | [] -> Ast.StNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_str_item t
                  in Ast.StSem (_loc, t, (stSem_of_list ts))
              
            let rec sgSem_of_list =
              function
              | [] -> Ast.SgNil ghost
              | [ t ] -> t
              | t :: ts ->
                  let _loc = loc_of_sig_item t
                  in Ast.SgSem (_loc, t, (sgSem_of_list ts))
              
            let rec biAnd_of_list =
              function
              | [] -> Ast.BiNil ghost
              | [ b ] -> b
              | b :: bs ->
                  let _loc = loc_of_binding b
                  in Ast.BiAnd (_loc, b, (biAnd_of_list bs))
              
            let rec rbSem_of_list =
              function
              | [] -> Ast.RbNil ghost
              | [ b ] -> b
              | b :: bs ->
                  let _loc = loc_of_rec_binding b
                  in Ast.RbSem (_loc, b, (rbSem_of_list bs))
              
            let rec wcAnd_of_list =
              function
              | [] -> Ast.WcNil ghost
              | [ w ] -> w
              | w :: ws ->
                  let _loc = loc_of_with_constr w
                  in Ast.WcAnd (_loc, w, (wcAnd_of_list ws))
              
            let rec idAcc_of_list =
              function
              | [] -> assert false
              | [ i ] -> i
              | i :: is ->
                  let _loc = loc_of_ident i
                  in Ast.IdAcc (_loc, i, (idAcc_of_list is))
              
            let rec idApp_of_list =
              function
              | [] -> assert false
              | [ i ] -> i
              | i :: is ->
                  let _loc = loc_of_ident i
                  in Ast.IdApp (_loc, i, (idApp_of_list is))
              
            let rec mcOr_of_list =
              function
              | [] -> Ast.McNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_match_case x
                  in Ast.McOr (_loc, x, (mcOr_of_list xs))
              
            let rec mbAnd_of_list =
              function
              | [] -> Ast.MbNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_module_binding x
                  in Ast.MbAnd (_loc, x, (mbAnd_of_list xs))
              
            let rec meApp_of_list =
              function
              | [] -> assert false
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_module_expr x
                  in Ast.MeApp (_loc, x, (meApp_of_list xs))
              
            let rec ceAnd_of_list =
              function
              | [] -> Ast.CeNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_class_expr x
                  in Ast.CeAnd (_loc, x, (ceAnd_of_list xs))
              
            let rec ctAnd_of_list =
              function
              | [] -> Ast.CtNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_class_type x
                  in Ast.CtAnd (_loc, x, (ctAnd_of_list xs))
              
            let rec cgSem_of_list =
              function
              | [] -> Ast.CgNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_class_sig_item x
                  in Ast.CgSem (_loc, x, (cgSem_of_list xs))
              
            let rec crSem_of_list =
              function
              | [] -> Ast.CrNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_class_str_item x
                  in Ast.CrSem (_loc, x, (crSem_of_list xs))
              
            let rec paSem_of_list =
              function
              | [] -> Ast.PaNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_patt x
                  in Ast.PaSem (_loc, x, (paSem_of_list xs))
              
            let rec paCom_of_list =
              function
              | [] -> Ast.PaNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_patt x
                  in Ast.PaCom (_loc, x, (paCom_of_list xs))
              
            let rec exSem_of_list =
              function
              | [] -> Ast.ExNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_expr x
                  in Ast.ExSem (_loc, x, (exSem_of_list xs))
              
            let rec exCom_of_list =
              function
              | [] -> Ast.ExNil ghost
              | [ x ] -> x
              | x :: xs ->
                  let _loc = loc_of_expr x
                  in Ast.ExCom (_loc, x, (exCom_of_list xs))
              
            let ty_of_stl =
              function
              | (_loc, s, []) -> Ast.TyId (_loc, (Ast.IdUid (_loc, s)))
              | (_loc, s, tl) ->
                  Ast.TyOf (_loc, (Ast.TyId (_loc, (Ast.IdUid (_loc, s)))),
                    (tyAnd_of_list tl))
              
            let ty_of_sbt =
              function
              | (_loc, s, true, t) ->
                  Ast.TyCol (_loc, (Ast.TyId (_loc, (Ast.IdLid (_loc, s)))),
                    (Ast.TyMut (_loc, t)))
              | (_loc, s, false, t) ->
                  Ast.TyCol (_loc, (Ast.TyId (_loc, (Ast.IdLid (_loc, s)))),
                    t)
              
            let bi_of_pe (p, e) =
              let _loc = loc_of_patt p in Ast.BiEq (_loc, p, e)
              
            let sum_type_of_list l = tyOr_of_list (List.map ty_of_stl l)
              
            let record_type_of_list l = tySem_of_list (List.map ty_of_sbt l)
              
            let binding_of_pel l = biAnd_of_list (List.map bi_of_pe l)
              
            let rec pel_of_binding =
              function
              | Ast.BiAnd (_, b1, b2) ->
                  (pel_of_binding b1) @ (pel_of_binding b2)
              | Ast.BiEq (_, p, e) -> [ (p, e) ]
              | _ -> assert false
              
            let rec list_of_binding x acc =
              match x with
              | Ast.BiAnd (_, b1, b2) ->
                  list_of_binding b1 (list_of_binding b2 acc)
              | t -> t :: acc
              
            let rec list_of_rec_binding x acc =
              match x with
              | Ast.RbSem (_, b1, b2) ->
                  list_of_rec_binding b1 (list_of_rec_binding b2 acc)
              | t -> t :: acc
              
            let rec list_of_with_constr x acc =
              match x with
              | Ast.WcAnd (_, w1, w2) ->
                  list_of_with_constr w1 (list_of_with_constr w2 acc)
              | t -> t :: acc
              
            let rec list_of_ctyp x acc =
              match x with
              | Ast.TyNil _ -> acc
              | Ast.TyAmp (_, x, y) | Ast.TyCom (_, x, y) |
                  Ast.TySta (_, x, y) | Ast.TySem (_, x, y) |
                  Ast.TyAnd (_, x, y) | Ast.TyOr (_, x, y) ->
                  list_of_ctyp x (list_of_ctyp y acc)
              | x -> x :: acc
              
            let rec list_of_patt x acc =
              match x with
              | Ast.PaNil _ -> acc
              | Ast.PaCom (_, x, y) | Ast.PaSem (_, x, y) ->
                  list_of_patt x (list_of_patt y acc)
              | x -> x :: acc
              
            let rec list_of_expr x acc =
              match x with
              | Ast.ExNil _ -> acc
              | Ast.ExCom (_, x, y) | Ast.ExSem (_, x, y) ->
                  list_of_expr x (list_of_expr y acc)
              | x -> x :: acc
              
            let rec list_of_str_item x acc =
              match x with
              | Ast.StNil _ -> acc
              | Ast.StSem (_, x, y) ->
                  list_of_str_item x (list_of_str_item y acc)
              | x -> x :: acc
              
            let rec list_of_sig_item x acc =
              match x with
              | Ast.SgNil _ -> acc
              | Ast.SgSem (_, x, y) ->
                  list_of_sig_item x (list_of_sig_item y acc)
              | x -> x :: acc
              
            let rec list_of_class_sig_item x acc =
              match x with
              | Ast.CgNil _ -> acc
              | Ast.CgSem (_, x, y) ->
                  list_of_class_sig_item x (list_of_class_sig_item y acc)
              | x -> x :: acc
              
            let rec list_of_class_str_item x acc =
              match x with
              | Ast.CrNil _ -> acc
              | Ast.CrSem (_, x, y) ->
                  list_of_class_str_item x (list_of_class_str_item y acc)
              | x -> x :: acc
              
            let rec list_of_class_type x acc =
              match x with
              | Ast.CtAnd (_, x, y) ->
                  list_of_class_type x (list_of_class_type y acc)
              | x -> x :: acc
              
            let rec list_of_class_expr x acc =
              match x with
              | Ast.CeAnd (_, x, y) ->
                  list_of_class_expr x (list_of_class_expr y acc)
              | x -> x :: acc
              
            let rec list_of_module_expr x acc =
              match x with
              | Ast.MeApp (_, x, y) ->
                  list_of_module_expr x (list_of_module_expr y acc)
              | x -> x :: acc
              
            let rec list_of_match_case x acc =
              match x with
              | Ast.McNil _ -> acc
              | Ast.McOr (_, x, y) ->
                  list_of_match_case x (list_of_match_case y acc)
              | x -> x :: acc
              
            let rec list_of_ident x acc =
              match x with
              | Ast.IdAcc (_, x, y) | Ast.IdApp (_, x, y) ->
                  list_of_ident x (list_of_ident y acc)
              | x -> x :: acc
              
            let rec list_of_module_binding x acc =
              match x with
              | Ast.MbAnd (_, x, y) ->
                  list_of_module_binding x (list_of_module_binding y acc)
              | x -> x :: acc
              
            module Meta =
              struct
                module type META_LOC =
                  sig
                    val meta_loc_patt : Loc.t -> Loc.t -> Ast.patt
                      
                    val meta_loc_expr : Loc.t -> Loc.t -> Ast.expr
                      
                  end
                  
                module MetaLoc =
                  struct
                    let meta_loc_patt _loc location =
                      let (a, b, c, d, e, f, g, h) = Loc.to_tuple location
                      in
                        Ast.PaApp (_loc,
                          (Ast.PaId (_loc,
                             (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Loc")),
                                (Ast.IdLid (_loc, "of_tuple")))))),
                          (Ast.PaTup (_loc,
                             (Ast.PaCom (_loc,
                                (Ast.PaStr (_loc,
                                   (Ast.safe_string_escaped a))),
                                (Ast.PaCom (_loc,
                                   (Ast.PaCom (_loc,
                                      (Ast.PaCom (_loc,
                                         (Ast.PaCom (_loc,
                                            (Ast.PaCom (_loc,
                                               (Ast.PaCom (_loc,
                                                  (Ast.PaInt (_loc,
                                                     (string_of_int b))),
                                                  (Ast.PaInt (_loc,
                                                     (string_of_int c))))),
                                               (Ast.PaInt (_loc,
                                                  (string_of_int d))))),
                                            (Ast.PaInt (_loc,
                                               (string_of_int e))))),
                                         (Ast.PaInt (_loc, (string_of_int f))))),
                                      (Ast.PaInt (_loc, (string_of_int g))))),
                                   (if h
                                    then
                                      Ast.PaId (_loc,
                                        (Ast.IdUid (_loc, "True")))
                                    else
                                      Ast.PaId (_loc,
                                        (Ast.IdUid (_loc, "False")))))))))))
                      
                    let meta_loc_expr _loc location =
                      let (a, b, c, d, e, f, g, h) = Loc.to_tuple location
                      in
                        Ast.ExApp (_loc,
                          (Ast.ExId (_loc,
                             (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Loc")),
                                (Ast.IdLid (_loc, "of_tuple")))))),
                          (Ast.ExTup (_loc,
                             (Ast.ExCom (_loc,
                                (Ast.ExStr (_loc,
                                   (Ast.safe_string_escaped a))),
                                (Ast.ExCom (_loc,
                                   (Ast.ExCom (_loc,
                                      (Ast.ExCom (_loc,
                                         (Ast.ExCom (_loc,
                                            (Ast.ExCom (_loc,
                                               (Ast.ExCom (_loc,
                                                  (Ast.ExInt (_loc,
                                                     (string_of_int b))),
                                                  (Ast.ExInt (_loc,
                                                     (string_of_int c))))),
                                               (Ast.ExInt (_loc,
                                                  (string_of_int d))))),
                                            (Ast.ExInt (_loc,
                                               (string_of_int e))))),
                                         (Ast.ExInt (_loc, (string_of_int f))))),
                                      (Ast.ExInt (_loc, (string_of_int g))))),
                                   (if h
                                    then
                                      Ast.ExId (_loc,
                                        (Ast.IdUid (_loc, "True")))
                                    else
                                      Ast.ExId (_loc,
                                        (Ast.IdUid (_loc, "False")))))))))))
                      
                  end
                  
                module MetaGhostLoc =
                  struct
                    let meta_loc_patt _loc _ =
                      Ast.PaId (_loc,
                        (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Loc")),
                           (Ast.IdLid (_loc, "ghost")))))
                      
                    let meta_loc_expr _loc _ =
                      Ast.ExId (_loc,
                        (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Loc")),
                           (Ast.IdLid (_loc, "ghost")))))
                      
                  end
                  
                module MetaLocVar =
                  struct
                    let meta_loc_patt _loc _ =
                      Ast.PaId (_loc, (Ast.IdLid (_loc, !Loc.name)))
                      
                    let meta_loc_expr _loc _ =
                      Ast.ExId (_loc, (Ast.IdLid (_loc, !Loc.name)))
                      
                  end
                  
                module Make (MetaLoc : META_LOC) =
                  struct
                    open MetaLoc
                      
                    let meta_loc = meta_loc_expr
                      
                    module Expr =
                      struct
                        let meta_string _loc s =
                          Ast.ExStr (_loc, (safe_string_escaped s))
                          
                        let meta_int _loc s = Ast.ExInt (_loc, s)
                          
                        let meta_float _loc s = Ast.ExFlo (_loc, s)
                          
                        let meta_char _loc s =
                          Ast.ExChr (_loc, (String.escaped s))
                          
                        let meta_bool _loc =
                          function
                          | false ->
                              Ast.ExId (_loc, (Ast.IdUid (_loc, "False")))
                          | true ->
                              Ast.ExId (_loc, (Ast.IdUid (_loc, "True")))
                          
                        let rec meta_list mf_a _loc =
                          function
                          | [] -> Ast.ExId (_loc, (Ast.IdUid (_loc, "[]")))
                          | x :: xs ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc, (Ast.IdUid (_loc, "::")))),
                                   (mf_a _loc x))),
                                (meta_list mf_a _loc xs))
                          
                        let rec meta_binding _loc =
                          function
                          | Ast.BiAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.BiEq (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "BiEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.BiAnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "BiAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_binding _loc x1))),
                                (meta_binding _loc x2))
                          | Ast.BiNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "BiNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_expr _loc =
                          function
                          | Ast.CeAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.CeAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CeAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_class_expr _loc x3))
                          | Ast.CeEq (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_class_expr _loc x2))
                          | Ast.CeAnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_class_expr _loc x2))
                          | Ast.CeTyc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CeStr (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeStr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_class_str_item _loc x2))
                          | Ast.CeLet (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CeLet")))))),
                                         (meta_loc _loc x0))),
                                      (meta_rec_flag _loc x1))),
                                   (meta_binding _loc x2))),
                                (meta_class_expr _loc x3))
                          | Ast.CeFun (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeFun")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_class_expr _loc x2))
                          | Ast.CeCon (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CeCon")))))),
                                         (meta_loc _loc x0))),
                                      (meta_virtual_flag _loc x1))),
                                   (meta_ident _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CeApp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.CeNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CeNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_sig_item _loc =
                          function
                          | Ast.CgAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.CgVir (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CgVir")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_private_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CgVal (x0, x1, x2, x3, x4) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExApp (_loc,
                                            (Ast.ExId (_loc,
                                               (Ast.IdAcc (_loc,
                                                  (Ast.IdUid (_loc, "Ast")),
                                                  (Ast.IdUid (_loc, "CgVal")))))),
                                            (meta_loc _loc x0))),
                                         (meta_string _loc x1))),
                                      (meta_mutable_flag _loc x2))),
                                   (meta_virtual_flag _loc x3))),
                                (meta_ctyp _loc x4))
                          | Ast.CgMth (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CgMth")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_private_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CgInh (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "CgInh")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.CgSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CgSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_sig_item _loc x1))),
                                (meta_class_sig_item _loc x2))
                          | Ast.CgCtr (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CgCtr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.CgNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CgNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_str_item _loc =
                          function
                          | Ast.CrAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.CrVvr (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CrVvr")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_mutable_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CrVir (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CrVir")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_private_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CrVal (x0, x1, x2, x3, x4) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExApp (_loc,
                                            (Ast.ExId (_loc,
                                               (Ast.IdAcc (_loc,
                                                  (Ast.IdUid (_loc, "Ast")),
                                                  (Ast.IdUid (_loc, "CrVal")))))),
                                            (meta_loc _loc x0))),
                                         (meta_string _loc x1))),
                                      (meta_override_flag _loc x2))),
                                   (meta_mutable_flag _loc x3))),
                                (meta_expr _loc x4))
                          | Ast.CrMth (x0, x1, x2, x3, x4, x5) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExApp (_loc,
                                            (Ast.ExApp (_loc,
                                               (Ast.ExId (_loc,
                                                  (Ast.IdAcc (_loc,
                                                     (Ast.IdUid (_loc, "Ast")),
                                                     (Ast.IdUid (_loc,
                                                        "CrMth")))))),
                                               (meta_loc _loc x0))),
                                            (meta_string _loc x1))),
                                         (meta_override_flag _loc x2))),
                                      (meta_private_flag _loc x3))),
                                   (meta_expr _loc x4))),
                                (meta_ctyp _loc x5))
                          | Ast.CrIni (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "CrIni")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.CrInh (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CrInh")))))),
                                         (meta_loc _loc x0))),
                                      (meta_override_flag _loc x1))),
                                   (meta_class_expr _loc x2))),
                                (meta_string _loc x3))
                          | Ast.CrCtr (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CrCtr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.CrSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CrSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_str_item _loc x1))),
                                (meta_class_str_item _loc x2))
                          | Ast.CrNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CrNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_type _loc =
                          function
                          | Ast.CtAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.CtAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CtAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_class_type _loc x3))
                          | Ast.CtEq (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_type _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtCol (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtCol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_type _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtAnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_type _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtSig (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtSig")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_class_sig_item _loc x2))
                          | Ast.CtFun (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtFun")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtCon (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CtCon")))))),
                                         (meta_loc _loc x0))),
                                      (meta_virtual_flag _loc x1))),
                                   (meta_ident _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CtNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CtNil")))))),
                                (meta_loc _loc x0))
                        and meta_ctyp _loc =
                          function
                          | Ast.TyAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.TyAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "TyAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.TyOpn x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyOpn")))))),
                                (meta_loc _loc x0))
                          | Ast.TyPkg (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyPkg")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_type _loc x1))
                          | Ast.TyOfAmp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOfAmp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyAmp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyAmp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyVrnInfSup (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyVrnInfSup")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyVrnInf (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrnInf")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyVrnSup (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrnSup")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyVrnEq (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrnEq")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TySta (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TySta")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyTup (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyTup")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyMut (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyMut")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyPrv (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyPrv")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyOr (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyAnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyOf (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOf")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TySum (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TySum")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyCom (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyCom")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TySem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TySem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyCol (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyCol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyRec (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyRec")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyVrn (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrn")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyAnM x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyAnM")))))),
                                (meta_loc _loc x0))
                          | Ast.TyAnP x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyAnP")))))),
                                (meta_loc _loc x0))
                          | Ast.TyQuM (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyQuM")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyQuP (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyQuP")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyQuo (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyQuo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyTypePol (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyTypePol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyPol (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyPol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyOlb (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOlb")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyObj (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyObj")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_row_var_flag _loc x2))
                          | Ast.TyExt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "TyExt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_ident _loc x1))),
                                   (meta_list meta_ctyp _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.TyDcl (x0, x1, x2, x3, x4) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExApp (_loc,
                                            (Ast.ExId (_loc,
                                               (Ast.IdAcc (_loc,
                                                  (Ast.IdUid (_loc, "Ast")),
                                                  (Ast.IdUid (_loc, "TyDcl")))))),
                                            (meta_loc _loc x0))),
                                         (meta_string _loc x1))),
                                      (meta_list meta_ctyp _loc x2))),
                                   (meta_ctyp _loc x3))),
                                (meta_list
                                   (fun _loc (x1, x2) ->
                                      Ast.ExTup (_loc,
                                        (Ast.ExCom (_loc,
                                           (meta_ctyp _loc x1),
                                           (meta_ctyp _loc x2)))))
                                   _loc x4))
                          | Ast.TyMan (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyMan")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyId (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.TyLab (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyLab")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyCls (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyCls")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.TyArr (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyArr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyApp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyAny x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyAny")))))),
                                (meta_loc _loc x0))
                          | Ast.TyAli (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyAli")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyNil")))))),
                                (meta_loc _loc x0))
                        and meta_direction_flag _loc =
                          function
                          | Ast.DiAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.DiDownto ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "DiDownto")))))
                          | Ast.DiTo ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "DiTo")))))
                        and meta_expr _loc =
                          function
                          | Ast.ExAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExPkg (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExPkg")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_expr _loc x1))
                          | Ast.ExFUN (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExFUN")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExOpI (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExOpI")))))),
                                         (meta_loc _loc x0))),
                                      (meta_ident _loc x1))),
                                   (meta_override_flag _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExWhi (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExWhi")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExVrn (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExVrn")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExTyc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.ExCom (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExCom")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExTup (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExTup")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExTry (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExTry")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_match_case _loc x2))
                          | Ast.ExStr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExStr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExSte (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExSte")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExSnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExSnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_string _loc x2))
                          | Ast.ExSeq (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExSeq")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExRec (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExRec")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_binding _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExOvr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExOvr")))))),
                                   (meta_loc _loc x0))),
                                (meta_rec_binding _loc x1))
                          | Ast.ExOlb (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExOlb")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExObj (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExObj")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_class_str_item _loc x2))
                          | Ast.ExNew (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExNew")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.ExMat (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExMat")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_match_case _loc x2))
                          | Ast.ExLmd (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExLmd")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_expr _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExLet (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExLet")))))),
                                         (meta_loc _loc x0))),
                                      (meta_rec_flag _loc x1))),
                                   (meta_binding _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExLaz (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExLaz")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExLab (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExLab")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExNativeInt (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExNativeInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExInt64 (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExInt64")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExInt32 (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExInt32")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExInt (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExIfe (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExIfe")))))),
                                         (meta_loc _loc x0))),
                                      (meta_expr _loc x1))),
                                   (meta_expr _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExFun (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExFun")))))),
                                   (meta_loc _loc x0))),
                                (meta_match_case _loc x1))
                          | Ast.ExFor (x0, x1, x2, x3, x4, x5) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExApp (_loc,
                                            (Ast.ExApp (_loc,
                                               (Ast.ExId (_loc,
                                                  (Ast.IdAcc (_loc,
                                                     (Ast.IdUid (_loc, "Ast")),
                                                     (Ast.IdUid (_loc,
                                                        "ExFor")))))),
                                               (meta_loc _loc x0))),
                                            (meta_patt _loc x1))),
                                         (meta_expr _loc x2))),
                                      (meta_expr _loc x3))),
                                   (meta_direction_flag _loc x4))),
                                (meta_expr _loc x5))
                          | Ast.ExFlo (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExFlo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExCoe (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExCoe")))))),
                                         (meta_loc _loc x0))),
                                      (meta_expr _loc x1))),
                                   (meta_ctyp _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.ExChr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExChr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExAss (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExAss")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExAsr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExAsr")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExAsf x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "ExAsf")))))),
                                (meta_loc _loc x0))
                          | Ast.ExSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExArr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExArr")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExAre (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExAre")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExApp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.ExAcc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExAcc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExId (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.ExNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "ExNil")))))),
                                (meta_loc _loc x0))
                        and meta_ident _loc =
                          function
                          | Ast.IdAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.IdUid (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "IdUid")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.IdLid (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "IdLid")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.IdApp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "IdApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.IdAcc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "IdAcc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                        and meta_match_case _loc =
                          function
                          | Ast.McAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.McArr (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "McArr")))))),
                                         (meta_loc _loc x0))),
                                      (meta_patt _loc x1))),
                                   (meta_expr _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.McOr (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "McOr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_match_case _loc x1))),
                                (meta_match_case _loc x2))
                          | Ast.McNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "McNil")))))),
                                (meta_loc _loc x0))
                        and meta_meta_bool _loc =
                          function
                          | Ast.BAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.BFalse ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "BFalse")))))
                          | Ast.BTrue ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "BTrue")))))
                        and meta_meta_list mf_a _loc =
                          function
                          | Ast.LAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.LCons (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "LCons")))))),
                                   (mf_a _loc x0))),
                                (meta_meta_list mf_a _loc x1))
                          | Ast.LNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "LNil")))))
                        and meta_meta_option mf_a _loc =
                          function
                          | Ast.OAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.OSome x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "OSome")))))),
                                (mf_a _loc x0))
                          | Ast.ONone ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ONone")))))
                        and meta_module_binding _loc =
                          function
                          | Ast.MbAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.MbCol (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MbCol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.MbColEq (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MbColEq")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_type _loc x2))),
                                (meta_module_expr _loc x3))
                          | Ast.MbAnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MbAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_binding _loc x1))),
                                (meta_module_binding _loc x2))
                          | Ast.MbNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "MbNil")))))),
                                (meta_loc _loc x0))
                        and meta_module_expr _loc =
                          function
                          | Ast.MeAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.MeAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MeAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_module_expr _loc x3))
                          | Ast.MePkg (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MePkg")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.MeTyc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MeTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_expr _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.MeStr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MeStr")))))),
                                   (meta_loc _loc x0))),
                                (meta_str_item _loc x1))
                          | Ast.MeFun (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MeFun")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_type _loc x2))),
                                (meta_module_expr _loc x3))
                          | Ast.MeApp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MeApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_expr _loc x1))),
                                (meta_module_expr _loc x2))
                          | Ast.MeId (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MeId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.MeNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "MeNil")))))),
                                (meta_loc _loc x0))
                        and meta_module_type _loc =
                          function
                          | Ast.MtAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.MtAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MtAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_module_type _loc x3))
                          | Ast.MtOf (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtOf")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_expr _loc x1))
                          | Ast.MtWit (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MtWit")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_type _loc x1))),
                                (meta_with_constr _loc x2))
                          | Ast.MtSig (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtSig")))))),
                                   (meta_loc _loc x0))),
                                (meta_sig_item _loc x1))
                          | Ast.MtQuo (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtQuo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.MtFun (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MtFun")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_type _loc x2))),
                                (meta_module_type _loc x3))
                          | Ast.MtAlias (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtAlias")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.MtId (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.MtNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "MtNil")))))),
                                (meta_loc _loc x0))
                        and meta_mutable_flag _loc =
                          function
                          | Ast.MuAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.MuNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "MuNil")))))
                          | Ast.MuMutable ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "MuMutable")))))
                        and meta_override_flag _loc =
                          function
                          | Ast.OvAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.OvNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "OvNil")))))
                          | Ast.OvOverride ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "OvOverride")))))
                        and meta_patt _loc =
                          function
                          | Ast.PaExc (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaExc")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaMod (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaMod")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaAtt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "PaAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_patt _loc x3))
                          | Ast.PaLaz (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaLaz")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaVrn (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaVrn")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaTyp (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaTyp")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.PaTyc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.PaTup (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaTup")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaStr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaStr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaEq (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaRec (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaRec")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaRng (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaRng")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaOrp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaOrp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaOlbi (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "PaOlbi")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_patt _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.PaOlb (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaOlb")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaLab (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaLab")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaFlo (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaFlo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaNativeInt (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaNativeInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaInt64 (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaInt64")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaInt32 (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaInt32")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaInt (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaChr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaChr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaCom (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaCom")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaArr (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaArr")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaApp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaAny x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "PaAny")))))),
                                (meta_loc _loc x0))
                          | Ast.PaAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.PaAli (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaAli")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaId (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.PaNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "PaNil")))))),
                                (meta_loc _loc x0))
                        and meta_private_flag _loc =
                          function
                          | Ast.PrAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.PrNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "PrNil")))))
                          | Ast.PrPrivate ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "PrPrivate")))))
                        and meta_rec_binding _loc =
                          function
                          | Ast.RbAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.RbEq (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "RbEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.RbSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "RbSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_binding _loc x1))),
                                (meta_rec_binding _loc x2))
                          | Ast.RbNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "RbNil")))))),
                                (meta_loc _loc x0))
                        and meta_rec_flag _loc =
                          function
                          | Ast.ReAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.ReNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ReNil")))))
                          | Ast.ReNonrecursive ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ReNonrecursive")))))
                          | Ast.ReRecursive ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ReRecursive")))))
                        and meta_row_var_flag _loc =
                          function
                          | Ast.RvAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.RvNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "RvNil")))))
                          | Ast.RvRowVar ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "RvRowVar")))))
                        and meta_sig_item _loc =
                          function
                          | Ast.SgAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.SgVal (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgVal")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.SgTyp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgTyp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_flag _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.SgOpn (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgOpn")))))),
                                      (meta_loc _loc x0))),
                                   (meta_override_flag _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.SgMty (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgMty")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.SgRecMod (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgRecMod")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_binding _loc x1))
                          | Ast.SgMod (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgMod")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.SgInc (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgInc")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_type _loc x1))
                          | Ast.SgExt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "SgExt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_ctyp _loc x2))),
                                (meta_meta_list meta_string _loc x3))
                          | Ast.SgExc (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgExc")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.SgDir (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgDir")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.SgSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_sig_item _loc x1))),
                                (meta_sig_item _loc x2))
                          | Ast.SgClt (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgClt")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.SgCls (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgCls")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.SgNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "SgNil")))))),
                                (meta_loc _loc x0))
                        and meta_str_item _loc =
                          function
                          | Ast.StAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.StVal (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StVal")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_flag _loc x1))),
                                (meta_binding _loc x2))
                          | Ast.StTyp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StTyp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_flag _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.StOpn (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StOpn")))))),
                                      (meta_loc _loc x0))),
                                   (meta_override_flag _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.StMty (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StMty")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.StRecMod (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StRecMod")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_binding _loc x1))
                          | Ast.StMod (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StMod")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_expr _loc x2))
                          | Ast.StInc (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StInc")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_expr _loc x1))
                          | Ast.StExt (x0, x1, x2, x3) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExApp (_loc,
                                         (Ast.ExId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "StExt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_ctyp _loc x2))),
                                (meta_meta_list meta_string _loc x3))
                          | Ast.StExp (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StExp")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.StExc (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StExc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_meta_option meta_ident _loc x2))
                          | Ast.StDir (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StDir")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.StSem (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_str_item _loc x1))),
                                (meta_str_item _loc x2))
                          | Ast.StClt (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StClt")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.StCls (x0, x1) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StCls")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_expr _loc x1))
                          | Ast.StNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "StNil")))))),
                                (meta_loc _loc x0))
                        and meta_virtual_flag _loc =
                          function
                          | Ast.ViAnt x0 -> Ast.ExAnt (_loc, x0)
                          | Ast.ViNil ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ViNil")))))
                          | Ast.ViVirtual ->
                              Ast.ExId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ViVirtual")))))
                        and meta_with_constr _loc =
                          function
                          | Ast.WcAnt (x0, x1) -> Ast.ExAnt (x0, x1)
                          | Ast.WcAnd (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_with_constr _loc x1))),
                                (meta_with_constr _loc x2))
                          | Ast.WcMoS (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcMoS")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.WcTyS (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcTyS")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.WcMod (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcMod")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.WcTyp (x0, x1, x2) ->
                              Ast.ExApp (_loc,
                                (Ast.ExApp (_loc,
                                   (Ast.ExApp (_loc,
                                      (Ast.ExId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcTyp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.WcNil x0 ->
                              Ast.ExApp (_loc,
                                (Ast.ExId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "WcNil")))))),
                                (meta_loc _loc x0))
                          
                      end
                      
                    let meta_loc = meta_loc_patt
                      
                    module Patt =
                      struct
                        let meta_string _loc s =
                          Ast.PaStr (_loc, (safe_string_escaped s))
                          
                        let meta_int _loc s = Ast.PaInt (_loc, s)
                          
                        let meta_float _loc s = Ast.PaFlo (_loc, s)
                          
                        let meta_char _loc s =
                          Ast.PaChr (_loc, (String.escaped s))
                          
                        let meta_bool _loc =
                          function
                          | false ->
                              Ast.PaId (_loc, (Ast.IdUid (_loc, "False")))
                          | true ->
                              Ast.PaId (_loc, (Ast.IdUid (_loc, "True")))
                          
                        let rec meta_list mf_a _loc =
                          function
                          | [] -> Ast.PaId (_loc, (Ast.IdUid (_loc, "[]")))
                          | x :: xs ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc, (Ast.IdUid (_loc, "::")))),
                                   (mf_a _loc x))),
                                (meta_list mf_a _loc xs))
                          
                        let rec meta_binding _loc =
                          function
                          | Ast.BiAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.BiEq (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "BiEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.BiAnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "BiAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_binding _loc x1))),
                                (meta_binding _loc x2))
                          | Ast.BiNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "BiNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_expr _loc =
                          function
                          | Ast.CeAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.CeAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CeAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_class_expr _loc x3))
                          | Ast.CeEq (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_class_expr _loc x2))
                          | Ast.CeAnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_class_expr _loc x2))
                          | Ast.CeTyc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CeStr (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeStr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_class_str_item _loc x2))
                          | Ast.CeLet (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CeLet")))))),
                                         (meta_loc _loc x0))),
                                      (meta_rec_flag _loc x1))),
                                   (meta_binding _loc x2))),
                                (meta_class_expr _loc x3))
                          | Ast.CeFun (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeFun")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_class_expr _loc x2))
                          | Ast.CeCon (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CeCon")))))),
                                         (meta_loc _loc x0))),
                                      (meta_virtual_flag _loc x1))),
                                   (meta_ident _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CeApp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CeApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.CeNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CeNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_sig_item _loc =
                          function
                          | Ast.CgAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.CgVir (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CgVir")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_private_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CgVal (x0, x1, x2, x3, x4) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaApp (_loc,
                                            (Ast.PaId (_loc,
                                               (Ast.IdAcc (_loc,
                                                  (Ast.IdUid (_loc, "Ast")),
                                                  (Ast.IdUid (_loc, "CgVal")))))),
                                            (meta_loc _loc x0))),
                                         (meta_string _loc x1))),
                                      (meta_mutable_flag _loc x2))),
                                   (meta_virtual_flag _loc x3))),
                                (meta_ctyp _loc x4))
                          | Ast.CgMth (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CgMth")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_private_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CgInh (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "CgInh")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.CgSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CgSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_sig_item _loc x1))),
                                (meta_class_sig_item _loc x2))
                          | Ast.CgCtr (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CgCtr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.CgNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CgNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_str_item _loc =
                          function
                          | Ast.CrAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.CrVvr (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CrVvr")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_mutable_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CrVir (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CrVir")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_private_flag _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CrVal (x0, x1, x2, x3, x4) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaApp (_loc,
                                            (Ast.PaId (_loc,
                                               (Ast.IdAcc (_loc,
                                                  (Ast.IdUid (_loc, "Ast")),
                                                  (Ast.IdUid (_loc, "CrVal")))))),
                                            (meta_loc _loc x0))),
                                         (meta_string _loc x1))),
                                      (meta_override_flag _loc x2))),
                                   (meta_mutable_flag _loc x3))),
                                (meta_expr _loc x4))
                          | Ast.CrMth (x0, x1, x2, x3, x4, x5) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaApp (_loc,
                                            (Ast.PaApp (_loc,
                                               (Ast.PaId (_loc,
                                                  (Ast.IdAcc (_loc,
                                                     (Ast.IdUid (_loc, "Ast")),
                                                     (Ast.IdUid (_loc,
                                                        "CrMth")))))),
                                               (meta_loc _loc x0))),
                                            (meta_string _loc x1))),
                                         (meta_override_flag _loc x2))),
                                      (meta_private_flag _loc x3))),
                                   (meta_expr _loc x4))),
                                (meta_ctyp _loc x5))
                          | Ast.CrIni (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "CrIni")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.CrInh (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CrInh")))))),
                                         (meta_loc _loc x0))),
                                      (meta_override_flag _loc x1))),
                                   (meta_class_expr _loc x2))),
                                (meta_string _loc x3))
                          | Ast.CrCtr (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CrCtr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.CrSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CrSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_str_item _loc x1))),
                                (meta_class_str_item _loc x2))
                          | Ast.CrNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CrNil")))))),
                                (meta_loc _loc x0))
                        and meta_class_type _loc =
                          function
                          | Ast.CtAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.CtAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CtAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_class_type _loc x3))
                          | Ast.CtEq (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_type _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtCol (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtCol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_type _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtAnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_class_type _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtSig (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtSig")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_class_sig_item _loc x2))
                          | Ast.CtFun (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "CtFun")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_class_type _loc x2))
                          | Ast.CtCon (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "CtCon")))))),
                                         (meta_loc _loc x0))),
                                      (meta_virtual_flag _loc x1))),
                                   (meta_ident _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.CtNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "CtNil")))))),
                                (meta_loc _loc x0))
                        and meta_ctyp _loc =
                          function
                          | Ast.TyAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.TyAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "TyAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.TyOpn x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyOpn")))))),
                                (meta_loc _loc x0))
                          | Ast.TyPkg (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyPkg")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_type _loc x1))
                          | Ast.TyOfAmp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOfAmp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyAmp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyAmp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyVrnInfSup (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyVrnInfSup")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyVrnInf (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrnInf")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyVrnSup (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrnSup")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyVrnEq (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrnEq")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TySta (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TySta")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyTup (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyTup")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyMut (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyMut")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyPrv (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyPrv")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyOr (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyAnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyOf (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOf")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TySum (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TySum")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyCom (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyCom")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TySem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TySem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyCol (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyCol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyRec (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyRec")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.TyVrn (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyVrn")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyAnM x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyAnM")))))),
                                (meta_loc _loc x0))
                          | Ast.TyAnP x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyAnP")))))),
                                (meta_loc _loc x0))
                          | Ast.TyQuM (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyQuM")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyQuP (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyQuP")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyQuo (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyQuo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.TyTypePol (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyTypePol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyPol (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyPol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyOlb (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyOlb")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyObj (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyObj")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_row_var_flag _loc x2))
                          | Ast.TyExt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "TyExt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_ident _loc x1))),
                                   (meta_list meta_ctyp _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.TyDcl (x0, x1, x2, x3, x4) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaApp (_loc,
                                            (Ast.PaId (_loc,
                                               (Ast.IdAcc (_loc,
                                                  (Ast.IdUid (_loc, "Ast")),
                                                  (Ast.IdUid (_loc, "TyDcl")))))),
                                            (meta_loc _loc x0))),
                                         (meta_string _loc x1))),
                                      (meta_list meta_ctyp _loc x2))),
                                   (meta_ctyp _loc x3))),
                                (meta_list
                                   (fun _loc (x1, x2) ->
                                      Ast.PaTup (_loc,
                                        (Ast.PaCom (_loc,
                                           (meta_ctyp _loc x1),
                                           (meta_ctyp _loc x2)))))
                                   _loc x4))
                          | Ast.TyMan (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyMan")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyId (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.TyLab (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyLab")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyCls (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "TyCls")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.TyArr (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyArr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyApp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyAny x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyAny")))))),
                                (meta_loc _loc x0))
                          | Ast.TyAli (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "TyAli")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.TyNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "TyNil")))))),
                                (meta_loc _loc x0))
                        and meta_direction_flag _loc =
                          function
                          | Ast.DiAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.DiDownto ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "DiDownto")))))
                          | Ast.DiTo ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "DiTo")))))
                        and meta_expr _loc =
                          function
                          | Ast.ExAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExPkg (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExPkg")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_expr _loc x1))
                          | Ast.ExFUN (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExFUN")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExOpI (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExOpI")))))),
                                         (meta_loc _loc x0))),
                                      (meta_ident _loc x1))),
                                   (meta_override_flag _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExWhi (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExWhi")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExVrn (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExVrn")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExTyc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.ExCom (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExCom")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExTup (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExTup")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExTry (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExTry")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_match_case _loc x2))
                          | Ast.ExStr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExStr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExSte (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExSte")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExSnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExSnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_string _loc x2))
                          | Ast.ExSeq (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExSeq")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExRec (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExRec")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_binding _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExOvr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExOvr")))))),
                                   (meta_loc _loc x0))),
                                (meta_rec_binding _loc x1))
                          | Ast.ExOlb (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExOlb")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExObj (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExObj")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_class_str_item _loc x2))
                          | Ast.ExNew (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExNew")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.ExMat (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExMat")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_match_case _loc x2))
                          | Ast.ExLmd (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExLmd")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_expr _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExLet (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExLet")))))),
                                         (meta_loc _loc x0))),
                                      (meta_rec_flag _loc x1))),
                                   (meta_binding _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExLaz (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExLaz")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExLab (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExLab")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExNativeInt (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExNativeInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExInt64 (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExInt64")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExInt32 (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExInt32")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExInt (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExIfe (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExIfe")))))),
                                         (meta_loc _loc x0))),
                                      (meta_expr _loc x1))),
                                   (meta_expr _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.ExFun (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExFun")))))),
                                   (meta_loc _loc x0))),
                                (meta_match_case _loc x1))
                          | Ast.ExFor (x0, x1, x2, x3, x4, x5) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaApp (_loc,
                                            (Ast.PaApp (_loc,
                                               (Ast.PaId (_loc,
                                                  (Ast.IdAcc (_loc,
                                                     (Ast.IdUid (_loc, "Ast")),
                                                     (Ast.IdUid (_loc,
                                                        "ExFor")))))),
                                               (meta_loc _loc x0))),
                                            (meta_patt _loc x1))),
                                         (meta_expr _loc x2))),
                                      (meta_expr _loc x3))),
                                   (meta_direction_flag _loc x4))),
                                (meta_expr _loc x5))
                          | Ast.ExFlo (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExFlo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExCoe (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "ExCoe")))))),
                                         (meta_loc _loc x0))),
                                      (meta_expr _loc x1))),
                                   (meta_ctyp _loc x2))),
                                (meta_ctyp _loc x3))
                          | Ast.ExChr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExChr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.ExAss (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExAss")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExAsr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExAsr")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExAsf x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "ExAsf")))))),
                                (meta_loc _loc x0))
                          | Ast.ExSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExArr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExArr")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.ExAre (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExAre")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExApp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.ExAcc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "ExAcc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_expr _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.ExId (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "ExId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.ExNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "ExNil")))))),
                                (meta_loc _loc x0))
                        and meta_ident _loc =
                          function
                          | Ast.IdAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.IdUid (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "IdUid")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.IdLid (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "IdLid")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.IdApp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "IdApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.IdAcc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "IdAcc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                        and meta_match_case _loc =
                          function
                          | Ast.McAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.McArr (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "McArr")))))),
                                         (meta_loc _loc x0))),
                                      (meta_patt _loc x1))),
                                   (meta_expr _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.McOr (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "McOr")))))),
                                      (meta_loc _loc x0))),
                                   (meta_match_case _loc x1))),
                                (meta_match_case _loc x2))
                          | Ast.McNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "McNil")))))),
                                (meta_loc _loc x0))
                        and meta_meta_bool _loc =
                          function
                          | Ast.BAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.BFalse ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "BFalse")))))
                          | Ast.BTrue ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "BTrue")))))
                        and meta_meta_list mf_a _loc =
                          function
                          | Ast.LAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.LCons (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "LCons")))))),
                                   (mf_a _loc x0))),
                                (meta_meta_list mf_a _loc x1))
                          | Ast.LNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "LNil")))))
                        and meta_meta_option mf_a _loc =
                          function
                          | Ast.OAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.OSome x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "OSome")))))),
                                (mf_a _loc x0))
                          | Ast.ONone ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ONone")))))
                        and meta_module_binding _loc =
                          function
                          | Ast.MbAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.MbCol (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MbCol")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.MbColEq (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MbColEq")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_type _loc x2))),
                                (meta_module_expr _loc x3))
                          | Ast.MbAnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MbAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_binding _loc x1))),
                                (meta_module_binding _loc x2))
                          | Ast.MbNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "MbNil")))))),
                                (meta_loc _loc x0))
                        and meta_module_expr _loc =
                          function
                          | Ast.MeAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.MeAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MeAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_module_expr _loc x3))
                          | Ast.MePkg (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MePkg")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.MeTyc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MeTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_expr _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.MeStr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MeStr")))))),
                                   (meta_loc _loc x0))),
                                (meta_str_item _loc x1))
                          | Ast.MeFun (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MeFun")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_type _loc x2))),
                                (meta_module_expr _loc x3))
                          | Ast.MeApp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MeApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_expr _loc x1))),
                                (meta_module_expr _loc x2))
                          | Ast.MeId (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MeId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.MeNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "MeNil")))))),
                                (meta_loc _loc x0))
                        and meta_module_type _loc =
                          function
                          | Ast.MtAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.MtAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MtAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_module_type _loc x3))
                          | Ast.MtOf (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtOf")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_expr _loc x1))
                          | Ast.MtWit (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "MtWit")))))),
                                      (meta_loc _loc x0))),
                                   (meta_module_type _loc x1))),
                                (meta_with_constr _loc x2))
                          | Ast.MtSig (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtSig")))))),
                                   (meta_loc _loc x0))),
                                (meta_sig_item _loc x1))
                          | Ast.MtQuo (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtQuo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.MtFun (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "MtFun")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_module_type _loc x2))),
                                (meta_module_type _loc x3))
                          | Ast.MtAlias (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtAlias")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.MtId (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "MtId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.MtNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "MtNil")))))),
                                (meta_loc _loc x0))
                        and meta_mutable_flag _loc =
                          function
                          | Ast.MuAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.MuNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "MuNil")))))
                          | Ast.MuMutable ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "MuMutable")))))
                        and meta_override_flag _loc =
                          function
                          | Ast.OvAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.OvNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "OvNil")))))
                          | Ast.OvOverride ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "OvOverride")))))
                        and meta_patt _loc =
                          function
                          | Ast.PaExc (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaExc")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaMod (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaMod")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaAtt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "PaAtt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_str_item _loc x2))),
                                (meta_patt _loc x3))
                          | Ast.PaLaz (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaLaz")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaVrn (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaVrn")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaTyp (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaTyp")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.PaTyc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaTyc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.PaTup (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaTup")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaStr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaStr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaEq (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaRec (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaRec")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaRng (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaRng")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaOrp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaOrp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaOlbi (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "PaOlbi")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_patt _loc x2))),
                                (meta_expr _loc x3))
                          | Ast.PaOlb (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaOlb")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaLab (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaLab")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaFlo (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaFlo")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaNativeInt (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaNativeInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaInt64 (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaInt64")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaInt32 (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaInt32")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaInt (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaInt")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaChr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaChr")))))),
                                   (meta_loc _loc x0))),
                                (meta_string _loc x1))
                          | Ast.PaSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaCom (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaCom")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaArr (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaArr")))))),
                                   (meta_loc _loc x0))),
                                (meta_patt _loc x1))
                          | Ast.PaApp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaApp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaAny x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "PaAny")))))),
                                (meta_loc _loc x0))
                          | Ast.PaAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.PaAli (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "PaAli")))))),
                                      (meta_loc _loc x0))),
                                   (meta_patt _loc x1))),
                                (meta_patt _loc x2))
                          | Ast.PaId (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "PaId")))))),
                                   (meta_loc _loc x0))),
                                (meta_ident _loc x1))
                          | Ast.PaNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "PaNil")))))),
                                (meta_loc _loc x0))
                        and meta_private_flag _loc =
                          function
                          | Ast.PrAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.PrNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "PrNil")))))
                          | Ast.PrPrivate ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "PrPrivate")))))
                        and meta_rec_binding _loc =
                          function
                          | Ast.RbAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.RbEq (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "RbEq")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.RbSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "RbSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_binding _loc x1))),
                                (meta_rec_binding _loc x2))
                          | Ast.RbNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "RbNil")))))),
                                (meta_loc _loc x0))
                        and meta_rec_flag _loc =
                          function
                          | Ast.ReAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.ReNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ReNil")))))
                          | Ast.ReNonrecursive ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ReNonrecursive")))))
                          | Ast.ReRecursive ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ReRecursive")))))
                        and meta_row_var_flag _loc =
                          function
                          | Ast.RvAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.RvNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "RvNil")))))
                          | Ast.RvRowVar ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "RvRowVar")))))
                        and meta_sig_item _loc =
                          function
                          | Ast.SgAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.SgVal (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgVal")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.SgTyp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgTyp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_flag _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.SgOpn (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgOpn")))))),
                                      (meta_loc _loc x0))),
                                   (meta_override_flag _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.SgMty (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgMty")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.SgRecMod (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgRecMod")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_binding _loc x1))
                          | Ast.SgMod (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgMod")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.SgInc (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgInc")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_type _loc x1))
                          | Ast.SgExt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "SgExt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_ctyp _loc x2))),
                                (meta_meta_list meta_string _loc x3))
                          | Ast.SgExc (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgExc")))))),
                                   (meta_loc _loc x0))),
                                (meta_ctyp _loc x1))
                          | Ast.SgDir (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgDir")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.SgSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "SgSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_sig_item _loc x1))),
                                (meta_sig_item _loc x2))
                          | Ast.SgClt (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgClt")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.SgCls (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "SgCls")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.SgNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "SgNil")))))),
                                (meta_loc _loc x0))
                        and meta_str_item _loc =
                          function
                          | Ast.StAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.StVal (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StVal")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_flag _loc x1))),
                                (meta_binding _loc x2))
                          | Ast.StTyp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StTyp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_rec_flag _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.StOpn (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StOpn")))))),
                                      (meta_loc _loc x0))),
                                   (meta_override_flag _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.StMty (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StMty")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_type _loc x2))
                          | Ast.StRecMod (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StRecMod")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_binding _loc x1))
                          | Ast.StMod (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StMod")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_module_expr _loc x2))
                          | Ast.StInc (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StInc")))))),
                                   (meta_loc _loc x0))),
                                (meta_module_expr _loc x1))
                          | Ast.StExt (x0, x1, x2, x3) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaApp (_loc,
                                         (Ast.PaId (_loc,
                                            (Ast.IdAcc (_loc,
                                               (Ast.IdUid (_loc, "Ast")),
                                               (Ast.IdUid (_loc, "StExt")))))),
                                         (meta_loc _loc x0))),
                                      (meta_string _loc x1))),
                                   (meta_ctyp _loc x2))),
                                (meta_meta_list meta_string _loc x3))
                          | Ast.StExp (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StExp")))))),
                                   (meta_loc _loc x0))),
                                (meta_expr _loc x1))
                          | Ast.StExc (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StExc")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_meta_option meta_ident _loc x2))
                          | Ast.StDir (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StDir")))))),
                                      (meta_loc _loc x0))),
                                   (meta_string _loc x1))),
                                (meta_expr _loc x2))
                          | Ast.StSem (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "StSem")))))),
                                      (meta_loc _loc x0))),
                                   (meta_str_item _loc x1))),
                                (meta_str_item _loc x2))
                          | Ast.StClt (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StClt")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_type _loc x1))
                          | Ast.StCls (x0, x1) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaId (_loc,
                                      (Ast.IdAcc (_loc,
                                         (Ast.IdUid (_loc, "Ast")),
                                         (Ast.IdUid (_loc, "StCls")))))),
                                   (meta_loc _loc x0))),
                                (meta_class_expr _loc x1))
                          | Ast.StNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "StNil")))))),
                                (meta_loc _loc x0))
                        and meta_virtual_flag _loc =
                          function
                          | Ast.ViAnt x0 -> Ast.PaAnt (_loc, x0)
                          | Ast.ViNil ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ViNil")))))
                          | Ast.ViVirtual ->
                              Ast.PaId (_loc,
                                (Ast.IdAcc (_loc, (Ast.IdUid (_loc, "Ast")),
                                   (Ast.IdUid (_loc, "ViVirtual")))))
                        and meta_with_constr _loc =
                          function
                          | Ast.WcAnt (x0, x1) -> Ast.PaAnt (x0, x1)
                          | Ast.WcAnd (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcAnd")))))),
                                      (meta_loc _loc x0))),
                                   (meta_with_constr _loc x1))),
                                (meta_with_constr _loc x2))
                          | Ast.WcMoS (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcMoS")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.WcTyS (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcTyS")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.WcMod (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcMod")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ident _loc x1))),
                                (meta_ident _loc x2))
                          | Ast.WcTyp (x0, x1, x2) ->
                              Ast.PaApp (_loc,
                                (Ast.PaApp (_loc,
                                   (Ast.PaApp (_loc,
                                      (Ast.PaId (_loc,
                                         (Ast.IdAcc (_loc,
                                            (Ast.IdUid (_loc, "Ast")),
                                            (Ast.IdUid (_loc, "WcTyp")))))),
                                      (meta_loc _loc x0))),
                                   (meta_ctyp _loc x1))),
                                (meta_ctyp _loc x2))
                          | Ast.WcNil x0 ->
                              Ast.PaApp (_loc,
                                (Ast.PaId (_loc,
                                   (Ast.IdAcc (_loc,
                                      (Ast.IdUid (_loc, "Ast")),
                                      (Ast.IdUid (_loc, "WcNil")))))),
                                (meta_loc _loc x0))
                          
                      end
                      
                  end
                  
              end
              
            class map =
              object ((o : 'self_type))
                method string : string -> string = o#unknown
                method list :
                  'a 'a_out.
                    ('self_type -> 'a -> 'a_out) -> 'a list -> 'a_out list =
                  fun _f_a ->
                    function
                    | [] -> []
                    | _x :: _x_i1 ->
                        let _x = _f_a o _x in
                        let _x_i1 = o#list _f_a _x_i1 in _x :: _x_i1
                method with_constr : with_constr -> with_constr =
                  function
                  | WcNil _x -> let _x = o#loc _x in WcNil _x
                  | WcTyp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in WcTyp (_x, _x_i1, _x_i2)
                  | WcMod (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#ident _x_i2 in WcMod (_x, _x_i1, _x_i2)
                  | WcTyS (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in WcTyS (_x, _x_i1, _x_i2)
                  | WcMoS (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#ident _x_i2 in WcMoS (_x, _x_i1, _x_i2)
                  | WcAnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#with_constr _x_i1 in
                      let _x_i2 = o#with_constr _x_i2
                      in WcAnd (_x, _x_i1, _x_i2)
                  | WcAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in WcAnt (_x, _x_i1)
                method virtual_flag : virtual_flag -> virtual_flag =
                  function
                  | ViVirtual -> ViVirtual
                  | ViNil -> ViNil
                  | ViAnt _x -> let _x = o#string _x in ViAnt _x
                method str_item : str_item -> str_item =
                  function
                  | StNil _x -> let _x = o#loc _x in StNil _x
                  | StCls (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_expr _x_i1 in StCls (_x, _x_i1)
                  | StClt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in StClt (_x, _x_i1)
                  | StSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#str_item _x_i1 in
                      let _x_i2 = o#str_item _x_i2
                      in StSem (_x, _x_i1, _x_i2)
                  | StDir (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#expr _x_i2 in StDir (_x, _x_i1, _x_i2)
                  | StExc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#meta_option (fun o -> o#ident) _x_i2
                      in StExc (_x, _x_i1, _x_i2)
                  | StExp (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in StExp (_x, _x_i1)
                  | StExt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in
                      let _x_i3 = o#meta_list (fun o -> o#string) _x_i3
                      in StExt (_x, _x_i1, _x_i2, _x_i3)
                  | StInc (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_expr _x_i1 in StInc (_x, _x_i1)
                  | StMod (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_expr _x_i2
                      in StMod (_x, _x_i1, _x_i2)
                  | StRecMod (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_binding _x_i1
                      in StRecMod (_x, _x_i1)
                  | StMty (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2
                      in StMty (_x, _x_i1, _x_i2)
                  | StOpn (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#override_flag _x_i1 in
                      let _x_i2 = o#ident _x_i2 in StOpn (_x, _x_i1, _x_i2)
                  | StTyp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_flag _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in StTyp (_x, _x_i1, _x_i2)
                  | StVal (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_flag _x_i1 in
                      let _x_i2 = o#binding _x_i2 in StVal (_x, _x_i1, _x_i2)
                  | StAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in StAnt (_x, _x_i1)
                method sig_item : sig_item -> sig_item =
                  function
                  | SgNil _x -> let _x = o#loc _x in SgNil _x
                  | SgCls (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in SgCls (_x, _x_i1)
                  | SgClt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in SgClt (_x, _x_i1)
                  | SgSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#sig_item _x_i1 in
                      let _x_i2 = o#sig_item _x_i2
                      in SgSem (_x, _x_i1, _x_i2)
                  | SgDir (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#expr _x_i2 in SgDir (_x, _x_i1, _x_i2)
                  | SgExc (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in SgExc (_x, _x_i1)
                  | SgExt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in
                      let _x_i3 = o#meta_list (fun o -> o#string) _x_i3
                      in SgExt (_x, _x_i1, _x_i2, _x_i3)
                  | SgInc (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_type _x_i1 in SgInc (_x, _x_i1)
                  | SgMod (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2
                      in SgMod (_x, _x_i1, _x_i2)
                  | SgRecMod (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_binding _x_i1
                      in SgRecMod (_x, _x_i1)
                  | SgMty (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2
                      in SgMty (_x, _x_i1, _x_i2)
                  | SgOpn (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#override_flag _x_i1 in
                      let _x_i2 = o#ident _x_i2 in SgOpn (_x, _x_i1, _x_i2)
                  | SgTyp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_flag _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in SgTyp (_x, _x_i1, _x_i2)
                  | SgVal (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in SgVal (_x, _x_i1, _x_i2)
                  | SgAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in SgAnt (_x, _x_i1)
                method row_var_flag : row_var_flag -> row_var_flag =
                  function
                  | RvRowVar -> RvRowVar
                  | RvNil -> RvNil
                  | RvAnt _x -> let _x = o#string _x in RvAnt _x
                method rec_flag : rec_flag -> rec_flag =
                  function
                  | ReRecursive -> ReRecursive
                  | ReNonrecursive -> ReNonrecursive
                  | ReNil -> ReNil
                  | ReAnt _x -> let _x = o#string _x in ReAnt _x
                method rec_binding : rec_binding -> rec_binding =
                  function
                  | RbNil _x -> let _x = o#loc _x in RbNil _x
                  | RbSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_binding _x_i1 in
                      let _x_i2 = o#rec_binding _x_i2
                      in RbSem (_x, _x_i1, _x_i2)
                  | RbEq (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#expr _x_i2 in RbEq (_x, _x_i1, _x_i2)
                  | RbAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in RbAnt (_x, _x_i1)
                method private_flag : private_flag -> private_flag =
                  function
                  | PrPrivate -> PrPrivate
                  | PrNil -> PrNil
                  | PrAnt _x -> let _x = o#string _x in PrAnt _x
                method patt : patt -> patt =
                  function
                  | PaNil _x -> let _x = o#loc _x in PaNil _x
                  | PaId (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in PaId (_x, _x_i1)
                  | PaAli (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaAli (_x, _x_i1, _x_i2)
                  | PaAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaAnt (_x, _x_i1)
                  | PaAny _x -> let _x = o#loc _x in PaAny _x
                  | PaApp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaApp (_x, _x_i1, _x_i2)
                  | PaArr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in PaArr (_x, _x_i1)
                  | PaCom (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaCom (_x, _x_i1, _x_i2)
                  | PaSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaSem (_x, _x_i1, _x_i2)
                  | PaChr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaChr (_x, _x_i1)
                  | PaInt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaInt (_x, _x_i1)
                  | PaInt32 (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaInt32 (_x, _x_i1)
                  | PaInt64 (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaInt64 (_x, _x_i1)
                  | PaNativeInt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaNativeInt (_x, _x_i1)
                  | PaFlo (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaFlo (_x, _x_i1)
                  | PaLab (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaLab (_x, _x_i1, _x_i2)
                  | PaOlb (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaOlb (_x, _x_i1, _x_i2)
                  | PaOlbi (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#patt _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in PaOlbi (_x, _x_i1, _x_i2, _x_i3)
                  | PaOrp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaOrp (_x, _x_i1, _x_i2)
                  | PaRng (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaRng (_x, _x_i1, _x_i2)
                  | PaRec (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in PaRec (_x, _x_i1)
                  | PaEq (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#patt _x_i2 in PaEq (_x, _x_i1, _x_i2)
                  | PaStr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaStr (_x, _x_i1)
                  | PaTup (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in PaTup (_x, _x_i1)
                  | PaTyc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in PaTyc (_x, _x_i1, _x_i2)
                  | PaTyp (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in PaTyp (_x, _x_i1)
                  | PaVrn (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaVrn (_x, _x_i1)
                  | PaLaz (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in PaLaz (_x, _x_i1)
                  | PaAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#patt _x_i3
                      in PaAtt (_x, _x_i1, _x_i2, _x_i3)
                  | PaMod (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in PaMod (_x, _x_i1)
                  | PaExc (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in PaExc (_x, _x_i1)
                method override_flag : override_flag -> override_flag =
                  function
                  | OvOverride -> OvOverride
                  | OvNil -> OvNil
                  | OvAnt _x -> let _x = o#string _x in OvAnt _x
                method mutable_flag : mutable_flag -> mutable_flag =
                  function
                  | MuMutable -> MuMutable
                  | MuNil -> MuNil
                  | MuAnt _x -> let _x = o#string _x in MuAnt _x
                method module_type : module_type -> module_type =
                  function
                  | MtNil _x -> let _x = o#loc _x in MtNil _x
                  | MtId (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in MtId (_x, _x_i1)
                  | MtAlias (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in MtAlias (_x, _x_i1)
                  | MtFun (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2 in
                      let _x_i3 = o#module_type _x_i3
                      in MtFun (_x, _x_i1, _x_i2, _x_i3)
                  | MtQuo (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in MtQuo (_x, _x_i1)
                  | MtSig (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#sig_item _x_i1 in MtSig (_x, _x_i1)
                  | MtWit (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_type _x_i1 in
                      let _x_i2 = o#with_constr _x_i2
                      in MtWit (_x, _x_i1, _x_i2)
                  | MtOf (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_expr _x_i1 in MtOf (_x, _x_i1)
                  | MtAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#module_type _x_i3
                      in MtAtt (_x, _x_i1, _x_i2, _x_i3)
                  | MtAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in MtAnt (_x, _x_i1)
                method module_expr : module_expr -> module_expr =
                  function
                  | MeNil _x -> let _x = o#loc _x in MeNil _x
                  | MeId (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in MeId (_x, _x_i1)
                  | MeApp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_expr _x_i1 in
                      let _x_i2 = o#module_expr _x_i2
                      in MeApp (_x, _x_i1, _x_i2)
                  | MeFun (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2 in
                      let _x_i3 = o#module_expr _x_i3
                      in MeFun (_x, _x_i1, _x_i2, _x_i3)
                  | MeStr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#str_item _x_i1 in MeStr (_x, _x_i1)
                  | MeTyc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_expr _x_i1 in
                      let _x_i2 = o#module_type _x_i2
                      in MeTyc (_x, _x_i1, _x_i2)
                  | MePkg (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in MePkg (_x, _x_i1)
                  | MeAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#module_expr _x_i3
                      in MeAtt (_x, _x_i1, _x_i2, _x_i3)
                  | MeAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in MeAnt (_x, _x_i1)
                method module_binding : module_binding -> module_binding =
                  function
                  | MbNil _x -> let _x = o#loc _x in MbNil _x
                  | MbAnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_binding _x_i1 in
                      let _x_i2 = o#module_binding _x_i2
                      in MbAnd (_x, _x_i1, _x_i2)
                  | MbColEq (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2 in
                      let _x_i3 = o#module_expr _x_i3
                      in MbColEq (_x, _x_i1, _x_i2, _x_i3)
                  | MbCol (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_type _x_i2
                      in MbCol (_x, _x_i1, _x_i2)
                  | MbAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in MbAnt (_x, _x_i1)
                method meta_option :
                  'a 'a_out.
                    ('self_type -> 'a -> 'a_out) ->
                      'a meta_option -> 'a_out meta_option =
                  fun _f_a ->
                    function
                    | ONone -> ONone
                    | OSome _x -> let _x = _f_a o _x in OSome _x
                    | OAnt _x -> let _x = o#string _x in OAnt _x
                method meta_list :
                  'a 'a_out.
                    ('self_type -> 'a -> 'a_out) ->
                      'a meta_list -> 'a_out meta_list =
                  fun _f_a ->
                    function
                    | LNil -> LNil
                    | LCons (_x, _x_i1) ->
                        let _x = _f_a o _x in
                        let _x_i1 = o#meta_list _f_a _x_i1
                        in LCons (_x, _x_i1)
                    | LAnt _x -> let _x = o#string _x in LAnt _x
                method meta_bool : meta_bool -> meta_bool =
                  function
                  | BTrue -> BTrue
                  | BFalse -> BFalse
                  | BAnt _x -> let _x = o#string _x in BAnt _x
                method match_case : match_case -> match_case =
                  function
                  | McNil _x -> let _x = o#loc _x in McNil _x
                  | McOr (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#match_case _x_i1 in
                      let _x_i2 = o#match_case _x_i2
                      in McOr (_x, _x_i1, _x_i2)
                  | McArr (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#expr _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in McArr (_x, _x_i1, _x_i2, _x_i3)
                  | McAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in McAnt (_x, _x_i1)
                method loc : loc -> loc = o#unknown
                method ident : ident -> ident =
                  function
                  | IdAcc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#ident _x_i2 in IdAcc (_x, _x_i1, _x_i2)
                  | IdApp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#ident _x_i2 in IdApp (_x, _x_i1, _x_i2)
                  | IdLid (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in IdLid (_x, _x_i1)
                  | IdUid (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in IdUid (_x, _x_i1)
                  | IdAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in IdAnt (_x, _x_i1)
                method expr : expr -> expr =
                  function
                  | ExNil _x -> let _x = o#loc _x in ExNil _x
                  | ExId (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in ExId (_x, _x_i1)
                  | ExAcc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExAcc (_x, _x_i1, _x_i2)
                  | ExAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExAnt (_x, _x_i1)
                  | ExApp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExApp (_x, _x_i1, _x_i2)
                  | ExAre (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExAre (_x, _x_i1, _x_i2)
                  | ExArr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in ExArr (_x, _x_i1)
                  | ExSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExSem (_x, _x_i1, _x_i2)
                  | ExAsf _x -> let _x = o#loc _x in ExAsf _x
                  | ExAsr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in ExAsr (_x, _x_i1)
                  | ExAss (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExAss (_x, _x_i1, _x_i2)
                  | ExChr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExChr (_x, _x_i1)
                  | ExCoe (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in ExCoe (_x, _x_i1, _x_i2, _x_i3)
                  | ExFlo (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExFlo (_x, _x_i1)
                  | ExFor (_x, _x_i1, _x_i2, _x_i3, _x_i4, _x_i5) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#expr _x_i2 in
                      let _x_i3 = o#expr _x_i3 in
                      let _x_i4 = o#direction_flag _x_i4 in
                      let _x_i5 = o#expr _x_i5
                      in ExFor (_x, _x_i1, _x_i2, _x_i3, _x_i4, _x_i5)
                  | ExFun (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#match_case _x_i1 in ExFun (_x, _x_i1)
                  | ExIfe (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in ExIfe (_x, _x_i1, _x_i2, _x_i3)
                  | ExInt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExInt (_x, _x_i1)
                  | ExInt32 (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExInt32 (_x, _x_i1)
                  | ExInt64 (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExInt64 (_x, _x_i1)
                  | ExNativeInt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExNativeInt (_x, _x_i1)
                  | ExLab (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExLab (_x, _x_i1, _x_i2)
                  | ExLaz (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in ExLaz (_x, _x_i1)
                  | ExLet (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_flag _x_i1 in
                      let _x_i2 = o#binding _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in ExLet (_x, _x_i1, _x_i2, _x_i3)
                  | ExLmd (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#module_expr _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in ExLmd (_x, _x_i1, _x_i2, _x_i3)
                  | ExMat (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#match_case _x_i2
                      in ExMat (_x, _x_i1, _x_i2)
                  | ExNew (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in ExNew (_x, _x_i1)
                  | ExObj (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#class_str_item _x_i2
                      in ExObj (_x, _x_i1, _x_i2)
                  | ExOlb (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExOlb (_x, _x_i1, _x_i2)
                  | ExOvr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_binding _x_i1 in ExOvr (_x, _x_i1)
                  | ExRec (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_binding _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExRec (_x, _x_i1, _x_i2)
                  | ExSeq (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in ExSeq (_x, _x_i1)
                  | ExSnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#string _x_i2 in ExSnd (_x, _x_i1, _x_i2)
                  | ExSte (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExSte (_x, _x_i1, _x_i2)
                  | ExStr (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExStr (_x, _x_i1)
                  | ExTry (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#match_case _x_i2
                      in ExTry (_x, _x_i1, _x_i2)
                  | ExTup (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in ExTup (_x, _x_i1)
                  | ExCom (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExCom (_x, _x_i1, _x_i2)
                  | ExTyc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in ExTyc (_x, _x_i1, _x_i2)
                  | ExVrn (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in ExVrn (_x, _x_i1)
                  | ExWhi (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExWhi (_x, _x_i1, _x_i2)
                  | ExOpI (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#override_flag _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in ExOpI (_x, _x_i1, _x_i2, _x_i3)
                  | ExFUN (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#expr _x_i2 in ExFUN (_x, _x_i1, _x_i2)
                  | ExPkg (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_expr _x_i1 in ExPkg (_x, _x_i1)
                  | ExAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#expr _x_i3
                      in ExAtt (_x, _x_i1, _x_i2, _x_i3)
                method direction_flag : direction_flag -> direction_flag =
                  function
                  | DiTo -> DiTo
                  | DiDownto -> DiDownto
                  | DiAnt _x -> let _x = o#string _x in DiAnt _x
                method ctyp : ctyp -> ctyp =
                  function
                  | TyNil _x -> let _x = o#loc _x in TyNil _x
                  | TyAli (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyAli (_x, _x_i1, _x_i2)
                  | TyAny _x -> let _x = o#loc _x in TyAny _x
                  | TyApp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyApp (_x, _x_i1, _x_i2)
                  | TyArr (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyArr (_x, _x_i1, _x_i2)
                  | TyCls (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in TyCls (_x, _x_i1)
                  | TyLab (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyLab (_x, _x_i1, _x_i2)
                  | TyId (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in TyId (_x, _x_i1)
                  | TyMan (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyMan (_x, _x_i1, _x_i2)
                  | TyDcl (_x, _x_i1, _x_i2, _x_i3, _x_i4) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#list (fun o -> o#ctyp) _x_i2 in
                      let _x_i3 = o#ctyp _x_i3 in
                      let _x_i4 =
                        o#list
                          (fun o (_x, _x_i1) ->
                             let _x = o#ctyp _x in
                             let _x_i1 = o#ctyp _x_i1 in (_x, _x_i1))
                          _x_i4
                      in TyDcl (_x, _x_i1, _x_i2, _x_i3, _x_i4)
                  | TyExt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ident _x_i1 in
                      let _x_i2 = o#list (fun o -> o#ctyp) _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in TyExt (_x, _x_i1, _x_i2, _x_i3)
                  | TyObj (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#row_var_flag _x_i2
                      in TyObj (_x, _x_i1, _x_i2)
                  | TyOlb (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyOlb (_x, _x_i1, _x_i2)
                  | TyPol (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyPol (_x, _x_i1, _x_i2)
                  | TyTypePol (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2
                      in TyTypePol (_x, _x_i1, _x_i2)
                  | TyQuo (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in TyQuo (_x, _x_i1)
                  | TyQuP (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in TyQuP (_x, _x_i1)
                  | TyQuM (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in TyQuM (_x, _x_i1)
                  | TyAnP _x -> let _x = o#loc _x in TyAnP _x
                  | TyAnM _x -> let _x = o#loc _x in TyAnM _x
                  | TyVrn (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in TyVrn (_x, _x_i1)
                  | TyRec (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyRec (_x, _x_i1)
                  | TyCol (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyCol (_x, _x_i1, _x_i2)
                  | TySem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TySem (_x, _x_i1, _x_i2)
                  | TyCom (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyCom (_x, _x_i1, _x_i2)
                  | TySum (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TySum (_x, _x_i1)
                  | TyOf (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyOf (_x, _x_i1, _x_i2)
                  | TyAnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyAnd (_x, _x_i1, _x_i2)
                  | TyOr (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyOr (_x, _x_i1, _x_i2)
                  | TyPrv (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyPrv (_x, _x_i1)
                  | TyMut (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyMut (_x, _x_i1)
                  | TyTup (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyTup (_x, _x_i1)
                  | TySta (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TySta (_x, _x_i1, _x_i2)
                  | TyVrnEq (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyVrnEq (_x, _x_i1)
                  | TyVrnSup (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyVrnSup (_x, _x_i1)
                  | TyVrnInf (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in TyVrnInf (_x, _x_i1)
                  | TyVrnInfSup (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2
                      in TyVrnInfSup (_x, _x_i1, _x_i2)
                  | TyAmp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyAmp (_x, _x_i1, _x_i2)
                  | TyOfAmp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in TyOfAmp (_x, _x_i1, _x_i2)
                  | TyPkg (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#module_type _x_i1 in TyPkg (_x, _x_i1)
                  | TyOpn _x -> let _x = o#loc _x in TyOpn _x
                  | TyAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in TyAtt (_x, _x_i1, _x_i2, _x_i3)
                  | TyAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in TyAnt (_x, _x_i1)
                method class_type : class_type -> class_type =
                  function
                  | CtNil _x -> let _x = o#loc _x in CtNil _x
                  | CtCon (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#virtual_flag _x_i1 in
                      let _x_i2 = o#ident _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in CtCon (_x, _x_i1, _x_i2, _x_i3)
                  | CtFun (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#class_type _x_i2
                      in CtFun (_x, _x_i1, _x_i2)
                  | CtSig (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#class_sig_item _x_i2
                      in CtSig (_x, _x_i1, _x_i2)
                  | CtAnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in
                      let _x_i2 = o#class_type _x_i2
                      in CtAnd (_x, _x_i1, _x_i2)
                  | CtCol (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in
                      let _x_i2 = o#class_type _x_i2
                      in CtCol (_x, _x_i1, _x_i2)
                  | CtEq (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in
                      let _x_i2 = o#class_type _x_i2
                      in CtEq (_x, _x_i1, _x_i2)
                  | CtAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#class_type _x_i3
                      in CtAtt (_x, _x_i1, _x_i2, _x_i3)
                  | CtAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in CtAnt (_x, _x_i1)
                method class_str_item : class_str_item -> class_str_item =
                  function
                  | CrNil _x -> let _x = o#loc _x in CrNil _x
                  | CrSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_str_item _x_i1 in
                      let _x_i2 = o#class_str_item _x_i2
                      in CrSem (_x, _x_i1, _x_i2)
                  | CrCtr (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in CrCtr (_x, _x_i1, _x_i2)
                  | CrInh (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#override_flag _x_i1 in
                      let _x_i2 = o#class_expr _x_i2 in
                      let _x_i3 = o#string _x_i3
                      in CrInh (_x, _x_i1, _x_i2, _x_i3)
                  | CrIni (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#expr _x_i1 in CrIni (_x, _x_i1)
                  | CrMth (_x, _x_i1, _x_i2, _x_i3, _x_i4, _x_i5) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#override_flag _x_i2 in
                      let _x_i3 = o#private_flag _x_i3 in
                      let _x_i4 = o#expr _x_i4 in
                      let _x_i5 = o#ctyp _x_i5
                      in CrMth (_x, _x_i1, _x_i2, _x_i3, _x_i4, _x_i5)
                  | CrVal (_x, _x_i1, _x_i2, _x_i3, _x_i4) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#override_flag _x_i2 in
                      let _x_i3 = o#mutable_flag _x_i3 in
                      let _x_i4 = o#expr _x_i4
                      in CrVal (_x, _x_i1, _x_i2, _x_i3, _x_i4)
                  | CrVir (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#private_flag _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in CrVir (_x, _x_i1, _x_i2, _x_i3)
                  | CrVvr (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#mutable_flag _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in CrVvr (_x, _x_i1, _x_i2, _x_i3)
                  | CrAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in CrAnt (_x, _x_i1)
                method class_sig_item : class_sig_item -> class_sig_item =
                  function
                  | CgNil _x -> let _x = o#loc _x in CgNil _x
                  | CgCtr (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#ctyp _x_i1 in
                      let _x_i2 = o#ctyp _x_i2 in CgCtr (_x, _x_i1, _x_i2)
                  | CgSem (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_sig_item _x_i1 in
                      let _x_i2 = o#class_sig_item _x_i2
                      in CgSem (_x, _x_i1, _x_i2)
                  | CgInh (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_type _x_i1 in CgInh (_x, _x_i1)
                  | CgMth (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#private_flag _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in CgMth (_x, _x_i1, _x_i2, _x_i3)
                  | CgVal (_x, _x_i1, _x_i2, _x_i3, _x_i4) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#mutable_flag _x_i2 in
                      let _x_i3 = o#virtual_flag _x_i3 in
                      let _x_i4 = o#ctyp _x_i4
                      in CgVal (_x, _x_i1, _x_i2, _x_i3, _x_i4)
                  | CgVir (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#private_flag _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in CgVir (_x, _x_i1, _x_i2, _x_i3)
                  | CgAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in CgAnt (_x, _x_i1)
                method class_expr : class_expr -> class_expr =
                  function
                  | CeNil _x -> let _x = o#loc _x in CeNil _x
                  | CeApp (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_expr _x_i1 in
                      let _x_i2 = o#expr _x_i2 in CeApp (_x, _x_i1, _x_i2)
                  | CeCon (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#virtual_flag _x_i1 in
                      let _x_i2 = o#ident _x_i2 in
                      let _x_i3 = o#ctyp _x_i3
                      in CeCon (_x, _x_i1, _x_i2, _x_i3)
                  | CeFun (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#class_expr _x_i2
                      in CeFun (_x, _x_i1, _x_i2)
                  | CeLet (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#rec_flag _x_i1 in
                      let _x_i2 = o#binding _x_i2 in
                      let _x_i3 = o#class_expr _x_i3
                      in CeLet (_x, _x_i1, _x_i2, _x_i3)
                  | CeStr (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#class_str_item _x_i2
                      in CeStr (_x, _x_i1, _x_i2)
                  | CeTyc (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_expr _x_i1 in
                      let _x_i2 = o#class_type _x_i2
                      in CeTyc (_x, _x_i1, _x_i2)
                  | CeAnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_expr _x_i1 in
                      let _x_i2 = o#class_expr _x_i2
                      in CeAnd (_x, _x_i1, _x_i2)
                  | CeEq (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#class_expr _x_i1 in
                      let _x_i2 = o#class_expr _x_i2
                      in CeEq (_x, _x_i1, _x_i2)
                  | CeAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in
                      let _x_i2 = o#str_item _x_i2 in
                      let _x_i3 = o#class_expr _x_i3
                      in CeAtt (_x, _x_i1, _x_i2, _x_i3)
                  | CeAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in CeAnt (_x, _x_i1)
                method binding : binding -> binding =
                  function
                  | BiNil _x -> let _x = o#loc _x in BiNil _x
                  | BiAnd (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#binding _x_i1 in
                      let _x_i2 = o#binding _x_i2 in BiAnd (_x, _x_i1, _x_i2)
                  | BiEq (_x, _x_i1, _x_i2) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#patt _x_i1 in
                      let _x_i2 = o#expr _x_i2 in BiEq (_x, _x_i1, _x_i2)
                  | BiAnt (_x, _x_i1) ->
                      let _x = o#loc _x in
                      let _x_i1 = o#string _x_i1 in BiAnt (_x, _x_i1)
                method unknown : 'a. 'a -> 'a = fun x -> x
              end
              
            class fold =
              object ((o : 'self_type))
                method string : string -> 'self_type = o#unknown
                method list :
                  'a.
                    ('self_type -> 'a -> 'self_type) -> 'a list -> 'self_type =
                  fun _f_a ->
                    function
                    | [] -> o
                    | _x :: _x_i1 ->
                        let o = _f_a o _x in let o = o#list _f_a _x_i1 in o
                method with_constr : with_constr -> 'self_type =
                  function
                  | WcNil _x -> let o = o#loc _x in o
                  | WcTyp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | WcMod (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in let o = o#ident _x_i2 in o
                  | WcTyS (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | WcMoS (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in let o = o#ident _x_i2 in o
                  | WcAnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#with_constr _x_i1 in
                      let o = o#with_constr _x_i2 in o
                  | WcAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method virtual_flag : virtual_flag -> 'self_type =
                  function
                  | ViVirtual -> o
                  | ViNil -> o
                  | ViAnt _x -> let o = o#string _x in o
                method str_item : str_item -> 'self_type =
                  function
                  | StNil _x -> let o = o#loc _x in o
                  | StCls (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#class_expr _x_i1 in o
                  | StClt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#class_type _x_i1 in o
                  | StSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#str_item _x_i1 in
                      let o = o#str_item _x_i2 in o
                  | StDir (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#expr _x_i2 in o
                  | StExc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in
                      let o = o#meta_option (fun o -> o#ident) _x_i2 in o
                  | StExp (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | StExt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#ctyp _x_i2 in
                      let o = o#meta_list (fun o -> o#string) _x_i3 in o
                  | StInc (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_expr _x_i1 in o
                  | StMod (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_expr _x_i2 in o
                  | StRecMod (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_binding _x_i1 in o
                  | StMty (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in o
                  | StOpn (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#override_flag _x_i1 in
                      let o = o#ident _x_i2 in o
                  | StTyp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#rec_flag _x_i1 in let o = o#ctyp _x_i2 in o
                  | StVal (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#rec_flag _x_i1 in
                      let o = o#binding _x_i2 in o
                  | StAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method sig_item : sig_item -> 'self_type =
                  function
                  | SgNil _x -> let o = o#loc _x in o
                  | SgCls (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#class_type _x_i1 in o
                  | SgClt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#class_type _x_i1 in o
                  | SgSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#sig_item _x_i1 in
                      let o = o#sig_item _x_i2 in o
                  | SgDir (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#expr _x_i2 in o
                  | SgExc (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | SgExt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#ctyp _x_i2 in
                      let o = o#meta_list (fun o -> o#string) _x_i3 in o
                  | SgInc (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_type _x_i1 in o
                  | SgMod (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in o
                  | SgRecMod (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_binding _x_i1 in o
                  | SgMty (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in o
                  | SgOpn (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#override_flag _x_i1 in
                      let o = o#ident _x_i2 in o
                  | SgTyp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#rec_flag _x_i1 in let o = o#ctyp _x_i2 in o
                  | SgVal (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#ctyp _x_i2 in o
                  | SgAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method row_var_flag : row_var_flag -> 'self_type =
                  function
                  | RvRowVar -> o
                  | RvNil -> o
                  | RvAnt _x -> let o = o#string _x in o
                method rec_flag : rec_flag -> 'self_type =
                  function
                  | ReRecursive -> o
                  | ReNonrecursive -> o
                  | ReNil -> o
                  | ReAnt _x -> let o = o#string _x in o
                method rec_binding : rec_binding -> 'self_type =
                  function
                  | RbNil _x -> let o = o#loc _x in o
                  | RbSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#rec_binding _x_i1 in
                      let o = o#rec_binding _x_i2 in o
                  | RbEq (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in let o = o#expr _x_i2 in o
                  | RbAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method private_flag : private_flag -> 'self_type =
                  function
                  | PrPrivate -> o
                  | PrNil -> o
                  | PrAnt _x -> let o = o#string _x in o
                method patt : patt -> 'self_type =
                  function
                  | PaNil _x -> let o = o#loc _x in o
                  | PaId (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | PaAli (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#patt _x_i2 in o
                  | PaAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaAny _x -> let o = o#loc _x in o
                  | PaApp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#patt _x_i2 in o
                  | PaArr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#patt _x_i1 in o
                  | PaCom (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#patt _x_i2 in o
                  | PaSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#patt _x_i2 in o
                  | PaChr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaInt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaInt32 (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaInt64 (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaNativeInt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaFlo (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaLab (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#patt _x_i2 in o
                  | PaOlb (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#patt _x_i2 in o
                  | PaOlbi (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#patt _x_i2 in let o = o#expr _x_i3 in o
                  | PaOrp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#patt _x_i2 in o
                  | PaRng (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#patt _x_i2 in o
                  | PaRec (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#patt _x_i1 in o
                  | PaEq (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in let o = o#patt _x_i2 in o
                  | PaStr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaTup (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#patt _x_i1 in o
                  | PaTyc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#ctyp _x_i2 in o
                  | PaTyp (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | PaVrn (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaLaz (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#patt _x_i1 in o
                  | PaAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in let o = o#patt _x_i3 in o
                  | PaMod (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | PaExc (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#patt _x_i1 in o
                method override_flag : override_flag -> 'self_type =
                  function
                  | OvOverride -> o
                  | OvNil -> o
                  | OvAnt _x -> let o = o#string _x in o
                method mutable_flag : mutable_flag -> 'self_type =
                  function
                  | MuMutable -> o
                  | MuNil -> o
                  | MuAnt _x -> let o = o#string _x in o
                method module_type : module_type -> 'self_type =
                  function
                  | MtNil _x -> let o = o#loc _x in o
                  | MtId (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | MtAlias (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | MtFun (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in
                      let o = o#module_type _x_i3 in o
                  | MtQuo (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | MtSig (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#sig_item _x_i1 in o
                  | MtWit (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#module_type _x_i1 in
                      let o = o#with_constr _x_i2 in o
                  | MtOf (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_expr _x_i1 in o
                  | MtAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in
                      let o = o#module_type _x_i3 in o
                  | MtAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method module_expr : module_expr -> 'self_type =
                  function
                  | MeNil _x -> let o = o#loc _x in o
                  | MeId (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | MeApp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#module_expr _x_i1 in
                      let o = o#module_expr _x_i2 in o
                  | MeFun (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in
                      let o = o#module_expr _x_i3 in o
                  | MeStr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#str_item _x_i1 in o
                  | MeTyc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#module_expr _x_i1 in
                      let o = o#module_type _x_i2 in o
                  | MePkg (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | MeAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in
                      let o = o#module_expr _x_i3 in o
                  | MeAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method module_binding : module_binding -> 'self_type =
                  function
                  | MbNil _x -> let o = o#loc _x in o
                  | MbAnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#module_binding _x_i1 in
                      let o = o#module_binding _x_i2 in o
                  | MbColEq (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in
                      let o = o#module_expr _x_i3 in o
                  | MbCol (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_type _x_i2 in o
                  | MbAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method meta_option :
                  'a.
                    ('self_type -> 'a -> 'self_type) ->
                      'a meta_option -> 'self_type =
                  fun _f_a ->
                    function
                    | ONone -> o
                    | OSome _x -> let o = _f_a o _x in o
                    | OAnt _x -> let o = o#string _x in o
                method meta_list :
                  'a.
                    ('self_type -> 'a -> 'self_type) ->
                      'a meta_list -> 'self_type =
                  fun _f_a ->
                    function
                    | LNil -> o
                    | LCons (_x, _x_i1) ->
                        let o = _f_a o _x in
                        let o = o#meta_list _f_a _x_i1 in o
                    | LAnt _x -> let o = o#string _x in o
                method meta_bool : meta_bool -> 'self_type =
                  function
                  | BTrue -> o
                  | BFalse -> o
                  | BAnt _x -> let o = o#string _x in o
                method match_case : match_case -> 'self_type =
                  function
                  | McNil _x -> let o = o#loc _x in o
                  | McOr (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#match_case _x_i1 in
                      let o = o#match_case _x_i2 in o
                  | McArr (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in
                      let o = o#expr _x_i2 in let o = o#expr _x_i3 in o
                  | McAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method loc : loc -> 'self_type = o#unknown
                method ident : ident -> 'self_type =
                  function
                  | IdAcc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in let o = o#ident _x_i2 in o
                  | IdApp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in let o = o#ident _x_i2 in o
                  | IdLid (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | IdUid (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | IdAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method expr : expr -> 'self_type =
                  function
                  | ExNil _x -> let o = o#loc _x in o
                  | ExId (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | ExAcc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExApp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExAre (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExArr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | ExSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExAsf _x -> let o = o#loc _x in o
                  | ExAsr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | ExAss (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExChr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExCoe (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in
                      let o = o#ctyp _x_i2 in let o = o#ctyp _x_i3 in o
                  | ExFlo (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExFor (_x, _x_i1, _x_i2, _x_i3, _x_i4, _x_i5) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in
                      let o = o#expr _x_i2 in
                      let o = o#expr _x_i3 in
                      let o = o#direction_flag _x_i4 in
                      let o = o#expr _x_i5 in o
                  | ExFun (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#match_case _x_i1 in o
                  | ExIfe (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in
                      let o = o#expr _x_i2 in let o = o#expr _x_i3 in o
                  | ExInt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExInt32 (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExInt64 (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExNativeInt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExLab (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#expr _x_i2 in o
                  | ExLaz (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | ExLet (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#rec_flag _x_i1 in
                      let o = o#binding _x_i2 in let o = o#expr _x_i3 in o
                  | ExLmd (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#module_expr _x_i2 in
                      let o = o#expr _x_i3 in o
                  | ExMat (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#match_case _x_i2 in o
                  | ExNew (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | ExObj (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in
                      let o = o#class_str_item _x_i2 in o
                  | ExOlb (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#expr _x_i2 in o
                  | ExOvr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#rec_binding _x_i1 in o
                  | ExRec (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#rec_binding _x_i1 in
                      let o = o#expr _x_i2 in o
                  | ExSeq (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | ExSnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#string _x_i2 in o
                  | ExSte (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExStr (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExTry (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#match_case _x_i2 in o
                  | ExTup (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | ExCom (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExTyc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#ctyp _x_i2 in o
                  | ExVrn (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | ExWhi (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#expr _x_i1 in let o = o#expr _x_i2 in o
                  | ExOpI (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in
                      let o = o#override_flag _x_i2 in
                      let o = o#expr _x_i3 in o
                  | ExFUN (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#expr _x_i2 in o
                  | ExPkg (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_expr _x_i1 in o
                  | ExAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in let o = o#expr _x_i3 in o
                method direction_flag : direction_flag -> 'self_type =
                  function
                  | DiTo -> o
                  | DiDownto -> o
                  | DiAnt _x -> let o = o#string _x in o
                method ctyp : ctyp -> 'self_type =
                  function
                  | TyNil _x -> let o = o#loc _x in o
                  | TyAli (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyAny _x -> let o = o#loc _x in o
                  | TyApp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyArr (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyCls (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | TyLab (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyId (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ident _x_i1 in o
                  | TyMan (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyDcl (_x, _x_i1, _x_i2, _x_i3, _x_i4) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#list (fun o -> o#ctyp) _x_i2 in
                      let o = o#ctyp _x_i3 in
                      let o =
                        o#list
                          (fun o (_x, _x_i1) ->
                             let o = o#ctyp _x in let o = o#ctyp _x_i1 in o)
                          _x_i4
                      in o
                  | TyExt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#ident _x_i1 in
                      let o = o#list (fun o -> o#ctyp) _x_i2 in
                      let o = o#ctyp _x_i3 in o
                  | TyObj (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in
                      let o = o#row_var_flag _x_i2 in o
                  | TyOlb (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyPol (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyTypePol (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyQuo (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | TyQuP (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | TyQuM (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | TyAnP _x -> let o = o#loc _x in o
                  | TyAnM _x -> let o = o#loc _x in o
                  | TyVrn (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                  | TyRec (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyCol (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TySem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyCom (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TySum (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyOf (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyAnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyOr (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyPrv (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyMut (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyTup (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TySta (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyVrnEq (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyVrnSup (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyVrnInf (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#ctyp _x_i1 in o
                  | TyVrnInfSup (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyAmp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyOfAmp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | TyPkg (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#module_type _x_i1 in o
                  | TyOpn _x -> let o = o#loc _x in o
                  | TyAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in let o = o#ctyp _x_i3 in o
                  | TyAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method class_type : class_type -> 'self_type =
                  function
                  | CtNil _x -> let o = o#loc _x in o
                  | CtCon (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#virtual_flag _x_i1 in
                      let o = o#ident _x_i2 in let o = o#ctyp _x_i3 in o
                  | CtFun (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#class_type _x_i2 in o
                  | CtSig (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in
                      let o = o#class_sig_item _x_i2 in o
                  | CtAnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_type _x_i1 in
                      let o = o#class_type _x_i2 in o
                  | CtCol (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_type _x_i1 in
                      let o = o#class_type _x_i2 in o
                  | CtEq (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_type _x_i1 in
                      let o = o#class_type _x_i2 in o
                  | CtAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in
                      let o = o#class_type _x_i3 in o
                  | CtAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method class_str_item : class_str_item -> 'self_type =
                  function
                  | CrNil _x -> let o = o#loc _x in o
                  | CrSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_str_item _x_i1 in
                      let o = o#class_str_item _x_i2 in o
                  | CrCtr (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | CrInh (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#override_flag _x_i1 in
                      let o = o#class_expr _x_i2 in
                      let o = o#string _x_i3 in o
                  | CrIni (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#expr _x_i1 in o
                  | CrMth (_x, _x_i1, _x_i2, _x_i3, _x_i4, _x_i5) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#override_flag _x_i2 in
                      let o = o#private_flag _x_i3 in
                      let o = o#expr _x_i4 in let o = o#ctyp _x_i5 in o
                  | CrVal (_x, _x_i1, _x_i2, _x_i3, _x_i4) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#override_flag _x_i2 in
                      let o = o#mutable_flag _x_i3 in
                      let o = o#expr _x_i4 in o
                  | CrVir (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#private_flag _x_i2 in
                      let o = o#ctyp _x_i3 in o
                  | CrVvr (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#mutable_flag _x_i2 in
                      let o = o#ctyp _x_i3 in o
                  | CrAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method class_sig_item : class_sig_item -> 'self_type =
                  function
                  | CgNil _x -> let o = o#loc _x in o
                  | CgCtr (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#ctyp _x_i1 in let o = o#ctyp _x_i2 in o
                  | CgSem (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_sig_item _x_i1 in
                      let o = o#class_sig_item _x_i2 in o
                  | CgInh (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#class_type _x_i1 in o
                  | CgMth (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#private_flag _x_i2 in
                      let o = o#ctyp _x_i3 in o
                  | CgVal (_x, _x_i1, _x_i2, _x_i3, _x_i4) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#mutable_flag _x_i2 in
                      let o = o#virtual_flag _x_i3 in
                      let o = o#ctyp _x_i4 in o
                  | CgVir (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#private_flag _x_i2 in
                      let o = o#ctyp _x_i3 in o
                  | CgAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method class_expr : class_expr -> 'self_type =
                  function
                  | CeNil _x -> let o = o#loc _x in o
                  | CeApp (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_expr _x_i1 in let o = o#expr _x_i2 in o
                  | CeCon (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#virtual_flag _x_i1 in
                      let o = o#ident _x_i2 in let o = o#ctyp _x_i3 in o
                  | CeFun (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#class_expr _x_i2 in o
                  | CeLet (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#rec_flag _x_i1 in
                      let o = o#binding _x_i2 in
                      let o = o#class_expr _x_i3 in o
                  | CeStr (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in
                      let o = o#class_str_item _x_i2 in o
                  | CeTyc (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_expr _x_i1 in
                      let o = o#class_type _x_i2 in o
                  | CeAnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_expr _x_i1 in
                      let o = o#class_expr _x_i2 in o
                  | CeEq (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#class_expr _x_i1 in
                      let o = o#class_expr _x_i2 in o
                  | CeAtt (_x, _x_i1, _x_i2, _x_i3) ->
                      let o = o#loc _x in
                      let o = o#string _x_i1 in
                      let o = o#str_item _x_i2 in
                      let o = o#class_expr _x_i3 in o
                  | CeAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method binding : binding -> 'self_type =
                  function
                  | BiNil _x -> let o = o#loc _x in o
                  | BiAnd (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#binding _x_i1 in let o = o#binding _x_i2 in o
                  | BiEq (_x, _x_i1, _x_i2) ->
                      let o = o#loc _x in
                      let o = o#patt _x_i1 in let o = o#expr _x_i2 in o
                  | BiAnt (_x, _x_i1) ->
                      let o = o#loc _x in let o = o#string _x_i1 in o
                method unknown : 'a. 'a -> 'self_type = fun _ -> o
              end
              
            let map_expr f =
              object
                inherit map as super
                method expr = fun x -> f (super#expr x)
              end
              
            let map_patt f =
              object
                inherit map as super
                method patt = fun x -> f (super#patt x)
              end
              
            let map_ctyp f =
              object
                inherit map as super
                method ctyp = fun x -> f (super#ctyp x)
              end
              
            let map_str_item f =
              object
                inherit map as super
                method str_item = fun x -> f (super#str_item x)
              end
              
            let map_sig_item f =
              object
                inherit map as super
                method sig_item = fun x -> f (super#sig_item x)
              end
              
            let map_loc f =
              object
                inherit map as super
                method loc = fun x -> f (super#loc x)
              end
              
          end
          
      end
      
    module DynAst =
      struct
        module Make (Ast : Sig.Ast) : Sig.DynAst with module Ast = Ast =
          struct
            module Ast = Ast
              
            type 'a tag =
              | Tag_ctyp
              | Tag_patt
              | Tag_expr
              | Tag_module_type
              | Tag_sig_item
              | Tag_with_constr
              | Tag_module_expr
              | Tag_str_item
              | Tag_class_type
              | Tag_class_sig_item
              | Tag_class_expr
              | Tag_class_str_item
              | Tag_match_case
              | Tag_ident
              | Tag_binding
              | Tag_rec_binding
              | Tag_module_binding
            
            let string_of_tag =
              function
              | Tag_ctyp -> "ctyp"
              | Tag_patt -> "patt"
              | Tag_expr -> "expr"
              | Tag_module_type -> "module_type"
              | Tag_sig_item -> "sig_item"
              | Tag_with_constr -> "with_constr"
              | Tag_module_expr -> "module_expr"
              | Tag_str_item -> "str_item"
              | Tag_class_type -> "class_type"
              | Tag_class_sig_item -> "class_sig_item"
              | Tag_class_expr -> "class_expr"
              | Tag_class_str_item -> "class_str_item"
              | Tag_match_case -> "match_case"
              | Tag_ident -> "ident"
              | Tag_binding -> "binding"
              | Tag_rec_binding -> "rec_binding"
              | Tag_module_binding -> "module_binding"
              
            let ctyp_tag = Tag_ctyp
              
            let patt_tag = Tag_patt
              
            let expr_tag = Tag_expr
              
            let module_type_tag = Tag_module_type
              
            let sig_item_tag = Tag_sig_item
              
            let with_constr_tag = Tag_with_constr
              
            let module_expr_tag = Tag_module_expr
              
            let str_item_tag = Tag_str_item
              
            let class_type_tag = Tag_class_type
              
            let class_sig_item_tag = Tag_class_sig_item
              
            let class_expr_tag = Tag_class_expr
              
            let class_str_item_tag = Tag_class_str_item
              
            let match_case_tag = Tag_match_case
              
            let ident_tag = Tag_ident
              
            let binding_tag = Tag_binding
              
            let rec_binding_tag = Tag_rec_binding
              
            let module_binding_tag = Tag_module_binding
              
            type dyn
            
            external dyn_tag : 'a tag -> dyn tag = "%identity"
              
            module Pack (X : sig type 'a t
                                  end) =
              struct
                type pack = ((dyn tag) * Obj.t)
                
                exception Pack_error
                  
                let pack tag v = ((dyn_tag tag), (Obj.repr v))
                  
                let unpack (tag : 'a tag) (tag', obj) =
                  if (dyn_tag tag) = tag'
                  then (Obj.obj obj : 'a X.t)
                  else raise Pack_error
                  
                let print_tag f (tag, _) =
                  Format.pp_print_string f (string_of_tag tag)
                  
              end
              
          end
          
      end
      
    module Quotation =
      struct
        module Make (Ast : Sig.Camlp4Ast) :
          Sig.Quotation with module Ast = Ast =
          struct
            module Ast = Ast
              
            module DynAst = DynAst.Make(Ast)
              
            module Loc = Ast.Loc
              
            open Format
              
            open Sig
              
            type 'a expand_fun = Loc.t -> string option -> string -> 'a
            
            module Exp_key = DynAst.Pack(struct type 'a t = unit
                                                 end)
              
            module Exp_fun =
              DynAst.Pack(struct type 'a t = 'a expand_fun
                                  end)
              
            let expanders_table :
              (((string * Exp_key.pack) * Exp_fun.pack) list) ref = ref []
              
            let default = ref ""
              
            let translate = ref (fun x -> x)
              
            let expander_name name =
              match !translate name with | "" -> !default | name -> name
              
            let find name tag =
              let key = ((expander_name name), (Exp_key.pack tag ()))
              in Exp_fun.unpack tag (List.assoc key !expanders_table)
              
            let add name tag f =
              let elt = ((name, (Exp_key.pack tag ())), (Exp_fun.pack tag f))
              in expanders_table := elt :: !expanders_table
              
            let dump_file = ref None
              
            module Error =
              struct
                type error =
                  | Finding
                  | Expanding
                  | ParsingResult of Loc.t * string
                  | Locating
                
                type t = (string * string * error * exn)
                
                exception E of t
                  
                let print ppf (name, position, ctx, exn) =
                  let name = if name = "" then !default else name in
                  let pp x =
                    fprintf ppf "@?@[<2>While %s %S in a position of %S:" x
                      name position in
                  let () =
                    match ctx with
                    | Finding ->
                        (pp "finding quotation";
                         if !expanders_table = []
                         then
                           fprintf ppf
                             "@ There is no quotation expander available."
                         else
                           (fprintf ppf
                              "@ @[<hv2>Available quotation expanders are:@\n";
                            List.iter
                              (fun ((s, t), _) ->
                                 fprintf ppf
                                   "@[<2>%s@ (in@ a@ position@ of %a)@]@ " s
                                   Exp_key.print_tag t)
                              !expanders_table;
                            fprintf ppf "@]"))
                    | Expanding -> pp "expanding quotation"
                    | Locating -> pp "parsing"
                    | ParsingResult (loc, str) ->
                        let () = pp "parsing result of quotation"
                        in
                          (match !dump_file with
                           | Some dump_file ->
                               let () = fprintf ppf " dumping result...\n"
                               in
                                 (try
                                    let oc = open_out_bin dump_file
                                    in
                                      (output_string oc str;
                                       output_string oc "\n";
                                       flush oc;
                                       close_out oc;
                                       fprintf ppf "%a:" Loc.print
                                         (Loc.set_file_name dump_file loc))
                                  with
                                  | _ ->
                                      fprintf ppf
                                        "Error while dumping result in file %S; dump aborted"
                                        dump_file)
                           | None ->
                               fprintf ppf
                                 "\n(consider setting variable Quotation.dump_file, or using the -QD option)")
                  in fprintf ppf "@\n%a@]@." ErrorHandler.print exn
                  
                let to_string x = Format.asprintf "%a" print x
                  
              end
              
            let _ = let module M = ErrorHandler.Register(Error) in ()
              
            open Error
              
            let expand_quotation loc expander pos_tag quot =
              let loc_name_opt =
                if quot.q_loc = "" then None else Some quot.q_loc
              in
                try expander loc loc_name_opt quot.q_contents
                with | (Loc.Exc_located (_, (Error.E _)) as exc) -> raise exc
                | Loc.Exc_located (iloc, exc) ->
                    let exc1 =
                      Error.E (((quot.q_name), pos_tag, Expanding, exc))
                    in raise (Loc.Exc_located (iloc, exc1))
                | exc ->
                    let exc1 =
                      Error.E (((quot.q_name), pos_tag, Expanding, exc))
                    in raise (Loc.Exc_located (loc, exc1))
              
            let parse_quotation_result parse loc quot pos_tag str =
              try parse loc str
              with
              | Loc.Exc_located (iloc,
                  (Error.E ((n, pos_tag, Expanding, exc)))) ->
                  let ctx = ParsingResult (iloc, quot.q_contents) in
                  let exc1 = Error.E ((n, pos_tag, ctx, exc))
                  in raise (Loc.Exc_located (iloc, exc1))
              | Loc.Exc_located (iloc, ((Error.E _ as exc))) ->
                  raise (Loc.Exc_located (iloc, exc))
              | Loc.Exc_located (iloc, exc) ->
                  let ctx = ParsingResult (iloc, quot.q_contents) in
                  let exc1 = Error.E (((quot.q_name), pos_tag, ctx, exc))
                  in raise (Loc.Exc_located (iloc, exc1))
              
            let expand loc quotation tag =
              let pos_tag = DynAst.string_of_tag tag in
              let name = quotation.q_name in
              let expander =
                try find name tag
                with | (Loc.Exc_located (_, (Error.E _)) as exc) -> raise exc
                | Loc.Exc_located (qloc, exc) ->
                    raise
                      (Loc.Exc_located (qloc,
                         (Error.E ((name, pos_tag, Finding, exc)))))
                | exc ->
                    raise
                      (Loc.Exc_located (loc,
                         (Error.E ((name, pos_tag, Finding, exc))))) in
              let loc = Loc.join (Loc.move `start quotation.q_shift loc)
              in expand_quotation loc expander pos_tag quotation
              
          end
          
      end
      
    module AstFilters =
      struct
        module Make (Ast : Sig.Camlp4Ast) :
          Sig.AstFilters with module Ast = Ast =
          struct
            module Ast = Ast
              
            type 'a filter = 'a -> 'a
            
            let interf_filters = Queue.create ()
              
            let fold_interf_filters f i = Queue.fold f i interf_filters
              
            let implem_filters = Queue.create ()
              
            let fold_implem_filters f i = Queue.fold f i implem_filters
              
            let topphrase_filters = Queue.create ()
              
            let fold_topphrase_filters f i = Queue.fold f i topphrase_filters
              
            let register_sig_item_filter f = Queue.add f interf_filters
              
            let register_str_item_filter f = Queue.add f implem_filters
              
            let register_topphrase_filter f = Queue.add f topphrase_filters
              
          end
          
      end
      
    module Camlp4Ast2OCamlAst :
      sig
        module Make (Camlp4Ast : Sig.Camlp4Ast) :
          sig
            open Camlp4Ast
              
            val sig_item : sig_item -> Parsetree.signature
              
            val str_item : str_item -> Parsetree.structure
              
            val phrase : str_item -> Parsetree.toplevel_phrase
              
          end
          
      end =
      struct
        module Location =
          struct
            type t =
              Location.t =
                { loc_start : Lexing.position; loc_end : Lexing.position;
                  loc_ghost : bool
                }
            
            type 'a loc = ('a Location.loc) = { txt : 'a; loc : t }
            
            let none =
              let loc =
                {
                  Lexing.pos_fname = "_none_";
                  pos_lnum = 1;
                  pos_bol = 0;
                  pos_cnum = (-1);
                }
              in { loc_start = loc; loc_end = loc; loc_ghost = true; }
              
            let mkloc txt loc = { txt = txt; loc = loc; }
              
          end
          
        module Longident =
          struct
            type t =
              Longident.t =
                | Lident of string | Ldot of t * string | Lapply of t * t
            
            let last =
              function
              | Lident s -> s
              | Ldot (_, s) -> s
              | Lapply (_, _) -> failwith "Longident.last"
              
          end
          
        module Make (Ast : Sig.Camlp4Ast) =
          struct
            open Format
              
            open Parsetree
              
            open Longident
              
            open Asttypes
              
            open Ast
              
            let error loc str = Loc.raise loc (Failure str)
              
            let char_of_char_token loc s =
              try Token.Eval.char s
              with | (Failure _ as exn) -> Loc.raise loc exn
              
            let string_of_string_token loc s =
              try Token.Eval.string s
              with | (Failure _ as exn) -> Loc.raise loc exn
              
            let remove_underscores s =
              let s = Bytes.of_string s in
              let l = Bytes.length s in
              let rec remove src dst =
                if src >= l
                then if dst >= l then s else Bytes.sub s 0 dst
                else
                  (match Bytes.get s src with
                   | '_' -> remove (src + 1) dst
                   | c -> (Bytes.set s dst c; remove (src + 1) (dst + 1)))
              in Bytes.to_string (remove 0 0)
              
            let mkloc = Loc.to_ocaml_location
              
            let mkghloc loc = Loc.to_ocaml_location (Loc.ghostify loc)
              
            let with_loc txt loc = Location.mkloc txt (mkloc loc)
              
            let mktyp loc d =
              {
                ptyp_desc = d;
                ptyp_loc = mkloc loc;
                ptyp_attributes = [];
                ptyp_loc_stack = [];
              }
              
            let mkpat loc d =
              {
                ppat_desc = d;
                ppat_loc = mkloc loc;
                ppat_attributes = [];
                ppat_loc_stack = [];
              }
              
            let mkghpat loc d =
              {
                ppat_desc = d;
                ppat_loc = mkghloc loc;
                ppat_attributes = [];
                ppat_loc_stack = [];
              }
              
            let mkexp loc d =
              {
                pexp_desc = d;
                pexp_loc = mkloc loc;
                pexp_attributes = [];
                pexp_loc_stack = [];
              }
              
            let mkmty loc d =
              { pmty_desc = d; pmty_loc = mkloc loc; pmty_attributes = []; }
              
            let mksig loc d = { psig_desc = d; psig_loc = mkloc loc; }
              
            let mkmod loc d =
              { pmod_desc = d; pmod_loc = mkloc loc; pmod_attributes = []; }
              
            let mkstr loc d = { pstr_desc = d; pstr_loc = mkloc loc; }
              
            let mkcty loc d =
              { pcty_desc = d; pcty_loc = mkloc loc; pcty_attributes = []; }
              
            let mkcl loc d =
              { pcl_desc = d; pcl_loc = mkloc loc; pcl_attributes = []; }
              
            let mkcf loc d =
              { pcf_desc = d; pcf_loc = mkloc loc; pcf_attributes = []; }
              
            let mkctf loc d =
              { pctf_desc = d; pctf_loc = mkloc loc; pctf_attributes = []; }
              
            let mkpolytype t =
              match t.ptyp_desc with
              | Ptyp_poly (_, _) -> t
              | _ -> { (t) with ptyp_desc = Ptyp_poly ([], t); }
              
            let mkvirtual =
              function
              | Ast.ViVirtual -> Virtual
              | Ast.ViNil -> Concrete
              | _ -> assert false
              
            let mkdirection =
              function
              | Ast.DiTo -> Upto
              | Ast.DiDownto -> Downto
              | _ -> assert false
              
            let lident s = Lident s
              
            let lident_with_loc s loc = with_loc (Lident s) loc
              
            let ldot l s = Ldot (l, s)
              
            let lapply l s = Lapply (l, s)
              
            let conv_con =
              let t = Hashtbl.create 73
              in
                (List.iter (fun (s, s') -> Hashtbl.add t s s')
                   [ ("True", "true"); ("False", "false"); (" True", "True");
                     (" False", "False") ];
                 fun s -> try Hashtbl.find t s with | Not_found -> s)
              
            let conv_lab =
              let t = Hashtbl.create 73
              in
                (List.iter (fun (s, s') -> Hashtbl.add t s s')
                   [ ("val", "contents") ];
                 fun s -> try Hashtbl.find t s with | Not_found -> s)
              
            let array_function_no_loc str name =
              ldot (lident str)
                (if !Camlp4_config.unsafe then "unsafe_" ^ name else name)
              
            let array_function loc str name =
              with_loc (array_function_no_loc str name) loc
              
            let mkrf =
              function
              | Ast.ReRecursive -> Recursive
              | Ast.ReNonrecursive | Ast.ReNil -> Nonrecursive
              | _ -> assert false
              
            let mknrf =
              function
              | Ast.ReNonrecursive -> Nonrecursive
              | Ast.ReRecursive | Ast.ReNil -> Recursive
              | _ -> assert false
              
            let mkli sloc s list =
              let rec loop f =
                function | i :: il -> loop (ldot (f i)) il | [] -> f s
              in with_loc (loop lident list) sloc
              
            let rec ctyp_fa al =
              function
              | TyApp (_, f, a) -> ctyp_fa (a :: al) f
              | f -> (f, al)
              
            let ident_tag ?(conv_lid = fun x -> x) i =
              let rec self i acc =
                match i with
                | Ast.IdAcc (_, i1, i2) -> self i2 (Some (self i1 acc))
                | Ast.IdApp (_, i1, i2) ->
                    let i' =
                      Lapply ((fst (self i1 None)), (fst (self i2 None))) in
                    let x =
                      (match acc with
                       | None -> i'
                       | _ ->
                           error (loc_of_ident i) "invalid long identifier")
                    in (x, `app)
                | Ast.IdUid (_, s) ->
                    let x =
                      (match acc with
                       | None -> lident s
                       | Some ((acc, (`uident | `app))) -> ldot acc s
                       | _ ->
                           error (loc_of_ident i) "invalid long identifier")
                    in (x, `uident)
                | Ast.IdLid (_, s) ->
                    let x =
                      (match acc with
                       | None -> lident (conv_lid s)
                       | Some ((acc, (`uident | `app))) ->
                           ldot acc (conv_lid s)
                       | _ ->
                           error (loc_of_ident i) "invalid long identifier")
                    in (x, `lident)
                | _ -> error (loc_of_ident i) "invalid long identifier"
              in self i None
              
            let ident_noloc ?conv_lid i = fst (ident_tag ?conv_lid i)
              
            let ident ?conv_lid i =
              with_loc (ident_noloc ?conv_lid i) (loc_of_ident i)
              
            let long_lident msg id =
              match ident_tag id with
              | (i, `lident) -> with_loc i (loc_of_ident id)
              | _ -> error (loc_of_ident id) msg
              
            let long_type_ident = long_lident "invalid long identifier type"
              
            let long_class_ident = long_lident "invalid class name"
              
            let long_uident_noloc ?(conv_con = fun x -> x) i =
              match ident_tag i with
              | (Ldot (i, s), `uident) -> ldot i (conv_con s)
              | (Lident s, `uident) -> lident (conv_con s)
              | (i, `app) -> i
              | _ -> error (loc_of_ident i) "uppercase identifier expected"
              
            let long_uident ?conv_con i =
              with_loc (long_uident_noloc ?conv_con i) (loc_of_ident i)
              
            let rec ctyp_long_id_prefix t =
              match t with
              | Ast.TyId (_, i) -> ident_noloc i
              | Ast.TyApp (_, m1, m2) ->
                  let li1 = ctyp_long_id_prefix m1 in
                  let li2 = ctyp_long_id_prefix m2 in Lapply (li1, li2)
              | t -> error (loc_of_ctyp t) "invalid module expression"
              
            let ctyp_long_id t =
              match t with
              | Ast.TyId (_, i) -> (false, (long_type_ident i))
              | TyApp (loc, _, _) -> error loc "invalid type name"
              | TyCls (_, i) -> (true, (ident i))
              | t -> error (loc_of_ctyp t) "invalid type"
              
            let rec ty_var_list_of_ctyp =
              function
              | Ast.TyApp (_, t1, t2) ->
                  (ty_var_list_of_ctyp t1) @ (ty_var_list_of_ctyp t2)
              | Ast.TyQuo (loc, s) -> [ with_loc s loc ]
              | _ -> assert false
              
            let attribute_fwd = ref (fun _ _ _ -> assert false)
              
            let attribute loc s str = !attribute_fwd loc s str
              
            let rec ctyp =
              function
              | TyId (loc, i) ->
                  let li = long_type_ident i
                  in mktyp loc (Ptyp_constr (li, []))
              | TyAli (loc, t1, t2) ->
                  let (t, i) =
                    (match (t1, t2) with
                     | (t, TyQuo (_, s)) -> (t, s)
                     | (TyQuo (_, s), t) -> (t, s)
                     | _ -> error loc "invalid alias type")
                  in mktyp loc (Ptyp_alias ((ctyp t), i))
              | TyAny loc -> mktyp loc Ptyp_any
              | (TyApp (loc, _, _) as f) ->
                  let (f, al) = ctyp_fa [] f in
                  let (is_cls, li) = ctyp_long_id f
                  in
                    if is_cls
                    then mktyp loc (Ptyp_class (li, (List.map ctyp al)))
                    else mktyp loc (Ptyp_constr (li, (List.map ctyp al)))
              | TyArr (loc, (TyLab (_, lab, t1)), t2) ->
                  mktyp loc
                    (Ptyp_arrow ((Labelled lab), (ctyp t1), (ctyp t2)))
              | TyArr (loc, (TyOlb (_, lab, t1)), t2) ->
                  mktyp loc
                    (Ptyp_arrow ((Optional lab), (ctyp t1), (ctyp t2)))
              | TyArr (loc, t1, t2) ->
                  mktyp loc (Ptyp_arrow (Nolabel, (ctyp t1), (ctyp t2)))
              | Ast.TyObj (loc, fl, Ast.RvNil) ->
                  mktyp loc (Ptyp_object ((meth_list fl []), Closed))
              | Ast.TyObj (loc, fl, Ast.RvRowVar) ->
                  mktyp loc (Ptyp_object ((meth_list fl []), Open))
              | TyCls (loc, id) -> mktyp loc (Ptyp_class ((ident id), []))
              | Ast.TyPkg (loc, pt) ->
                  let (i, cs) = package_type pt
                  in mktyp loc (Ptyp_package (i, cs))
              | TyAtt (loc, s, str, e) ->
                  let e = ctyp e
                  in
                    {
                      (e)
                      with
                      ptyp_attributes =
                        e.ptyp_attributes @ [ attribute loc s str ];
                    }
              | TyLab (loc, _, _) ->
                  error loc "labelled type not allowed here"
              | TyMan (loc, _, _) ->
                  error loc "manifest type not allowed here"
              | TyOlb (loc, _, _) ->
                  error loc "labelled type not allowed here"
              | TyPol (loc, t1, t2) ->
                  mktyp loc (Ptyp_poly ((ty_var_list_of_ctyp t1), (ctyp t2)))
              | TyQuo (loc, s) -> mktyp loc (Ptyp_var s)
              | TyRec (loc, _) -> error loc "record type not allowed here"
              | TySum (loc, _) -> error loc "sum type not allowed here"
              | TyPrv (loc, _) -> error loc "private type not allowed here"
              | TyMut (loc, _) -> error loc "mutable type not allowed here"
              | TyOr (loc, _, _) ->
                  error loc "type1 | type2 not allowed here"
              | TyAnd (loc, _, _) ->
                  error loc "type1 and type2 not allowed here"
              | TyOf (loc, _, _) ->
                  error loc "type1 of type2 not allowed here"
              | TyCol (loc, _, _) ->
                  error loc "type1 : type2 not allowed here"
              | TySem (loc, _, _) ->
                  error loc "type1 ; type2 not allowed here"
              | TyTypePol (loc, _, _) ->
                  error loc "locally abstract type not allowed here"
              | Ast.TyTup (loc, (Ast.TySta (_, t1, t2))) ->
                  mktyp loc
                    (Ptyp_tuple
                       (List.map ctyp (list_of_ctyp t1 (list_of_ctyp t2 []))))
              | Ast.TyVrnEq (loc, t) ->
                  mktyp loc (Ptyp_variant ((row_field t), Closed, None))
              | Ast.TyVrnSup (loc, t) ->
                  mktyp loc (Ptyp_variant ((row_field t), Open, None))
              | Ast.TyVrnInf (loc, t) ->
                  mktyp loc (Ptyp_variant ((row_field t), Closed, (Some [])))
              | Ast.TyVrnInfSup (loc, t, t') ->
                  mktyp loc
                    (Ptyp_variant ((row_field t), Closed,
                       (Some (name_tags t'))))
              | TyAnt (loc, _) -> error loc "antiquotation not allowed here"
              | TyOfAmp (loc, _, _) | TyAmp (loc, _, _) | TySta (loc, _, _) |
                  TyCom (loc, _, _) | TyVrn (loc, _) | TyQuM (loc, _) |
                  TyQuP (loc, _) | TyDcl (loc, _, _, _, _) |
                  TyExt (loc, _, _, _) | TyAnP loc | TyAnM loc |
                  TyObj (loc, _, (RvAnt _)) | TyNil loc | TyOpn loc |
                  TyTup (loc, _) ->
                  error loc "this construction is not allowed here"
            and row_field =
              let mk loc x =
                { prf_loc = mkloc loc; prf_desc = x; prf_attributes = []; }
              in
                function
                | Ast.TyNil _ -> []
                | Ast.TyVrn (loc, i) ->
                    [ mk loc (Rtag ((with_loc (conv_con i) loc), true, [])) ]
                | Ast.TyOfAmp (loc, (Ast.TyVrn (_, i)), t) ->
                    [ mk loc
                        (Rtag ((with_loc (conv_con i) loc), true,
                           (List.map ctyp (list_of_ctyp t [])))) ]
                | Ast.TyOf (loc, (Ast.TyVrn (_, i)), t) ->
                    [ mk loc
                        (Rtag ((with_loc (conv_con i) loc), false,
                           (List.map ctyp (list_of_ctyp t [])))) ]
                | Ast.TyOr (_, t1, t2) -> (row_field t1) @ (row_field t2)
                | t -> [ mk (Ast.loc_of_ctyp t) (Rinherit (ctyp t)) ]
            and name_tags =
              function
              | Ast.TyApp (_, t1, t2) -> (name_tags t1) @ (name_tags t2)
              | Ast.TyVrn (_, s) -> [ s ]
              | _ -> assert false
            and meth_list fl acc =
              match fl with
              | Ast.TyNil _ -> acc
              | Ast.TySem (_, t1, t2) -> meth_list t1 (meth_list t2 acc)
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdLid (_, lab)))), t) ->
                  {
                    pof_loc = mkloc loc;
                    pof_desc =
                      Otag ((with_loc lab loc), (mkpolytype (ctyp t)));
                    pof_attributes = [];
                  } :: acc
              | _ -> assert false
            and package_type_constraints wc acc =
              match wc with
              | Ast.WcNil _ -> acc
              | Ast.WcTyp (_, (Ast.TyId (_, id)), ct) ->
                  ((ident id), (ctyp ct)) :: acc
              | Ast.WcAnd (_, wc1, wc2) ->
                  package_type_constraints wc1
                    (package_type_constraints wc2 acc)
              | _ ->
                  error (loc_of_with_constr wc)
                    "unexpected `with constraint' for a package type"
            and package_type : module_type -> package_type =
              function
              | Ast.MtWit (_, (Ast.MtId (_, i)), wc) ->
                  ((long_uident i), (package_type_constraints wc []))
              | Ast.MtId (_, i) -> ((long_uident i), [])
              | mt -> error (loc_of_module_type mt) "unexpected package type"
              
            let mktype loc name tl cl tk tp tm =
              {
                ptype_name = name;
                ptype_params = tl;
                ptype_cstrs = cl;
                ptype_kind = tk;
                ptype_private = tp;
                ptype_manifest = tm;
                ptype_loc = mkloc loc;
                ptype_attributes = [];
              }
              
            let mktypext loc path tl tc tp =
              {
                ptyext_loc = loc;
                ptyext_path = path;
                ptyext_params = tl;
                ptyext_constructors = tc;
                ptyext_private = tp;
                ptyext_attributes = [];
              }
              
            let mkprivate' m = if m then Private else Public
              
            let mkprivate =
              function
              | Ast.PrPrivate -> Private
              | Ast.PrNil -> Public
              | _ -> assert false
              
            let mktrecord =
              function
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdLid (sloc, s)))),
                  (Ast.TyMut (_, t))) ->
                  {
                    pld_name = with_loc s sloc;
                    pld_mutable = Mutable;
                    pld_type = mkpolytype (ctyp t);
                    pld_loc = mkloc loc;
                    pld_attributes = [];
                  }
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdLid (sloc, s)))), t) ->
                  {
                    pld_name = with_loc s sloc;
                    pld_mutable = Immutable;
                    pld_type = mkpolytype (ctyp t);
                    pld_loc = mkloc loc;
                    pld_attributes = [];
                  }
              | _ -> assert false
              
            let mkvariant =
              function
              | Ast.TyId (loc, (Ast.IdUid (sloc, s))) ->
                  {
                    pcd_name = with_loc (conv_con s) sloc;
                    pcd_args = Pcstr_tuple [];
                    pcd_res = None;
                    pcd_loc = mkloc loc;
                    pcd_attributes = [];
                  }
              | Ast.TyOf (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))), t) ->
                  {
                    pcd_name = with_loc (conv_con s) sloc;
                    pcd_args =
                      Pcstr_tuple (List.map ctyp (list_of_ctyp t []));
                    pcd_res = None;
                    pcd_loc = mkloc loc;
                    pcd_attributes = [];
                  }
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))),
                  (Ast.TyArr (_, t, u))) ->
                  {
                    pcd_name = with_loc (conv_con s) sloc;
                    pcd_args =
                      Pcstr_tuple (List.map ctyp (list_of_ctyp t []));
                    pcd_res = Some (ctyp u);
                    pcd_loc = mkloc loc;
                    pcd_attributes = [];
                  }
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))), t) ->
                  {
                    pcd_name = with_loc (conv_con s) sloc;
                    pcd_args = Pcstr_tuple [];
                    pcd_res = Some (ctyp t);
                    pcd_loc = mkloc loc;
                    pcd_attributes = [];
                  }
              | _ -> assert false
              
            let mkextension_constructor =
              function
              | Ast.TyId (loc, (Ast.IdUid (sloc, s))) ->
                  {
                    pext_name = with_loc (conv_con s) sloc;
                    pext_kind = Pext_decl ((Pcstr_tuple []), None);
                    pext_loc = mkloc loc;
                    pext_attributes = [];
                  }
              | Ast.TyOf (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))), t) ->
                  {
                    pext_name = with_loc (conv_con s) sloc;
                    pext_kind =
                      Pext_decl
                        ((Pcstr_tuple (List.map ctyp (list_of_ctyp t []))),
                        None);
                    pext_loc = mkloc loc;
                    pext_attributes = [];
                  }
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))),
                  (Ast.TyArr (_, t, u))) ->
                  {
                    pext_name = with_loc (conv_con s) sloc;
                    pext_kind =
                      Pext_decl
                        ((Pcstr_tuple (List.map ctyp (list_of_ctyp t []))),
                        (Some (ctyp u)));
                    pext_loc = mkloc loc;
                    pext_attributes = [];
                  }
              | Ast.TyCol (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))), t) ->
                  {
                    pext_name = with_loc (conv_con s) sloc;
                    pext_kind = Pext_decl ((Pcstr_tuple []), (Some (ctyp t)));
                    pext_loc = mkloc loc;
                    pext_attributes = [];
                  }
              | Ast.TyMan (loc, (Ast.TyId (_, (Ast.IdUid (sloc, s)))),
                  (Ast.TyId (_, r))) ->
                  {
                    pext_name = with_loc (conv_con s) sloc;
                    pext_kind = Pext_rebind (long_uident r);
                    pext_loc = mkloc loc;
                    pext_attributes = [];
                  }
              | _ -> assert false
              
            let rec type_decl name tl cl loc m pflag =
              function
              | Ast.TyMan (_, t1, t2) ->
                  type_decl name tl cl loc (Some (ctyp t1)) pflag t2
              | Ast.TyPrv (_loc, t) ->
                  if pflag
                  then
                    error _loc
                      "multiple private keyword used, use only one instead"
                  else type_decl name tl cl loc m true t
              | Ast.TyRec (_, t) ->
                  mktype loc name tl cl
                    (Ptype_record (List.map mktrecord (list_of_ctyp t [])))
                    (mkprivate' pflag) m
              | Ast.TySum (_, t) ->
                  mktype loc name tl cl
                    (Ptype_variant (List.map mkvariant (list_of_ctyp t [])))
                    (mkprivate' pflag) m
              | TyOpn loc ->
                  mktype loc name tl cl Ptype_open (mkprivate' pflag) m
              | t ->
                  if m <> None
                  then
                    error loc "only one manifest type allowed by definition"
                  else
                    (let m =
                       match t with
                       | Ast.TyNil _ -> None
                       | _ -> Some (ctyp t)
                     in
                       mktype loc name tl cl Ptype_abstract
                         (mkprivate' pflag) m)
              
            let rec type_ext path tl loc pflag =
              function
              | Ast.TyMan (_loc, _, _) ->
                  error _loc "manifest type not allowed for extensions"
              | Ast.TyPrv (_loc, t) ->
                  if pflag
                  then
                    error _loc
                      "multiple private keyword used, use only one instead"
                  else type_ext path tl loc true t
              | Ast.TySum (_, t) ->
                  mktypext (mkloc loc) path tl
                    (List.map mkextension_constructor (list_of_ctyp t []))
                    (mkprivate' pflag)
              | _ -> error loc "invalid type extension"
              
            let type_decl name tl cl t loc =
              type_decl name tl cl loc None false t
              
            let type_ext path tl t loc = type_ext path tl loc false t
              
            let mkvalue_desc loc name t p =
              {
                pval_name = name;
                pval_type = ctyp t;
                pval_prim = p;
                pval_loc = mkloc loc;
                pval_attributes = [];
              }
              
            let rec list_of_meta_list =
              function
              | Ast.LNil -> []
              | Ast.LCons (x, xs) -> x :: (list_of_meta_list xs)
              | Ast.LAnt _ -> assert false
              
            let mkmutable =
              function
              | Ast.MuMutable -> Mutable
              | Ast.MuNil -> Immutable
              | _ -> assert false
              
            let paolab lab p =
              match (lab, p) with
              | ("",
                 (Ast.PaId (_, (Ast.IdLid (_, i))) |
                    Ast.PaTyc (_, (Ast.PaId (_, (Ast.IdLid (_, i)))), _)))
                  -> i
              | ("", p) -> error (loc_of_patt p) "bad ast in label"
              | _ -> lab
              
            let opt_private_ctyp =
              function
              | Ast.TyPrv (_, t) -> (Ptype_abstract, Private, (ctyp t))
              | t -> (Ptype_abstract, Public, (ctyp t))
              
            let rec type_parameters t acc =
              match t with
              | Ast.TyApp (_, t1, t2) ->
                  type_parameters t1 (type_parameters t2 acc)
              | Ast.TyQuP (_, s) -> (s, (Covariant, NoInjectivity)) :: acc
              | Ast.TyQuM (_, s) -> (s, (Contravariant, NoInjectivity)) :: acc
              | Ast.TyQuo (_, s) -> (s, (NoVariance, NoInjectivity)) :: acc
              | _ -> assert false
              
            let core_type loc ty =
              {
                ptyp_desc = ty;
                ptyp_loc = mkloc loc;
                ptyp_attributes = [];
                ptyp_loc_stack = [];
              }
              
            let ptyp_var loc s = core_type loc (Ptyp_var s)
              
            let ptyp_any loc = core_type loc Ptyp_any
              
            let rec optional_type_parameters t acc =
              match t with
              | Ast.TyApp (_, t1, t2) ->
                  optional_type_parameters t1
                    (optional_type_parameters t2 acc)
              | Ast.TyQuP (loc, s) -> ((ptyp_var loc s), (Covariant, NoInjectivity)) :: acc
              | Ast.TyAnP loc -> ((ptyp_any loc), (Covariant, NoInjectivity)) :: acc
              | Ast.TyQuM (loc, s) ->
                  ((ptyp_var loc s), (Contravariant, NoInjectivity)) :: acc
              | Ast.TyAnM loc -> ((ptyp_any loc), (Contravariant, NoInjectivity)) :: acc
              | Ast.TyQuo (loc, s) -> ((ptyp_var loc s), (NoVariance, NoInjectivity)) :: acc
              | Ast.TyAny loc -> ((ptyp_any loc), (NoVariance, NoInjectivity)) :: acc
              | _ -> assert false
              
            let rec class_parameters t acc =
              match t with
              | Ast.TyCom (_, t1, t2) ->
                  class_parameters t1 (class_parameters t2 acc)
              | Ast.TyQuP (loc, s) -> ((ptyp_var loc s), (Covariant, NoInjectivity)) :: acc
              | Ast.TyQuM (loc, s) ->
                  ((ptyp_var loc s), (Contravariant, NoInjectivity)) :: acc
              | Ast.TyQuo (loc, s) -> ((ptyp_var loc s), (NoVariance, NoInjectivity)) :: acc
              | _ -> assert false
              
            let rec type_parameters_and_type_name t acc =
              match t with
              | Ast.TyApp (_, t1, t2) ->
                  type_parameters_and_type_name t1
                    (optional_type_parameters t2 acc)
              | Ast.TyId (_, i) -> ((ident i), acc)
              | _ -> assert false
              
            let mkwithtyp pwith_type loc id_tpl ct =
              let (id, tpl) = type_parameters_and_type_name id_tpl [] in
              let (kind, priv, ct) = opt_private_ctyp ct
              in
                pwith_type id
                  {
                    ptype_name =
                      Location.mkloc (Longident.last id.txt) id.loc;
                    ptype_params = tpl;
                    ptype_cstrs = [];
                    ptype_kind = kind;
                    ptype_private = priv;
                    ptype_manifest = Some ct;
                    ptype_loc = mkloc loc;
                    ptype_attributes = [];
                  }
              
            let rec mkwithc wc acc =
              match wc with
              | Ast.WcNil _ -> acc
              | Ast.WcTyp (loc, id_tpl, ct) ->
                  (mkwithtyp (fun lid x -> Pwith_type (lid, x)) loc id_tpl ct) ::
                    acc
              | Ast.WcMod (_, i1, i2) ->
                  (Pwith_module ((long_uident i1), (long_uident i2))) :: acc
              | Ast.WcTyS (loc, id_tpl, ct) ->
                  (mkwithtyp (fun lid x -> Pwith_typesubst (lid, x)) loc
                     id_tpl ct) ::
                    acc
              | Ast.WcMoS (_, i1, i2) ->
                  (Pwith_modsubst ((long_uident i1), (long_uident i2))) ::
                    acc
              | Ast.WcAnd (_, wc1, wc2) -> mkwithc wc1 (mkwithc wc2 acc)
              | Ast.WcAnt (loc, _) ->
                  error loc "bad with constraint (antiquotation)"
              
            let rec patt_fa al =
              function
              | PaApp (_, f, a) -> patt_fa (a :: al) f
              | f -> (f, al)
              
            let rec deep_mkrangepat loc c1 c2 =
              if c1 = c2
              then mkghpat loc (Ppat_constant (Pconst_char c1))
              else
                mkghpat loc
                  (Ppat_or ((mkghpat loc (Ppat_constant (Pconst_char c1))),
                     (deep_mkrangepat loc (Char.chr ((Char.code c1) + 1)) c2)))
              
            let rec mkrangepat loc c1 c2 =
              if c1 > c2
              then mkrangepat loc c2 c1
              else
                if c1 = c2
                then mkpat loc (Ppat_constant (Pconst_char c1))
                else
                  mkpat loc
                    (Ppat_or ((mkghpat loc (Ppat_constant (Pconst_char c1))),
                       (deep_mkrangepat loc (Char.chr ((Char.code c1) + 1))
                          c2)))
              
            let rec patt =
              function
              | Ast.PaId (loc, (Ast.IdLid (sloc, s))) ->
                  mkpat loc (Ppat_var (with_loc s sloc))
              | Ast.PaId (loc, i) ->
                  let p = Ppat_construct ((long_uident ~conv_con i), None)
                  in mkpat loc p
              | PaAli (loc, p1, p2) ->
                  let (p, i) =
                    (match (p1, p2) with
                     | (p, Ast.PaId (_, (Ast.IdLid (sloc, s)))) ->
                         (p, (with_loc s sloc))
                     | (Ast.PaId (_, (Ast.IdLid (sloc, s))), p) ->
                         (p, (with_loc s sloc))
                     | _ -> error loc "invalid alias pattern")
                  in mkpat loc (Ppat_alias ((patt p), i))
              | PaAnt (loc, _) -> error loc "antiquotation not allowed here"
              | PaAny loc -> mkpat loc Ppat_any
              | Ast.PaApp (loc, (Ast.PaId (_, (Ast.IdUid (sloc, s)))),
                  (Ast.PaTup (_, (Ast.PaAny loc_any)))) ->
                  mkpat loc
                    (Ppat_construct ((lident_with_loc (conv_con s) sloc),
                       (Some ([], mkpat loc_any Ppat_any))))
              | (PaApp (loc, _, _) as f) ->
                  let (f, al) = patt_fa [] f in
                  let al = List.map patt al
                  in
                    (match (patt f).ppat_desc with
                     | Ppat_construct (li, None) ->
                         let a =
                           (match al with
                            | [ a ] -> a
                            | _ -> mkpat loc (Ppat_tuple al))
                         in mkpat loc (Ppat_construct (li, (Some ([], a))))
                     | Ppat_variant (s, None) ->
                         let a =
                           (match al with
                            | [ a ] -> a
                            | _ -> mkpat loc (Ppat_tuple al))
                         in mkpat loc (Ppat_variant (s, (Some a)))
                     | _ ->
                         error (loc_of_patt f)
                           "this is not a constructor, it cannot be applied in a pattern")
              | PaArr (loc, p) ->
                  mkpat loc (Ppat_array (List.map patt (list_of_patt p [])))
              | PaChr (loc, s) ->
                  mkpat loc
                    (Ppat_constant (Pconst_char (char_of_char_token loc s)))
              | PaInt (loc, s) ->
                  mkpat loc (Ppat_constant (Pconst_integer ((s, None))))
              | PaInt32 (loc, s) ->
                  mkpat loc
                    (Ppat_constant (Pconst_integer ((s, (Some 'l')))))
              | PaInt64 (loc, s) ->
                  mkpat loc
                    (Ppat_constant (Pconst_integer ((s, (Some 'L')))))
              | PaNativeInt (loc, s) ->
                  mkpat loc
                    (Ppat_constant (Pconst_integer ((s, (Some 'n')))))
              | PaFlo (loc, s) ->
                  mkpat loc
                    (Ppat_constant
                       (Pconst_float (((remove_underscores s), None))))
              | PaLab (loc, _, _) ->
                  error loc "labeled pattern not allowed here"
              | PaOlb (loc, _, _) | PaOlbi (loc, _, _, _) ->
                  error loc "labeled pattern not allowed here"
              | PaOrp (loc, p1, p2) ->
                  mkpat loc (Ppat_or ((patt p1), (patt p2)))
              | PaRng (loc, p1, p2) ->
                  (match (p1, p2) with
                   | (PaChr (loc1, c1), PaChr (loc2, c2)) ->
                       let c1 = char_of_char_token loc1 c1 in
                       let c2 = char_of_char_token loc2 c2
                       in mkrangepat loc c1 c2
                   | _ ->
                       error loc "range pattern allowed only for characters")
              | PaRec (loc, p) ->
                  let ps = list_of_patt p [] in
                  let is_wildcard =
                    (function | Ast.PaAny _ -> true | _ -> false) in
                  let (wildcards, ps) = List.partition is_wildcard ps in
                  let is_closed = if wildcards = [] then Closed else Open
                  in
                    mkpat loc
                      (Ppat_record (((List.map mklabpat ps), is_closed)))
              | PaStr (loc, s) ->
                  mkpat loc
                    (Ppat_constant
                       (Pconst_string ((string_of_string_token loc s), Loc.to_ocaml_location loc, None)))
              | Ast.PaTup (loc, (Ast.PaCom (_, p1, p2))) ->
                  mkpat loc
                    (Ppat_tuple
                       (List.map patt (list_of_patt p1 (list_of_patt p2 []))))
              | Ast.PaTup (loc, _) -> error loc "singleton tuple pattern"
              | PaTyc (loc, p, t) ->
                  mkpat loc (Ppat_constraint ((patt p), (ctyp t)))
              | PaTyp (loc, i) -> mkpat loc (Ppat_type (long_type_ident i))
              | PaVrn (loc, s) ->
                  mkpat loc (Ppat_variant ((conv_con s), None))
              | PaLaz (loc, p) -> mkpat loc (Ppat_lazy (patt p))
              | PaMod (loc, m) -> mkpat loc (Ppat_unpack (with_loc (Some m) loc))
              | PaExc (loc, p) -> mkpat loc (Ppat_exception (patt p))
              | PaAtt (loc, s, str, e) ->
                  let e = patt e
                  in
                    {
                      (e)
                      with
                      ppat_attributes =
                        e.ppat_attributes @ [ attribute loc s str ];
                    }
              | (PaEq (_, _, _) | PaSem (_, _, _) | PaCom (_, _, _) | PaNil _
                 as p) -> error (loc_of_patt p) "invalid pattern"
            and mklabpat =
              function
              | Ast.PaEq (_, i, p) ->
                  ((ident ~conv_lid: conv_lab i), (patt p))
              | p -> error (loc_of_patt p) "invalid pattern"
              
            let rec expr_fa al =
              function
              | ExApp (_, f, a) -> expr_fa (a :: al) f
              | f -> (f, al)
              
            let rec class_expr_fa al =
              function
              | CeApp (_, ce, a) -> class_expr_fa (a :: al) ce
              | ce -> (ce, al)
              
            let rec sep_expr_acc l =
              function
              | ExAcc (_, e1, e2) -> sep_expr_acc (sep_expr_acc l e2) e1
              | (Ast.ExId (loc, (Ast.IdUid (_, s))) as e) ->
                  (match l with
                   | [] -> [ (loc, [], e) ]
                   | (loc', sl, e) :: l ->
                       ((Loc.merge loc loc'), (s :: sl), e) :: l)
              | Ast.ExId (_, ((Ast.IdAcc (_, _, _) as i))) ->
                  let rec normalize_acc =
                    (function
                     | Ast.IdAcc (_loc, i1, i2) ->
                         Ast.ExAcc (_loc, (normalize_acc i1),
                           (normalize_acc i2))
                     | Ast.IdApp (_loc, i1, i2) ->
                         Ast.ExApp (_loc, (normalize_acc i1),
                           (normalize_acc i2))
                     | (Ast.IdAnt (_loc, _) | Ast.IdUid (_loc, _) |
                          Ast.IdLid (_loc, _)
                        as i) -> Ast.ExId (_loc, i))
                  in sep_expr_acc l (normalize_acc i)
              | e -> ((loc_of_expr e), [], e) :: l
              
            let override_flag loc =
              function
              | Ast.OvOverride -> Override
              | Ast.OvNil -> Fresh
              | _ -> error loc "antiquotation not allowed here"
              
            let list_of_opt_ctyp ot acc =
              match ot with | Ast.TyNil _ -> acc | t -> list_of_ctyp t acc
              
            let varify_constructors var_names =
              let rec loop t =
                let desc =
                  match t.ptyp_desc with
                  | Ptyp_any -> Ptyp_any
                  | Ptyp_var x -> Ptyp_var x
                  | Ptyp_arrow (label, core_type, core_type') ->
                      Ptyp_arrow (label, (loop core_type), (loop core_type'))
                  | Ptyp_tuple lst -> Ptyp_tuple (List.map loop lst)
                  | Ptyp_constr ({ txt = Lident s }, []) when
                      List.exists (fun x -> s = x.txt) var_names ->
                      Ptyp_var ("&" ^ s)
                  | Ptyp_constr (longident, lst) ->
                      Ptyp_constr (longident, (List.map loop lst))
                  | Ptyp_object ((lst, o)) ->
                      Ptyp_object (((List.map loop_object_field lst), o))
                  | Ptyp_class (longident, lst) ->
                      Ptyp_class ((longident, (List.map loop lst)))
                  | Ptyp_alias (core_type, string) ->
                      Ptyp_alias (((loop core_type), string))
                  | Ptyp_variant (row_field_list, flag, lbl_lst_option) ->
                      Ptyp_variant
                        (((List.map loop_row_field row_field_list), flag,
                          lbl_lst_option))
                  | Ptyp_poly (string_lst, core_type) ->
                      Ptyp_poly ((string_lst, (loop core_type)))
                  | Ptyp_package (longident, lst) ->
                      Ptyp_package
                        ((longident,
                          (List.map (fun (n, typ) -> (n, (loop typ))) lst)))
                  | Ptyp_extension x -> Ptyp_extension x
                in { (t) with ptyp_desc = desc; }
              and loop_object_field x =
                let pof_desc =
                  match x.pof_desc with
                  | Otag (s, t) -> Otag (s, (loop t))
                  | Oinherit t -> Oinherit (loop t)
                in { (x) with pof_desc = pof_desc; }
              and loop_row_field x =
                let prf_desc =
                  match x.prf_desc with
                  | Rtag ((label, flag, lst)) ->
                      Rtag ((label, flag, (List.map loop lst)))
                  | Rinherit t -> Rinherit (loop t)
                in { (x) with prf_desc = prf_desc; }
              in loop
              
            let rec expr =
              function
              | Ast.ExAcc (loc, x, (Ast.ExId (_, (Ast.IdLid (_, "val"))))) ->
                  mkexp loc
                    (Pexp_apply
                       ((mkexp loc (Pexp_ident (lident_with_loc "!" loc))),
                       [ (Nolabel, (expr x)) ]))
              | (ExAcc (loc, _, _) | Ast.ExId (loc, (Ast.IdAcc (_, _, _))) as
                 e) ->
                  let (e, l) =
                    (match sep_expr_acc [] e with
                     | (loc, ml, Ast.ExId (_, (Ast.IdUid (_, s)))) :: l ->
                         ((mkexp loc
                             (Pexp_construct ((mkli loc (conv_con s) ml),
                                None))),
                          l)
                     | (loc, ml, Ast.ExId (_, (Ast.IdLid (_, s)))) :: l ->
                         ((mkexp loc (Pexp_ident (mkli loc s ml))), l)
                     | (_, [], e) :: l -> ((expr e), l)
                     | _ -> error loc "bad ast in expression") in
                  let (_, e) =
                    List.fold_left
                      (fun (loc_bp, e1) (loc_ep, ml, e2) ->
                         match e2 with
                         | Ast.ExId (sloc, (Ast.IdLid (_, s))) ->
                             let loc = Loc.merge loc_bp loc_ep
                             in
                               (loc,
                                (mkexp loc
                                   (Pexp_field (e1,
                                      (mkli sloc (conv_lab s) ml)))))
                         | _ ->
                             error (loc_of_expr e2)
                               "lowercase identifier expected")
                      (loc, e) l
                  in e
              | ExAnt (loc, _) -> error loc "antiquotation not allowed here"
              | (ExApp (loc, _, _) as f) ->
                  let (f, al) = expr_fa [] f in
                  let al = List.map label_expr al
                  in
                    (match (expr f).pexp_desc with
                     | Pexp_construct (li, None) ->
                         let al = List.map snd al in
                         let a =
                           (match al with
                            | [ a ] -> a
                            | _ -> mkexp loc (Pexp_tuple al))
                         in mkexp loc (Pexp_construct (li, (Some a)))
                     | Pexp_variant (s, None) ->
                         let al = List.map snd al in
                         let a =
                           (match al with
                            | [ a ] -> a
                            | _ -> mkexp loc (Pexp_tuple al))
                         in mkexp loc (Pexp_variant (s, (Some a)))
                     | _ -> mkexp loc (Pexp_apply ((expr f), al)))
              | ExAre (loc, e1, e2) ->
                  mkexp loc
                    (Pexp_apply
                       ((mkexp loc
                           (Pexp_ident (array_function loc "Array" "get"))),
                       [ (Nolabel, (expr e1)); (Nolabel, (expr e2)) ]))
              | ExArr (loc, e) ->
                  mkexp loc (Pexp_array (List.map expr (list_of_expr e [])))
              | ExAsf loc ->
                  mkexp loc
                    (Pexp_assert
                       (mkexp loc
                          (Pexp_construct
                             ({ txt = Lident "false"; loc = mkloc loc; },
                             None))))
              | ExAss (loc, e, v) ->
                  let e =
                    (match e with
                     | Ast.ExAcc (loc, x,
                         (Ast.ExId (_, (Ast.IdLid (_, "val"))))) ->
                         Pexp_apply
                           ((mkexp loc
                               (Pexp_ident (lident_with_loc ":=" loc))),
                           [ (Nolabel, (expr x)); (Nolabel, (expr v)) ])
                     | ExAcc (loc, _, _) ->
                         (match (expr e).pexp_desc with
                          | Pexp_field (e, lab) ->
                              Pexp_setfield (e, lab, (expr v))
                          | _ -> error loc "bad record access")
                     | ExAre (loc, e1, e2) ->
                         Pexp_apply
                           ((mkexp loc
                               (Pexp_ident (array_function loc "Array" "set"))),
                           [ (Nolabel, (expr e1)); (Nolabel, (expr e2));
                             (Nolabel, (expr v)) ])
                     | Ast.ExId (_, (Ast.IdLid (lloc, lab))) ->
                         Pexp_setinstvar ((with_loc lab lloc), (expr v))
                     | ExSte (loc, e1, e2) ->
                         Pexp_apply
                           ((mkexp loc
                               (Pexp_ident
                                  (array_function loc "String" "set"))),
                           [ (Nolabel, (expr e1)); (Nolabel, (expr e2));
                             (Nolabel, (expr v)) ])
                     | _ -> error loc "bad left part of assignment")
                  in mkexp loc e
              | ExAsr (loc, e) -> mkexp loc (Pexp_assert (expr e))
              | ExChr (loc, s) ->
                  mkexp loc
                    (Pexp_constant (Pconst_char (char_of_char_token loc s)))
              | ExCoe (loc, e, t1, t2) ->
                  let t1 =
                    (match t1 with | Ast.TyNil _ -> None | t -> Some (ctyp t))
                  in mkexp loc (Pexp_coerce ((expr e), t1, (ctyp t2)))
              | ExFlo (loc, s) ->
                  mkexp loc
                    (Pexp_constant
                       (Pconst_float (((remove_underscores s), None))))
              | ExFor (loc, p, e1, e2, df, el) ->
                  let e3 = ExSeq (loc, el)
                  in
                    mkexp loc
                      (Pexp_for ((patt p), (expr e1), (expr e2),
                         (mkdirection df), (expr e3)))
              | Ast.ExFun (loc, (Ast.McArr (_, (PaLab (_, lab, po)), w, e)))
                  ->
                  mkfun loc (Labelled lab) None (patt_of_lab loc lab po) e w
              | Ast.ExFun (loc,
                  (Ast.McArr (_, (PaOlbi (_, lab, p, e1)), w, e2))) ->
                  let lab = paolab lab p
                  in mkfun loc (Optional lab) (Some (expr e1)) (patt p) e2 w
              | Ast.ExFun (loc, (Ast.McArr (_, (PaOlb (_, lab, p)), w, e)))
                  ->
                  let lab = paolab lab p
                  in
                    mkfun loc (Optional lab) None (patt_of_lab loc lab p) e w
              | ExFun (loc, a) -> mkexp loc (Pexp_function (match_case a []))
              | ExIfe (loc, e1, e2, e3) ->
                  mkexp loc
                    (Pexp_ifthenelse ((expr e1), (expr e2), (Some (expr e3))))
              | ExInt (loc, s) ->
                  mkexp loc (Pexp_constant (Pconst_integer ((s, None))))
              | ExInt32 (loc, s) ->
                  mkexp loc
                    (Pexp_constant (Pconst_integer ((s, (Some 'l')))))
              | ExInt64 (loc, s) ->
                  mkexp loc
                    (Pexp_constant (Pconst_integer ((s, (Some 'L')))))
              | ExNativeInt (loc, s) ->
                  mkexp loc
                    (Pexp_constant (Pconst_integer ((s, (Some 'n')))))
              | ExLab (loc, _, _) ->
                  error loc "labeled expression not allowed here"
              | ExLaz (loc, e) -> mkexp loc (Pexp_lazy (expr e))
              | ExLet (loc, rf, bi, e) ->
                  let e = expr e
                  in
                    (match binding bi [] with
                     | [] -> e
                     | bi -> mkexp loc (Pexp_let ((mkrf rf), bi, e)))
              | ExLmd (loc, i, me, e) ->
                  mkexp loc
                    (Pexp_letmodule ((with_loc (Some i) loc), (module_expr me),
                       (expr e)))
              | ExMat (loc, e, a) ->
                  mkexp loc (Pexp_match ((expr e), (match_case a [])))
              | ExNew (loc, id) -> mkexp loc (Pexp_new (long_type_ident id))
              | ExObj (loc, po, cfl) ->
                  let p =
                    (match po with | Ast.PaNil _ -> Ast.PaAny loc | p -> p) in
                  let cil = class_str_item cfl []
                  in
                    mkexp loc
                      (Pexp_object
                         { pcstr_self = patt p; pcstr_fields = cil; })
              | ExOlb (loc, _, _) ->
                  error loc "labeled expression not allowed here"
              | ExOvr (loc, iel) ->
                  mkexp loc (Pexp_override (mkideexp iel []))
              | ExRec (loc, lel, eo) ->
                  (match lel with
                   | Ast.RbNil _ -> error loc "empty record"
                   | _ ->
                       let eo =
                         (match eo with
                          | Ast.ExNil _ -> None
                          | e -> Some (expr e))
                       in mkexp loc (Pexp_record ((mklabexp lel []), eo)))
              | ExSeq (_loc, e) ->
                  let rec loop =
                    (function
                     | [] -> expr (Ast.ExId (_loc, (Ast.IdUid (_loc, "()"))))
                     | [ e ] -> expr e
                     | e :: el ->
                         let _loc = Loc.merge (loc_of_expr e) _loc
                         in mkexp _loc (Pexp_sequence ((expr e), (loop el))))
                  in loop (list_of_expr e [])
              | ExSnd (loc, e, s) ->
                  mkexp loc (Pexp_send ((expr e), (with_loc s loc)))
              | ExSte (loc, e1, e2) ->
                  mkexp loc
                    (Pexp_apply
                       ((mkexp loc
                           (Pexp_ident (array_function loc "String" "get"))),
                       [ (Nolabel, (expr e1)); (Nolabel, (expr e2)) ]))
              | ExStr (loc, s) ->
                  mkexp loc
                    (Pexp_constant
                       (Pconst_string ((string_of_string_token loc s), Loc.to_ocaml_location loc, None)))
              | ExTry (loc, e, a) ->
                  mkexp loc (Pexp_try ((expr e), (match_case a [])))
              | Ast.ExTup (loc, (Ast.ExCom (_, e1, e2))) ->
                  mkexp loc
                    (Pexp_tuple
                       (List.map expr (list_of_expr e1 (list_of_expr e2 []))))
              | Ast.ExTup (loc, _) -> error loc "singleton tuple"
              | ExTyc (loc, e, t) ->
                  mkexp loc (Pexp_constraint ((expr e), (ctyp t)))
              | Ast.ExId (loc, (Ast.IdUid (_, "()"))) ->
                  mkexp loc
                    (Pexp_construct ((lident_with_loc "()" loc), None))
              | Ast.ExId (loc, (Ast.IdLid (_, s))) ->
                  mkexp loc (Pexp_ident (lident_with_loc s loc))
              | Ast.ExId (loc, (Ast.IdUid (_, s))) ->
                  mkexp loc
                    (Pexp_construct ((lident_with_loc (conv_con s) loc),
                       None))
              | ExVrn (loc, s) ->
                  mkexp loc (Pexp_variant ((conv_con s), None))
              | ExWhi (loc, e1, el) ->
                  let e2 = ExSeq (loc, el)
                  in mkexp loc (Pexp_while ((expr e1), (expr e2)))
              | ExOpI (loc, i, ov, e) ->
                  let fresh = override_flag loc ov
                  in
                    mkexp loc
                      (Pexp_open
                         (({
                             popen_loc = mkloc loc;
                             popen_override = fresh;
                             popen_attributes = [];
                             popen_expr =
                               {
                                 pmod_desc = Pmod_ident (long_uident i);
                                 pmod_loc = mkloc loc;
                                 pmod_attributes = [];
                               };
                           }, (expr e))))
              | Ast.ExPkg (loc, (Ast.MeTyc (_, me, pt))) ->
                  mkexp loc
                    (Pexp_constraint
                       (((mkexp loc (Pexp_pack (module_expr me))),
                         (mktyp loc (Ptyp_package (package_type pt))))))
              | Ast.ExPkg (loc, me) -> mkexp loc (Pexp_pack (module_expr me))
              | ExFUN (loc, i, e) ->
                  mkexp loc (Pexp_newtype ((with_loc i loc), (expr e)))
              | Ast.ExCom (loc, _, _) ->
                  error loc "expr, expr: not allowed here"
              | Ast.ExSem (loc, _, _) ->
                  error loc
                    "expr; expr: not allowed here, use do {...} or [|...|] to surround them"
              | ExAtt (loc, s, str, e) ->
                  let e = expr e
                  in
                    {
                      (e)
                      with
                      pexp_attributes =
                        e.pexp_attributes @ [ attribute loc s str ];
                    }
              | (ExId (_, _) | ExNil _ as e) ->
                  error (loc_of_expr e) "invalid expr"
            and patt_of_lab _loc lab =
              function
              | Ast.PaNil _ ->
                  patt (Ast.PaId (_loc, (Ast.IdLid (_loc, lab))))
              | p -> patt p
            and expr_of_lab _loc lab =
              function
              | Ast.ExNil _ ->
                  expr (Ast.ExId (_loc, (Ast.IdLid (_loc, lab))))
              | e -> expr e
            and label_expr =
              function
              | ExLab (loc, lab, eo) ->
                  ((Labelled lab), (expr_of_lab loc lab eo))
              | ExOlb (loc, lab, eo) ->
                  ((Optional lab), (expr_of_lab loc lab eo))
              | e -> (Nolabel, (expr e))
            and binding x acc =
              match x with
              | Ast.BiAnd (_, x, y) -> binding x (binding y acc)
              | Ast.BiEq (_loc,
                  (Ast.PaId (sloc, (Ast.IdLid (_, bind_name)))),
                  (Ast.ExTyc (_, e, (TyTypePol (_, vs, ty))))) ->
                  let rec id_to_string x =
                    (match x with
                     | Ast.TyId (loc, (Ast.IdLid (_, x))) ->
                         [ with_loc x loc ]
                     | Ast.TyApp (_, x, y) ->
                         (id_to_string x) @ (id_to_string y)
                     | _ -> assert false) in
                  let vars = id_to_string vs in
                  let ampersand_vars =
                    List.map (fun x -> { loc = x.loc; txt = "&" ^ x.txt; })
                      vars in
                  let ty' = varify_constructors vars (ctyp ty) in
                  let mkexp = mkexp _loc in
                  let mkpat = mkpat _loc in
                  let e = mkexp (Pexp_constraint ((expr e), (ctyp ty))) in
                  let rec mk_newtypes x =
                    (match x with
                     | [ newtype ] -> mkexp (Pexp_newtype ((newtype, e)))
                     | newtype :: newtypes ->
                         mkexp
                           (Pexp_newtype ((newtype, (mk_newtypes newtypes))))
                     | [] -> assert false) in
                  let pat =
                    mkpat
                      (Ppat_constraint
                         (((mkpat (Ppat_var (with_loc bind_name sloc))),
                           (mktyp _loc (Ptyp_poly (ampersand_vars, ty')))))) in
                  let e = mk_newtypes vars
                  in
                    {
                      pvb_pat = pat;
                      pvb_expr = e;
                      pvb_attributes = [];
                      pvb_loc = mkloc _loc;
                    } :: acc
              | Ast.BiEq (_loc, p,
                  (Ast.ExTyc (_, e, (Ast.TyPol (_, vs, ty))))) ->
                  {
                    pvb_pat =
                      patt (Ast.PaTyc (_loc, p, (Ast.TyPol (_loc, vs, ty))));
                    pvb_expr = expr e;
                    pvb_attributes = [];
                    pvb_loc = mkloc _loc;
                  } :: acc
              | Ast.BiEq (_loc, p, e) ->
                  {
                    pvb_pat = patt p;
                    pvb_expr = expr e;
                    pvb_attributes = [];
                    pvb_loc = mkloc _loc;
                  } :: acc
              | Ast.BiNil _ -> acc
              | _ -> assert false
            and match_case x acc =
              match x with
              | Ast.McOr (_, x, y) -> match_case x (match_case y acc)
              | Ast.McArr (_, p, w, e) -> (when_expr (patt p) e w) :: acc
              | Ast.McNil _ -> acc
              | _ -> assert false
            and when_expr p e w =
              let g = match w with | Ast.ExNil _ -> None | g -> Some (expr g)
              in { pc_lhs = p; pc_guard = g; pc_rhs = expr e; }
            and mkfun loc lab def p e w =
              let () = match w with | Ast.ExNil _ -> () | _ -> assert false
              in mkexp loc (Pexp_fun (lab, def, p, (expr e)))
            and mklabexp x acc =
              match x with
              | Ast.RbSem (_, x, y) -> mklabexp x (mklabexp y acc)
              | Ast.RbEq (_, i, e) ->
                  ((ident ~conv_lid: conv_lab i), (expr e)) :: acc
              | _ -> assert false
            and mkideexp x acc =
              match x with
              | Ast.RbNil _ -> acc
              | Ast.RbSem (_, x, y) -> mkideexp x (mkideexp y acc)
              | Ast.RbEq (_, (Ast.IdLid (sloc, s)), e) ->
                  ((with_loc s sloc), (expr e)) :: acc
              | _ -> assert false
            and mktype_decl_or_ext x acc =
              match x with
              | Ast.TyAnd (_, x, y) ->
                  mktype_decl_or_ext x (mktype_decl_or_ext y acc)
              | Ast.TyDcl (cloc, c, tl, td, cl) ->
                  let cl =
                    List.map
                      (fun (t1, t2) ->
                         let loc =
                           Loc.merge (loc_of_ctyp t1) (loc_of_ctyp t2)
                         in ((ctyp t1), (ctyp t2), (mkloc loc)))
                      cl in
                  let td =
                    type_decl (with_loc c cloc)
                      (List.fold_right optional_type_parameters tl []) cl td
                      cloc
                  in
                    (match acc with
                     | `Unknown -> `Dcl [ td ]
                     | `Dcl acc -> `Dcl (td :: acc)
                     | `Ext _ ->
                         error cloc
                           "cannot mix type declaration and extension")
              | Ast.TyExt (cloc, c, tl, td) ->
                  (match acc with
                   | `Unknown ->
                       `Ext
                         (type_ext (long_type_ident c)
                            (List.fold_right optional_type_parameters tl [])
                            td cloc)
                   | `Dcl _ ->
                       error cloc "cannot mix type declaration and extension"
                   | `Ext _ -> error cloc "only one type extension allowed")
              | _ -> assert false
            and module_type =
              function
              | Ast.MtNil loc ->
                  error loc "abstract/nil module type not allowed here"
              | Ast.MtId (loc, i) -> mkmty loc (Pmty_ident (long_uident i))
              | Ast.MtFun ((loc, "*", Ast.MtNil _, mt)) ->
                  mkmty loc
                    (Pmty_functor (Unit, (module_type mt)))
              | Ast.MtFun (loc, n, nt, mt) ->
                  mkmty loc
                    (Pmty_functor (Named ((with_loc (Some n) loc), (module_type nt)),
                       (module_type mt)))
              | Ast.MtQuo (loc, _) ->
                  error loc "module type variable not allowed here"
              | Ast.MtSig (loc, sl) ->
                  mkmty loc (Pmty_signature (sig_item sl []))
              | Ast.MtWit (loc, mt, wc) ->
                  mkmty loc (Pmty_with ((module_type mt), (mkwithc wc [])))
              | Ast.MtOf (loc, me) ->
                  mkmty loc (Pmty_typeof (module_expr me))
              | MtAtt (loc, s, str, e) ->
                  let e = module_type e
                  in
                    {
                      (e)
                      with
                      pmty_attributes =
                        e.pmty_attributes @ [ attribute loc s str ];
                    }
              | Ast.MtAlias ((loc, id)) ->
                  mkmty loc (Pmty_alias (long_uident id))
              | Ast.MtAnt (_, _) -> assert false
            and sig_item s l =
              match s with
              | Ast.SgNil _ -> l
              | SgCls (loc, cd) ->
                  (mksig loc
                     (Psig_class
                        (List.map class_info_class_type
                           (list_of_class_type cd [])))) ::
                    l
              | SgClt (loc, ctd) ->
                  (mksig loc
                     (Psig_class_type
                        (List.map class_info_class_type
                           (list_of_class_type ctd [])))) ::
                    l
              | Ast.SgSem (_, sg1, sg2) -> sig_item sg1 (sig_item sg2 l)
              | SgDir (_, _, _) -> l
              | Ast.SgExc (loc, (Ast.TyId (_, (Ast.IdUid (_, s))))) ->
                  (mksig loc
                     (Psig_exception
                        {
                          ptyexn_constructor =
                            {
                              pext_name = with_loc (conv_con s) loc;
                              pext_kind =
                                Pext_decl (((Pcstr_tuple []), None));
                              pext_attributes = [];
                              pext_loc = mkloc loc;
                            };
                          ptyexn_loc = mkloc loc;
                          ptyexn_attributes = [];
                        })) ::
                    l
              | Ast.SgExc (loc,
                  (Ast.TyOf (_, (Ast.TyId (_, (Ast.IdUid (_, s)))), t))) ->
                  (mksig loc
                     (Psig_exception
                        {
                          ptyexn_loc = mkloc loc;
                          ptyexn_attributes = [];
                          ptyexn_constructor =
                            {
                              pext_name = with_loc (conv_con s) loc;
                              pext_kind =
                                Pext_decl
                                  (((Pcstr_tuple
                                       (List.map ctyp (list_of_ctyp t []))),
                                    None));
                              pext_attributes = [];
                              pext_loc = mkloc loc;
                            };
                        })) ::
                    l
              | SgExc (_, _) -> assert false
              | SgExt (loc, n, t, sl) ->
                  (mksig loc
                     (Psig_value
                        (mkvalue_desc loc (with_loc n loc) t
                           (list_of_meta_list sl)))) ::
                    l
              | SgInc (loc, mt) ->
                  (mksig loc
                     (Psig_include
                        {
                          pincl_mod = module_type mt;
                          pincl_attributes = [];
                          pincl_loc = mkloc loc;
                        })) ::
                    l
              | SgMod (loc, n, mt) ->
                  (mksig loc
                     (Psig_module
                        {
                          pmd_loc = mkloc loc;
                          pmd_name = with_loc (Some n) loc;
                          pmd_type = module_type mt;
                          pmd_attributes = [];
                        })) ::
                    l
              | SgRecMod (loc, mb) ->
                  (mksig loc (Psig_recmodule (module_sig_binding mb []))) ::
                    l
              | SgMty (loc, n, mt) ->
                  let si =
                    (match mt with
                     | MtQuo (_, _) -> None
                     | _ -> Some (module_type mt))
                  in
                    (mksig loc
                       (Psig_modtype
                          {
                            pmtd_loc = mkloc loc;
                            pmtd_name = with_loc n loc;
                            pmtd_type = si;
                            pmtd_attributes = [];
                          })) ::
                      l
              | SgOpn (loc, ov, id) ->
                  let fresh = override_flag loc ov
                  in
                    (mksig loc
                       (Psig_open
                          {
                            popen_override = fresh;
                            popen_expr = long_uident id;
                            popen_attributes = [];
                            popen_loc = mkloc loc;
                          })) ::
                      l
              | SgTyp (loc, rf, tdl) ->
                  let rf = mknrf rf in
                  let ty =
                    (match mktype_decl_or_ext tdl `Unknown with
                     | `Unknown -> Psig_type ((rf, []))
                     | `Dcl l -> Psig_type ((rf, l))
                     | `Ext e -> Psig_typext e)
                  in (mksig loc ty) :: l
              | SgVal (loc, n, t) ->
                  (mksig loc
                     (Psig_value (mkvalue_desc loc (with_loc n loc) t []))) ::
                    l
              | Ast.SgAnt (loc, _) -> error loc "antiquotation in sig_item"
            and module_sig_binding x acc =
              match x with
              | Ast.MbAnd (_, x, y) ->
                  module_sig_binding x (module_sig_binding y acc)
              | Ast.MbCol (loc, s, mt) ->
                  {
                    pmd_loc = mkloc loc;
                    pmd_name = with_loc (Some s) loc;
                    pmd_type = module_type mt;
                    pmd_attributes = [];
                  } :: acc
              | _ -> assert false
            and module_str_binding x acc =
              match x with
              | Ast.MbAnd (_, x, y) ->
                  module_str_binding x (module_str_binding y acc)
              | Ast.MbColEq (loc, s, mt, me) ->
                  {
                    pmb_loc = mkloc loc;
                    pmb_name = with_loc (Some s) loc;
                    pmb_expr =
                      {
                        pmod_loc = Location.none;
                        pmod_desc =
                          Pmod_constraint
                            (((module_expr me), (module_type mt)));
                        pmod_attributes = [];
                      };
                    pmb_attributes = [];
                  } :: acc
              | _ -> assert false
            and module_expr =
              function
              | Ast.MeNil loc -> error loc "nil module expression"
              | Ast.MeId (loc, i) -> mkmod loc (Pmod_ident (long_uident i))
              | Ast.MeApp (loc, me1, me2) ->
                  mkmod loc
                    (Pmod_apply ((module_expr me1), (module_expr me2)))
              | Ast.MeFun ((loc, "*", Ast.MtNil _, me)) ->
                  mkmod loc
                    (Pmod_functor (Unit, (module_expr me)))
              | Ast.MeFun (loc, n, mt, me) ->
                  mkmod loc
                    (Pmod_functor (Named ((with_loc (Some n) loc), (module_type mt)),
                       (module_expr me)))
              | Ast.MeStr (loc, sl) ->
                  mkmod loc (Pmod_structure (str_item sl []))
              | Ast.MeTyc (loc, me, mt) ->
                  mkmod loc
                    (Pmod_constraint ((module_expr me), (module_type mt)))
              | Ast.MePkg (loc, (Ast.ExTyc (_, e, (Ast.TyPkg (_, pt))))) ->
                  mkmod loc
                    (Pmod_unpack
                       (mkexp loc
                          (Pexp_constraint
                             (((expr e),
                               (mktyp loc (Ptyp_package (package_type pt))))))))
              | Ast.MePkg (loc, e) -> mkmod loc (Pmod_unpack (expr e))
              | MeAtt (loc, s, str, e) ->
                  let e = module_expr e
                  in
                    {
                      (e)
                      with
                      pmod_attributes =
                        e.pmod_attributes @ [ attribute loc s str ];
                    }
              | Ast.MeAnt (loc, _) ->
                  error loc "antiquotation in module_expr"
            and str_item s l =
              match s with
              | Ast.StNil _ -> l
              | StCls (loc, cd) ->
                  (mkstr loc
                     (Pstr_class
                        (List.map class_info_class_expr
                           (list_of_class_expr cd [])))) ::
                    l
              | StClt (loc, ctd) ->
                  (mkstr loc
                     (Pstr_class_type
                        (List.map class_info_class_type
                           (list_of_class_type ctd [])))) ::
                    l
              | Ast.StSem (_, st1, st2) -> str_item st1 (str_item st2 l)
              | StDir (_, _, _) -> l
              | Ast.StExc (loc, (Ast.TyId (_, (Ast.IdUid (_, s)))), Ast.
                  ONone) ->
                  (mkstr loc
                     (Pstr_exception
                        {
                          ptyexn_loc = mkloc loc;
                          ptyexn_attributes = [];
                          ptyexn_constructor =
                            {
                              pext_name = with_loc (conv_con s) loc;
                              pext_kind =
                                Pext_decl (((Pcstr_tuple []), None));
                              pext_attributes = [];
                              pext_loc = mkloc loc;
                            };
                        })) ::
                    l
              | Ast.StExc (loc,
                  (Ast.TyOf (_, (Ast.TyId (_, (Ast.IdUid (_, s)))), t)), Ast.
                  ONone) ->
                  (mkstr loc
                     (Pstr_exception
                        {
                          ptyexn_loc = mkloc loc;
                          ptyexn_attributes = [];
                          ptyexn_constructor =
                            {
                              pext_name = with_loc (conv_con s) loc;
                              pext_kind =
                                Pext_decl
                                  (((Pcstr_tuple
                                       (List.map ctyp (list_of_ctyp t []))),
                                    None));
                              pext_attributes = [];
                              pext_loc = mkloc loc;
                            };
                        })) ::
                    l
              | Ast.StExc (loc, (Ast.TyId (_, (Ast.IdUid (_, s)))),
                  (Ast.OSome i)) ->
                  (mkstr loc
                     (Pstr_exception
                        {
                          ptyexn_loc = mkloc loc;
                          ptyexn_attributes = [];
                          ptyexn_constructor =
                            {
                              pext_name = with_loc (conv_con s) loc;
                              pext_kind =
                                Pext_rebind (long_uident ~conv_con i);
                              pext_attributes = [];
                              pext_loc = mkloc loc;
                            };
                        })) ::
                    l
              | Ast.StExc (loc,
                  (Ast.TyOf (_, (Ast.TyId (_, (Ast.IdUid (_, _)))), _)),
                  (Ast.OSome _)) -> error loc "type in exception alias"
              | StExc (_, _, _) -> assert false
              | StExp (loc, e) -> (mkstr loc (Pstr_eval ((expr e), []))) :: l
              | StExt (loc, n, t, sl) ->
                  (mkstr loc
                     (Pstr_primitive
                        (mkvalue_desc loc (with_loc n loc) t
                           (list_of_meta_list sl)))) ::
                    l
              | StInc (loc, me) ->
                  (mkstr loc
                     (Pstr_include
                        {
                          pincl_mod = module_expr me;
                          pincl_attributes = [];
                          pincl_loc = mkloc loc;
                        })) ::
                    l
              | StMod (loc, n, me) ->
                  (mkstr loc
                     (Pstr_module
                        {
                          pmb_loc = mkloc loc;
                          pmb_name = with_loc (Some n) loc;
                          pmb_expr = module_expr me;
                          pmb_attributes = [];
                        })) ::
                    l
              | StRecMod (loc, mb) ->
                  (mkstr loc (Pstr_recmodule (module_str_binding mb []))) ::
                    l
              | StMty (loc, n, mt) ->
                  let si =
                    (match mt with
                     | MtQuo (_, _) -> None
                     | _ -> Some (module_type mt))
                  in
                    (mkstr loc
                       (Pstr_modtype
                          {
                            pmtd_loc = mkloc loc;
                            pmtd_name = with_loc n loc;
                            pmtd_type = si;
                            pmtd_attributes = [];
                          })) ::
                      l
              | StOpn (loc, ov, id) ->
                  let fresh = override_flag loc ov
                  in
                    (mkstr loc
                       (Pstr_open
                          {
                            popen_override = fresh;
                            popen_expr =
                              {
                                pmod_desc = Pmod_ident (long_uident id);
                                pmod_loc = mkloc loc;
                                pmod_attributes = [];
                              };
                            popen_attributes = [];
                            popen_loc = mkloc loc;
                          })) ::
                      l
              | StTyp (loc, rf, tdl) ->
                  let rf = mknrf rf in
                  let ty =
                    (match mktype_decl_or_ext tdl `Unknown with
                     | `Unknown -> Pstr_type ((rf, []))
                     | `Dcl l -> Pstr_type ((rf, l))
                     | `Ext e -> Pstr_typext e)
                  in (mkstr loc ty) :: l
              | StVal (loc, rf, bi) ->
                  (mkstr loc (Pstr_value ((mkrf rf), (binding bi [])))) :: l
              | Ast.StAnt (loc, _) -> error loc "antiquotation in str_item"
            and class_type =
              function
              | CtCon (loc, ViNil, id, tl) ->
                  mkcty loc
                    (Pcty_constr ((long_class_ident id),
                       (List.map ctyp (list_of_opt_ctyp tl []))))
              | CtFun (loc, (TyLab (_, lab, t)), ct) ->
                  mkcty loc
                    (Pcty_arrow ((Labelled lab), (ctyp t), (class_type ct)))
              | CtFun (loc, (TyOlb (_, lab, t)), ct) ->
                  mkcty loc
                    (Pcty_arrow ((Optional lab), (ctyp t), (class_type ct)))
              | CtFun (loc, t, ct) ->
                  mkcty loc (Pcty_arrow (Nolabel, (ctyp t), (class_type ct)))
              | CtSig (loc, t_o, ctfl) ->
                  let t =
                    (match t_o with | Ast.TyNil _ -> Ast.TyAny loc | t -> t) in
                  let cil = class_sig_item ctfl []
                  in
                    mkcty loc
                      (Pcty_signature
                         { pcsig_self = ctyp t; pcsig_fields = cil; })
              | CtAtt (loc, s, str, e) ->
                  let e = class_type e
                  in
                    {
                      (e)
                      with
                      pcty_attributes =
                        e.pcty_attributes @ [ attribute loc s str ];
                    }
              | CtCon (loc, _, _, _) ->
                  error loc "invalid virtual class inside a class type"
              | CtAnt (_, _) | CtEq (_, _, _) | CtCol (_, _, _) |
                  CtAnd (_, _, _) | CtNil _ -> assert false
            and class_info_class_expr ci =
              match ci with
              | CeEq (_, (CeCon (loc, vir, (IdLid (nloc, name)), params)),
                  ce) ->
                  let params =
                    (match params with
                     | Ast.TyNil _ -> []
                     | t -> class_parameters t [])
                  in
                    {
                      pci_virt = mkvirtual vir;
                      pci_params = params;
                      pci_name = with_loc name nloc;
                      pci_expr = class_expr ce;
                      pci_loc = mkloc loc;
                      pci_attributes = [];
                    }
              | ce -> error (loc_of_class_expr ce) "bad class definition"
            and class_info_class_type ci =
              match ci with
              | CtEq (_, (CtCon (loc, vir, (IdLid (nloc, name)), params)),
                  ct) |
                  CtCol (_, (CtCon (loc, vir, (IdLid (nloc, name)), params)),
                    ct)
                  ->
                  let params =
                    (match params with
                     | Ast.TyNil _ -> []
                     | t -> class_parameters t [])
                  in
                    {
                      pci_virt = mkvirtual vir;
                      pci_params = params;
                      pci_name = with_loc name nloc;
                      pci_expr = class_type ct;
                      pci_attributes = [];
                      pci_loc = mkloc loc;
                    }
              | ct ->
                  error (loc_of_class_type ct)
                    "bad class/class type declaration/definition"
            and class_sig_item c l =
              match c with
              | Ast.CgNil _ -> l
              | CgCtr (loc, t1, t2) ->
                  (mkctf loc (Pctf_constraint (((ctyp t1), (ctyp t2))))) :: l
              | Ast.CgSem (_, csg1, csg2) ->
                  class_sig_item csg1 (class_sig_item csg2 l)
              | CgInh (loc, ct) ->
                  (mkctf loc (Pctf_inherit (class_type ct))) :: l
              | CgMth (loc, s, pf, t) ->
                  (mkctf loc
                     (Pctf_method
                        (((with_loc s loc), (mkprivate pf), Concrete,
                          (mkpolytype (ctyp t)))))) ::
                    l
              | CgVal (loc, s, b, v, t) ->
                  (mkctf loc
                     (Pctf_val
                        (((with_loc s loc), (mkmutable b), (mkvirtual v),
                          (ctyp t))))) ::
                    l
              | CgVir (loc, s, b, t) ->
                  (mkctf loc
                     (Pctf_method
                        (((with_loc s loc), (mkprivate b), Virtual,
                          (mkpolytype (ctyp t)))))) ::
                    l
              | CgAnt (_, _) -> assert false
            and class_expr =
              function
              | (CeApp (loc, _, _) as c) ->
                  let (ce, el) = class_expr_fa [] c in
                  let el = List.map label_expr el
                  in mkcl loc (Pcl_apply ((class_expr ce), el))
              | CeCon (loc, ViNil, id, tl) ->
                  mkcl loc
                    (Pcl_constr ((long_class_ident id),
                       (List.map ctyp (list_of_opt_ctyp tl []))))
              | CeFun (loc, (PaLab (_, lab, po)), ce) ->
                  mkcl loc
                    (Pcl_fun ((Labelled lab), None, (patt_of_lab loc lab po),
                       (class_expr ce)))
              | CeFun (loc, (PaOlbi (_, lab, p, e)), ce) ->
                  let lab = paolab lab p
                  in
                    mkcl loc
                      (Pcl_fun ((Optional lab), (Some (expr e)), (patt p),
                         (class_expr ce)))
              | CeFun (loc, (PaOlb (_, lab, p)), ce) ->
                  let lab = paolab lab p
                  in
                    mkcl loc
                      (Pcl_fun ((Optional lab), None,
                         (patt_of_lab loc lab p), (class_expr ce)))
              | CeFun (loc, p, ce) ->
                  mkcl loc
                    (Pcl_fun (Nolabel, None, (patt p), (class_expr ce)))
              | CeLet (loc, rf, bi, ce) ->
                  mkcl loc
                    (Pcl_let ((mkrf rf), (binding bi []), (class_expr ce)))
              | CeStr (loc, po, cfl) ->
                  let p =
                    (match po with | Ast.PaNil _ -> Ast.PaAny loc | p -> p) in
                  let cil = class_str_item cfl []
                  in
                    mkcl loc
                      (Pcl_structure
                         { pcstr_self = patt p; pcstr_fields = cil; })
              | CeTyc (loc, ce, ct) ->
                  mkcl loc
                    (Pcl_constraint ((class_expr ce), (class_type ct)))
              | CeAtt (loc, s, str, e) ->
                  let e = class_expr e
                  in
                    {
                      (e)
                      with
                      pcl_attributes =
                        e.pcl_attributes @ [ attribute loc s str ];
                    }
              | CeCon (loc, _, _, _) ->
                  error loc "invalid virtual class inside a class expression"
              | CeAnt (_, _) | CeEq (_, _, _) | CeAnd (_, _, _) | CeNil _ ->
                  assert false
            and class_str_item c l =
              match c with
              | CrNil _ -> l
              | CrCtr (loc, t1, t2) ->
                  (mkcf loc (Pcf_constraint (((ctyp t1), (ctyp t2))))) :: l
              | Ast.CrSem (_, cst1, cst2) ->
                  class_str_item cst1 (class_str_item cst2 l)
              | CrInh (loc, ov, ce, pb) ->
                  let opb = if pb = "" then None else Some (with_loc pb loc)
                  in
                    (mkcf loc
                       (Pcf_inherit ((override_flag loc ov), (class_expr ce),
                          opb))) ::
                      l
              | CrIni (loc, e) -> (mkcf loc (Pcf_initializer (expr e))) :: l
              | CrMth (loc, s, ov, pf, e, t) ->
                  let t =
                    (match t with
                     | Ast.TyNil _ -> None
                     | t -> Some (mkpolytype (ctyp t))) in
                  let e = mkexp loc (Pexp_poly ((expr e), t))
                  in
                    (mkcf loc
                       (Pcf_method
                          (((with_loc s loc), (mkprivate pf),
                            (Cfk_concrete (((override_flag loc ov), e))))))) ::
                      l
              | CrVal (loc, s, ov, mf, e) ->
                  (mkcf loc
                     (Pcf_val
                        (((with_loc s loc), (mkmutable mf),
                          (Cfk_concrete (((override_flag loc ov), (expr e)))))))) ::
                    l
              | CrVir (loc, s, pf, t) ->
                  (mkcf loc
                     (Pcf_method
                        (((with_loc s loc), (mkprivate pf),
                          (Cfk_virtual (mkpolytype (ctyp t))))))) ::
                    l
              | CrVvr (loc, s, mf, t) ->
                  (mkcf loc
                     (Pcf_val
                        (((with_loc s loc), (mkmutable mf),
                          (Cfk_virtual (ctyp t)))))) ::
                    l
              | CrAnt (_, _) -> assert false
              
            let sig_item ast = sig_item ast []
              
            let str_item ast = str_item ast []
              
            let directive_arg x =
              let x =
                match x with
                | ExStr (loc, s) -> Some ((loc, (Pdir_string s)))
                | ExInt (loc, i) -> Some ((loc, (Pdir_int ((i, None)))))
                | Ast.ExId (loc, (Ast.IdUid (_, "True"))) ->
                    Some ((loc, (Pdir_bool true)))
                | Ast.ExId (loc, (Ast.IdUid (_, "False"))) ->
                    Some ((loc, (Pdir_bool false)))
                | Ast.ExNil _ -> None
                | e ->
                    Some
                      (((Ast.loc_of_expr e),
                        (Pdir_ident (ident_noloc (ident_of_expr e)))))
              in
                match x with
                | None -> None
                | Some ((loc, x)) ->
                    Some { pdira_desc = x; pdira_loc = mkloc loc; }
              
            let phrase =
              function
              | StDir (loc, d, arg) ->
                  Ptop_dir
                    {
                      pdir_name = with_loc d loc;
                      pdir_arg = directive_arg arg;
                      pdir_loc = mkloc loc;
                    }
              | si -> Ptop_def (str_item si)
              
            let attribute loc s str =
              {
                attr_name = with_loc s loc;
                attr_payload = PStr (str_item str);
                attr_loc = mkloc loc;
              }
              
            let () = attribute_fwd := attribute
              
          end
          
      end
      
    module CleanAst =
      struct
        module Make (Ast : Sig.Camlp4Ast) =
          struct
            class clean_ast =
              object inherit Ast.map as super
                method with_constr =
                  fun wc ->
                    match super#with_constr wc with
                    | Ast.WcAnd (_, (Ast.WcNil _), wc) |
                        Ast.WcAnd (_, wc, (Ast.WcNil _)) -> wc
                    | wc -> wc
                method expr =
                  fun e ->
                    match super#expr e with
                    | Ast.ExLet (_, _, (Ast.BiNil _), e) |
                        Ast.ExRec (_, (Ast.RbNil _), e) |
                        Ast.ExCom (_, (Ast.ExNil _), e) |
                        Ast.ExCom (_, e, (Ast.ExNil _)) |
                        Ast.ExSem (_, (Ast.ExNil _), e) |
                        Ast.ExSem (_, e, (Ast.ExNil _)) -> e
                    | e -> e
                method patt =
                  fun p ->
                    match super#patt p with
                    | Ast.PaAli (_, p, (Ast.PaNil _)) |
                        Ast.PaOrp (_, (Ast.PaNil _), p) |
                        Ast.PaOrp (_, p, (Ast.PaNil _)) |
                        Ast.PaCom (_, (Ast.PaNil _), p) |
                        Ast.PaCom (_, p, (Ast.PaNil _)) |
                        Ast.PaSem (_, (Ast.PaNil _), p) |
                        Ast.PaSem (_, p, (Ast.PaNil _)) -> p
                    | p -> p
                method match_case =
                  fun mc ->
                    match super#match_case mc with
                    | Ast.McOr (_, (Ast.McNil _), mc) |
                        Ast.McOr (_, mc, (Ast.McNil _)) -> mc
                    | mc -> mc
                method binding =
                  fun bi ->
                    match super#binding bi with
                    | Ast.BiAnd (_, (Ast.BiNil _), bi) |
                        Ast.BiAnd (_, bi, (Ast.BiNil _)) -> bi
                    | bi -> bi
                method rec_binding =
                  fun rb ->
                    match super#rec_binding rb with
                    | Ast.RbSem (_, (Ast.RbNil _), bi) |
                        Ast.RbSem (_, bi, (Ast.RbNil _)) -> bi
                    | bi -> bi
                method module_binding =
                  fun mb ->
                    match super#module_binding mb with
                    | Ast.MbAnd (_, (Ast.MbNil _), mb) |
                        Ast.MbAnd (_, mb, (Ast.MbNil _)) -> mb
                    | mb -> mb
                method ctyp =
                  fun t ->
                    match super#ctyp t with
                    | Ast.TyPol (_, (Ast.TyNil _), t) |
                        Ast.TyAli (_, (Ast.TyNil _), t) |
                        Ast.TyAli (_, t, (Ast.TyNil _)) |
                        Ast.TyArr (_, t, (Ast.TyNil _)) |
                        Ast.TyArr (_, (Ast.TyNil _), t) |
                        Ast.TyOr (_, (Ast.TyNil _), t) |
                        Ast.TyOr (_, t, (Ast.TyNil _)) |
                        Ast.TyOf (_, t, (Ast.TyNil _)) |
                        Ast.TyAnd (_, (Ast.TyNil _), t) |
                        Ast.TyAnd (_, t, (Ast.TyNil _)) |
                        Ast.TySem (_, t, (Ast.TyNil _)) |
                        Ast.TySem (_, (Ast.TyNil _), t) |
                        Ast.TyCom (_, (Ast.TyNil _), t) |
                        Ast.TyCom (_, t, (Ast.TyNil _)) |
                        Ast.TyAmp (_, t, (Ast.TyNil _)) |
                        Ast.TyAmp (_, (Ast.TyNil _), t) |
                        Ast.TySta (_, (Ast.TyNil _), t) |
                        Ast.TySta (_, t, (Ast.TyNil _)) -> t
                    | t -> t
                method sig_item =
                  fun sg ->
                    match super#sig_item sg with
                    | Ast.SgSem (_, (Ast.SgNil _), sg) |
                        Ast.SgSem (_, sg, (Ast.SgNil _)) -> sg
                    | Ast.SgTyp ((loc, _, Ast.TyNil _)) -> Ast.SgNil loc
                    | sg -> sg
                method str_item =
                  fun st ->
                    match super#str_item st with
                    | Ast.StSem (_, (Ast.StNil _), st) |
                        Ast.StSem (_, st, (Ast.StNil _)) -> st
                    | Ast.StTyp ((loc, _, Ast.TyNil _)) -> Ast.StNil loc
                    | Ast.StVal (loc, _, (Ast.BiNil _)) -> Ast.StNil loc
                    | st -> st
                method module_type =
                  fun mt ->
                    match super#module_type mt with
                    | Ast.MtWit (_, mt, (Ast.WcNil _)) -> mt
                    | mt -> mt
                method class_expr =
                  fun ce ->
                    match super#class_expr ce with
                    | Ast.CeAnd (_, (Ast.CeNil _), ce) |
                        Ast.CeAnd (_, ce, (Ast.CeNil _)) -> ce
                    | ce -> ce
                method class_type =
                  fun ct ->
                    match super#class_type ct with
                    | Ast.CtAnd (_, (Ast.CtNil _), ct) |
                        Ast.CtAnd (_, ct, (Ast.CtNil _)) -> ct
                    | ct -> ct
                method class_sig_item =
                  fun csg ->
                    match super#class_sig_item csg with
                    | Ast.CgSem (_, (Ast.CgNil _), csg) |
                        Ast.CgSem (_, csg, (Ast.CgNil _)) -> csg
                    | csg -> csg
                method class_str_item =
                  fun cst ->
                    match super#class_str_item cst with
                    | Ast.CrSem (_, (Ast.CrNil _), cst) |
                        Ast.CrSem (_, cst, (Ast.CrNil _)) -> cst
                    | cst -> cst
              end
              
          end
          
      end
      
    module CommentFilter :
      sig
        module Make (Token : Sig.Camlp4Token) :
          sig
            open Token
              
            type t
            
            val mk : unit -> t
              
            val define : Token.Filter.t -> t -> unit
              
            val filter :
              t -> (Token.t * Loc.t) Stream.t -> (Token.t * Loc.t) Stream.t
              
            val take_list : t -> (string * Loc.t) list
              
            val take_stream : t -> (string * Loc.t) Stream.t
              
          end
          
      end =
      struct
        module Make (Token : Sig.Camlp4Token) =
          struct
            open Token
              
            type t =
              (((string * Loc.t) Stream.t) * ((string * Loc.t) Queue.t))
            
            let mk () =
              let q = Queue.create () in
              let f _ = try Some (Queue.take q) with | Queue.Empty -> None
              in ((Stream.from f), q)
              
            let filter (_, q) =
              let rec self (__strm : _ Stream.t) =
                match Stream.peek __strm with
                | Some ((Sig.COMMENT x, loc)) ->
                    (Stream.junk __strm;
                     let xs = __strm in (Queue.add (x, loc) q; self xs))
                | Some x ->
                    (Stream.junk __strm;
                     let xs = __strm
                     in Stream.icons x (Stream.slazy (fun _ -> self xs)))
                | _ -> Stream.sempty
              in self
              
            let take_list (_, q) =
              let rec self accu =
                if Queue.is_empty q
                then accu
                else self ((Queue.take q) :: accu)
              in self []
              
            let take_stream = fst
              
            let define token_fiter comments_strm =
              Token.Filter.define_filter token_fiter
                (fun previous strm -> previous (filter comments_strm strm))
              
          end
          
      end
      
    module DynLoader : sig include Sig.DynLoader
                              end =
      struct
        type t = string Queue.t
        
        exception Error of string * string
          
        let include_dir x y = Queue.add y x
          
        let fold_load_path x f acc = Queue.fold (fun x y -> f y x) acc x
          
        let mk ?(ocaml_stdlib = true) ?(camlp4_stdlib = true) () =
          let q = Queue.create ()
          in
            (if ocaml_stdlib
             then include_dir q Camlp4_config.ocaml_standard_library
             else ();
             if camlp4_stdlib
             then
               (include_dir q Camlp4_config.camlp4_standard_library;
                include_dir q
                  (Filename.concat Camlp4_config.camlp4_standard_library
                     "Camlp4Parsers");
                include_dir q
                  (Filename.concat Camlp4_config.camlp4_standard_library
                     "Camlp4Printers");
                include_dir q
                  (Filename.concat Camlp4_config.camlp4_standard_library
                     "Camlp4Filters"))
             else ();
             include_dir q ".";
             q)
          
        let find_in_path x name =
          if not (Filename.is_implicit name)
          then if Sys.file_exists name then name else raise Not_found
          else
            (let res =
               fold_load_path x
                 (fun dir ->
                    function
                    | None ->
                        let fullname = Filename.concat dir name
                        in
                          if Sys.file_exists fullname
                          then Some fullname
                          else None
                    | x -> x)
                 None
             in match res with | None -> raise Not_found | Some x -> x)
          
        let load =
          let _initialized = ref false
          in
            fun _path file ->
              (if not !_initialized
               then
                 (try
                    (Dynlink.allow_unsafe_modules true; _initialized := true)
                  with
                  | Dynlink.Error e ->
                      raise
                        (Error ("Camlp4's dynamic loader initialization",
                           (Dynlink.error_message e))))
               else ();
               let fname =
                 try find_in_path _path file
                 with
                 | Not_found ->
                     raise (Error (file, "file not found in path"))
               in
                 try Dynlink.loadfile fname
                 with
                 | Dynlink.Error e ->
                     raise (Error (fname, (Dynlink.error_message e))))
          
        let is_native = Dynlink.is_native
          
      end
      
    module EmptyError : sig include Sig.Error
                               end =
      struct
        type t = unit
        
        exception E of t
          
        let print _ = assert false
          
        let to_string _ = assert false
          
      end
      
    module EmptyPrinter :
      sig module Make (Ast : Sig.Ast) : Sig.Printer(Ast).S
             end =
      struct
        module Make (Ast : Sig.Ast) =
          struct
            let print_interf ?input_file:(_) ?output_file:(_) _ =
              failwith "No interface printer"
              
            let print_implem ?input_file:(_) ?output_file:(_) _ =
              failwith "No implementation printer"
              
          end
          
      end
      
    module FreeVars :
      sig
        module Make (Ast : Sig.Camlp4Ast) :
          sig
            module S : Set.S with type elt = string
              
            val fold_binding_vars :
              (string -> 'accu -> 'accu) -> Ast.binding -> 'accu -> 'accu
              
            class ['accu] c_fold_pattern_vars :
              (string -> 'accu -> 'accu) ->
                'accu ->
                  object inherit Ast.fold val acc : 'accu method acc : 'accu
                  end
              
            val fold_pattern_vars :
              (string -> 'accu -> 'accu) -> Ast.patt -> 'accu -> 'accu
              
            class ['accu] fold_free_vars :
              (string -> 'accu -> 'accu) ->
                ?env_init: S.t ->
                  'accu ->
                    object ('self_type)
                      inherit Ast.fold
                      val free : 'accu
                      val env : S.t
                      method free : 'accu
                      method set_env : S.t -> 'self_type
                      method add_atom : string -> 'self_type
                      method add_patt : Ast.patt -> 'self_type
                      method add_binding : Ast.binding -> 'self_type
                    end
              
            val free_vars : S.t -> Ast.expr -> S.t
              
          end
          
      end =
      struct
        module Make (Ast : Sig.Camlp4Ast) =
          struct
            module S = Set.Make(String)
              
            class ['accu] c_fold_pattern_vars f init =
              object inherit Ast.fold as super
                val acc = init
                method acc : 'accu = acc
                method patt =
                  function
                  | Ast.PaId (_, (Ast.IdLid (_, s))) |
                      Ast.PaLab (_, s, (Ast.PaNil _)) |
                      Ast.PaOlb (_, s, (Ast.PaNil _)) -> {< acc = f s acc; >}
                  | p -> super#patt p
              end
              
            let fold_pattern_vars f p init =
              ((new c_fold_pattern_vars f init)#patt p)#acc
              
            let rec fold_binding_vars f bi acc =
              match bi with
              | Ast.BiAnd (_, bi1, bi2) ->
                  fold_binding_vars f bi1 (fold_binding_vars f bi2 acc)
              | Ast.BiEq (_, p, _) -> fold_pattern_vars f p acc
              | Ast.BiNil _ -> acc
              | Ast.BiAnt (_, _) -> assert false
              
            class ['accu] fold_free_vars (f : string -> 'accu -> 'accu)
                    ?(env_init = S.empty) free_init =
              object (o)
                inherit Ast.fold as super
                val free = (free_init : 'accu)
                val env = (env_init : S.t)
                method free = free
                method set_env = fun env -> {< env = env; >}
                method add_atom = fun s -> {< env = S.add s env; >}
                method add_patt =
                  fun p -> {< env = fold_pattern_vars S.add p env; >}
                method add_binding =
                  fun bi -> {< env = fold_binding_vars S.add bi env; >}
                method expr =
                  function
                  | Ast.ExId (_, (Ast.IdLid (_, s))) |
                      Ast.ExLab (_, s, (Ast.ExNil _)) |
                      Ast.ExOlb (_, s, (Ast.ExNil _)) ->
                      if S.mem s env then o else {< free = f s free; >}
                  | Ast.ExLet (_, Ast.ReNil, bi, e) ->
                      (((o#add_binding bi)#expr e)#set_env env)#binding bi
                  | Ast.ExLet (_, Ast.ReRecursive, bi, e) ->
                      (((o#add_binding bi)#expr e)#binding bi)#set_env env
                  | Ast.ExFor (_, p, e1, e2, _, e3) ->
                      ((((o#expr e1)#expr e2)#patt p)#expr e3)#set_env env
                  | Ast.ExId (_, _) | Ast.ExNew (_, _) -> o
                  | Ast.ExObj (_, p, cst) ->
                      ((o#add_patt p)#class_str_item cst)#set_env env
                  | e -> super#expr e
                method match_case =
                  function
                  | Ast.McArr (_, p, e1, e2) ->
                      (((o#add_patt p)#expr e1)#expr e2)#set_env env
                  | m -> super#match_case m
                method str_item =
                  function
                  | Ast.StExt (_, s, t, _) -> (o#ctyp t)#add_atom s
                  | Ast.StVal (_, Ast.ReNil, bi) ->
                      (o#binding bi)#add_binding bi
                  | Ast.StVal (_, Ast.ReRecursive, bi) ->
                      (o#add_binding bi)#binding bi
                  | st -> super#str_item st
                method class_expr =
                  function
                  | Ast.CeFun (_, p, ce) ->
                      ((o#add_patt p)#class_expr ce)#set_env env
                  | Ast.CeLet (_, Ast.ReNil, bi, ce) ->
                      (((o#binding bi)#add_binding bi)#class_expr ce)#set_env
                        env
                  | Ast.CeLet (_, Ast.ReRecursive, bi, ce) ->
                      (((o#add_binding bi)#binding bi)#class_expr ce)#set_env
                        env
                  | Ast.CeStr (_, p, cst) ->
                      ((o#add_patt p)#class_str_item cst)#set_env env
                  | ce -> super#class_expr ce
                method class_str_item =
                  function
                  | (Ast.CrInh (_, _, _, "") as cst) ->
                      super#class_str_item cst
                  | Ast.CrInh (_, _, ce, s) -> (o#class_expr ce)#add_atom s
                  | Ast.CrVal (_, s, _, _, e) -> (o#expr e)#add_atom s
                  | Ast.CrVvr (_, s, _, t) -> (o#ctyp t)#add_atom s
                  | cst -> super#class_str_item cst
                method module_expr =
                  function
                  | Ast.MeStr (_, st) -> (o#str_item st)#set_env env
                  | me -> super#module_expr me
              end
              
            let free_vars env_init e =
              let fold = new fold_free_vars S.add ~env_init S.empty
              in (fold#expr e)#free
              
          end
          
      end
      
    module Grammar =
      struct
        module Structure =
          struct
            open Sig.Grammar
              
            module type S =
              sig
                module Loc : Sig.Loc
                  
                module Token : Sig.Token with module Loc = Loc
                  
                module Lexer : Sig.Lexer with module Loc = Loc
                  and module Token = Token
                  
                module Action : Sig.Grammar.Action
                  
                type gram =
                  { gfilter : Token.Filter.t;
                    gkeywords : (string, int ref) Hashtbl.t;
                    glexer :
                      Loc.t -> char Stream.t -> (Token.t * Loc.t) Stream.t;
                    warning_verbose : bool ref; error_verbose : bool ref
                  }
                
                type token_info =
                  { prev_loc : Loc.t; cur_loc : Loc.t; prev_loc_only : bool
                  }
                
                type token_stream = (Token.t * token_info) Stream.t
                
                type efun = token_stream -> Action.t
                
                type token_pattern = ((Token.t -> bool) * string)
                
                type internal_entry =
                  { egram : gram; ename : string;
                    mutable estart : int -> efun;
                    mutable econtinue : int -> Loc.t -> Action.t -> efun;
                    mutable edesc : desc
                  }
                  and desc =
                  | Dlevels of level list
                  | Dparser of (token_stream -> Action.t)
                  and level =
                  { assoc : assoc; lname : string option; lsuffix : tree;
                    lprefix : tree
                  }
                  and symbol =
                  | Smeta of string * symbol list * Action.t
                  | Snterm of internal_entry
                  | Snterml of internal_entry * string
                  | Slist0 of symbol
                  | Slist0sep of symbol * symbol
                  | Slist1 of symbol
                  | Slist1sep of symbol * symbol
                  | Sopt of symbol
                  | Stry of symbol
                  | Sself
                  | Snext
                  | Stoken of token_pattern
                  | Skeyword of string
                  | Stree of tree
                  and tree =
                  | Node of node
                  | LocAct of Action.t * Action.t list
                  | DeadEnd
                  and node =
                  { node : symbol; son : tree; brother : tree
                  }
                
                type production_rule = ((symbol list) * Action.t)
                
                type single_extend_statment =
                  ((string option) * (assoc option) * (production_rule list))
                
                type extend_statment =
                  ((position option) * (single_extend_statment list))
                
                type delete_statment = symbol list
                
                type ('a, 'b, 'c) fold =
                  internal_entry ->
                    symbol list -> ('a Stream.t -> 'b) -> 'a Stream.t -> 'c
                
                type ('a, 'b, 'c) foldsep =
                  internal_entry ->
                    symbol list ->
                      ('a Stream.t -> 'b) ->
                        ('a Stream.t -> unit) -> 'a Stream.t -> 'c
                
                val get_filter : gram -> Token.Filter.t
                  
                val using : gram -> string -> unit
                  
                val removing : gram -> string -> unit
                  
              end
              
            module Make (Lexer : Sig.Lexer) =
              struct
                module Loc = Lexer.Loc
                  
                module Token = Lexer.Token
                  
                module Action : Sig.Grammar.Action =
                  struct
                    type t = Obj.t
                    
                    let mk = Obj.repr
                      
                    let get = Obj.obj
                      
                    let getf = Obj.obj
                      
                    let getf2 = Obj.obj
                      
                  end
                  
                module Lexer = Lexer
                  
                type gram =
                  { gfilter : Token.Filter.t;
                    gkeywords : (string, int ref) Hashtbl.t;
                    glexer :
                      Loc.t -> char Stream.t -> (Token.t * Loc.t) Stream.t;
                    warning_verbose : bool ref; error_verbose : bool ref
                  }
                
                type token_info =
                  { prev_loc : Loc.t; cur_loc : Loc.t; prev_loc_only : bool
                  }
                
                type token_stream = (Token.t * token_info) Stream.t
                
                type efun = token_stream -> Action.t
                
                type token_pattern = ((Token.t -> bool) * string)
                
                type internal_entry =
                  { egram : gram; ename : string;
                    mutable estart : int -> efun;
                    mutable econtinue : int -> Loc.t -> Action.t -> efun;
                    mutable edesc : desc
                  }
                  and desc =
                  | Dlevels of level list
                  | Dparser of (token_stream -> Action.t)
                  and level =
                  { assoc : assoc; lname : string option; lsuffix : tree;
                    lprefix : tree
                  }
                  and symbol =
                  | Smeta of string * symbol list * Action.t
                  | Snterm of internal_entry
                  | Snterml of internal_entry * string
                  | Slist0 of symbol
                  | Slist0sep of symbol * symbol
                  | Slist1 of symbol
                  | Slist1sep of symbol * symbol
                  | Sopt of symbol
                  | Stry of symbol
                  | Sself
                  | Snext
                  | Stoken of token_pattern
                  | Skeyword of string
                  | Stree of tree
                  and tree =
                  | Node of node
                  | LocAct of Action.t * Action.t list
                  | DeadEnd
                  and node =
                  { node : symbol; son : tree; brother : tree
                  }
                
                type production_rule = ((symbol list) * Action.t)
                
                type single_extend_statment =
                  ((string option) * (assoc option) * (production_rule list))
                
                type extend_statment =
                  ((position option) * (single_extend_statment list))
                
                type delete_statment = symbol list
                
                type ('a, 'b, 'c) fold =
                  internal_entry ->
                    symbol list -> ('a Stream.t -> 'b) -> 'a Stream.t -> 'c
                
                type ('a, 'b, 'c) foldsep =
                  internal_entry ->
                    symbol list ->
                      ('a Stream.t -> 'b) ->
                        ('a Stream.t -> unit) -> 'a Stream.t -> 'c
                
                let get_filter g = g.gfilter
                  
                let token_location r = r.cur_loc
                  
                type 'a not_filtered = 'a
                
                let using { gkeywords = table; gfilter = filter } kwd =
                  let r =
                    try Hashtbl.find table kwd
                    with
                    | Not_found ->
                        let r = ref 0 in (Hashtbl.add table kwd r; r)
                  in (Token.Filter.keyword_added filter kwd (!r = 0); incr r)
                  
                let removing { gkeywords = table; gfilter = filter } kwd =
                  let r = Hashtbl.find table kwd in
                  let () = decr r
                  in
                    if !r = 0
                    then
                      (Token.Filter.keyword_removed filter kwd;
                       Hashtbl.remove table kwd)
                    else ()
                  
              end
              
          end
          
        module Search =
          struct
            module Make (Structure : Structure.S) =
              struct
                open Structure
                  
                let tree_in_entry prev_symb tree =
                  function
                  | Dlevels levels ->
                      let rec search_levels =
                        (function
                         | [] -> tree
                         | level :: levels ->
                             (match search_level level with
                              | Some tree -> tree
                              | None -> search_levels levels))
                      and search_level level =
                        (match search_tree level.lsuffix with
                         | Some t ->
                             Some
                               (Node
                                  { node = Sself; son = t; brother = DeadEnd;
                                  })
                         | None -> search_tree level.lprefix)
                      and search_tree t =
                        if (tree <> DeadEnd) && (t == tree)
                        then Some t
                        else
                          (match t with
                           | Node n ->
                               (match search_symbol n.node with
                                | Some symb ->
                                    Some
                                      (Node
                                         {
                                           node = symb;
                                           son = n.son;
                                           brother = DeadEnd;
                                         })
                                | None ->
                                    (match search_tree n.son with
                                     | Some t ->
                                         Some
                                           (Node
                                              {
                                                node = n.node;
                                                son = t;
                                                brother = DeadEnd;
                                              })
                                     | None -> search_tree n.brother))
                           | LocAct (_, _) | DeadEnd -> None)
                      and search_symbol symb =
                        (match symb with
                         | Snterm _ | Snterml (_, _) | Slist0 _ |
                             Slist0sep (_, _) | Slist1 _ | Slist1sep (_, _) |
                             Sopt _ | Stry _ | Stoken _ | Stree _ |
                             Skeyword _ when symb == prev_symb -> Some symb
                         | Slist0 symb ->
                             (match search_symbol symb with
                              | Some symb -> Some (Slist0 symb)
                              | None -> None)
                         | Slist0sep (symb, sep) ->
                             (match search_symbol symb with
                              | Some symb -> Some (Slist0sep (symb, sep))
                              | None ->
                                  (match search_symbol sep with
                                   | Some sep -> Some (Slist0sep (symb, sep))
                                   | None -> None))
                         | Slist1 symb ->
                             (match search_symbol symb with
                              | Some symb -> Some (Slist1 symb)
                              | None -> None)
                         | Slist1sep (symb, sep) ->
                             (match search_symbol symb with
                              | Some symb -> Some (Slist1sep (symb, sep))
                              | None ->
                                  (match search_symbol sep with
                                   | Some sep -> Some (Slist1sep (symb, sep))
                                   | None -> None))
                         | Sopt symb ->
                             (match search_symbol symb with
                              | Some symb -> Some (Sopt symb)
                              | None -> None)
                         | Stry symb ->
                             (match search_symbol symb with
                              | Some symb -> Some (Stry symb)
                              | None -> None)
                         | Stree t ->
                             (match search_tree t with
                              | Some t -> Some (Stree t)
                              | None -> None)
                         | _ -> None)
                      in search_levels levels
                  | Dparser _ -> tree
                  
              end
              
          end
          
        module Tools =
          struct
            let get_prev_loc_only = ref false
              
            module Make (Structure : Structure.S) =
              struct
                open Structure
                  
                let empty_entry ename _ =
                  raise (Stream.Error ("entry [" ^ (ename ^ "] is empty")))
                  
                let rec stream_map f (__strm : _ Stream.t) =
                  match Stream.peek __strm with
                  | Some x ->
                      (Stream.junk __strm;
                       let strm = __strm
                       in
                         Stream.lcons (fun _ -> f x)
                           (Stream.slazy (fun _ -> stream_map f strm)))
                  | _ -> Stream.sempty
                  
                let keep_prev_loc strm =
                  match Stream.peek strm with
                  | None -> Stream.sempty
                  | Some ((tok0, init_loc)) ->
                      let rec go prev_loc strm1 =
                        if !get_prev_loc_only
                        then
                          Stream.lcons
                            (fun _ ->
                               (tok0,
                                {
                                  prev_loc = prev_loc;
                                  cur_loc = prev_loc;
                                  prev_loc_only = true;
                                }))
                            (Stream.slazy (fun _ -> go prev_loc strm1))
                        else
                          (let (__strm : _ Stream.t) = strm1
                           in
                             match Stream.peek __strm with
                             | Some ((tok, cur_loc)) ->
                                 (Stream.junk __strm;
                                  let strm = __strm
                                  in
                                    Stream.lcons
                                      (fun _ ->
                                         (tok,
                                          {
                                            prev_loc = prev_loc;
                                            cur_loc = cur_loc;
                                            prev_loc_only = false;
                                          }))
                                      (Stream.slazy
                                         (fun _ -> go cur_loc strm)))
                             | _ -> Stream.sempty)
                      in go init_loc strm
                  
                let drop_prev_loc strm =
                  stream_map (fun (tok, r) -> (tok, (r.cur_loc))) strm
                  
                let get_cur_loc strm =
                  match Stream.peek strm with
                  | Some ((_, r)) -> r.cur_loc
                  | None -> Loc.ghost
                  
                let get_prev_loc strm =
                  (get_prev_loc_only := true;
                   let result =
                     match Stream.peek strm with
                     | Some
                         ((_, { prev_loc = prev_loc; prev_loc_only = true }))
                         -> (Stream.junk strm; prev_loc)
                     | Some
                         ((_, { prev_loc = prev_loc; prev_loc_only = false }))
                         -> prev_loc
                     | None -> Loc.ghost
                   in (get_prev_loc_only := false; result))
                  
                let is_level_labelled n lev =
                  match lev.lname with | Some n1 -> n = n1 | None -> false
                  
                let warning_verbose = ref true
                  
                let rec get_token_list entry tokl last_tok tree =
                  match tree with
                  | Node
                      {
                        node = (Stoken _ | Skeyword _ as tok);
                        son = son;
                        brother = DeadEnd
                      } -> get_token_list entry (last_tok :: tokl) tok son
                  | _ ->
                      if tokl = []
                      then None
                      else
                        Some
                          (((List.rev (last_tok :: tokl)), last_tok, tree))
                  
                let is_antiquot s =
                  let len = String.length s in (len > 1) && (s.[0] = '$')
                  
                let eq_Stoken_ids s1 s2 =
                  (not (is_antiquot s1)) &&
                    ((not (is_antiquot s2)) && (s1 = s2))
                  
                let logically_eq_symbols entry =
                  let rec eq_symbols s1 s2 =
                    match (s1, s2) with
                    | (Snterm e1, Snterm e2) -> e1.ename = e2.ename
                    | (Snterm e1, Sself) -> e1.ename = entry.ename
                    | (Sself, Snterm e2) -> entry.ename = e2.ename
                    | (Snterml (e1, l1), Snterml (e2, l2)) ->
                        (e1.ename = e2.ename) && (l1 = l2)
                    | (Slist0 s1, Slist0 s2) | (Slist1 s1, Slist1 s2) |
                        (Sopt s1, Sopt s2) | (Stry s1, Stry s2) ->
                        eq_symbols s1 s2
                    | (Slist0sep (s1, sep1), Slist0sep (s2, sep2)) |
                        (Slist1sep (s1, sep1), Slist1sep (s2, sep2)) ->
                        (eq_symbols s1 s2) && (eq_symbols sep1 sep2)
                    | (Stree t1, Stree t2) -> eq_trees t1 t2
                    | (Stoken ((_, s1)), Stoken ((_, s2))) ->
                        eq_Stoken_ids s1 s2
                    | _ -> s1 = s2
                  and eq_trees t1 t2 =
                    match (t1, t2) with
                    | (Node n1, Node n2) ->
                        (eq_symbols n1.node n2.node) &&
                          ((eq_trees n1.son n2.son) &&
                             (eq_trees n1.brother n2.brother))
                    | ((LocAct (_, _) | DeadEnd), (LocAct (_, _) | DeadEnd))
                        -> true
                    | _ -> false
                  in eq_symbols
                  
                let rec eq_symbol s1 s2 =
                  match (s1, s2) with
                  | (Snterm e1, Snterm e2) -> e1 == e2
                  | (Snterml (e1, l1), Snterml (e2, l2)) ->
                      (e1 == e2) && (l1 = l2)
                  | (Slist0 s1, Slist0 s2) | (Slist1 s1, Slist1 s2) |
                      (Sopt s1, Sopt s2) | (Stry s1, Stry s2) ->
                      eq_symbol s1 s2
                  | (Slist0sep (s1, sep1), Slist0sep (s2, sep2)) |
                      (Slist1sep (s1, sep1), Slist1sep (s2, sep2)) ->
                      (eq_symbol s1 s2) && (eq_symbol sep1 sep2)
                  | (Stree _, Stree _) -> false
                  | (Stoken ((_, s1)), Stoken ((_, s2))) ->
                      eq_Stoken_ids s1 s2
                  | _ -> s1 = s2
                  
              end
              
          end
          
        module Print :
          sig
            module Make (Structure : Structure.S) :
              sig
                val flatten_tree :
                  Structure.tree -> (Structure.symbol list) list
                  
                val print_symbol :
                  Format.formatter -> Structure.symbol -> unit
                  
                val print_meta :
                  Format.formatter -> string -> Structure.symbol list -> unit
                  
                val print_symbol1 :
                  Format.formatter -> Structure.symbol -> unit
                  
                val print_rule :
                  Format.formatter -> Structure.symbol list -> unit
                  
                val print_level :
                  Format.formatter ->
                    (Format.formatter -> unit -> unit) ->
                      (Structure.symbol list) list -> unit
                  
                val levels : Format.formatter -> Structure.level list -> unit
                  
                val entry :
                  Format.formatter -> Structure.internal_entry -> unit
                  
              end
              
            module MakeDump (Structure : Structure.S) :
              sig
                val print_symbol :
                  Format.formatter -> Structure.symbol -> unit
                  
                val print_meta :
                  Format.formatter -> string -> Structure.symbol list -> unit
                  
                val print_symbol1 :
                  Format.formatter -> Structure.symbol -> unit
                  
                val print_rule :
                  Format.formatter -> Structure.symbol list -> unit
                  
                val print_level :
                  Format.formatter ->
                    (Format.formatter -> unit -> unit) ->
                      (Structure.symbol list) list -> unit
                  
                val levels : Format.formatter -> Structure.level list -> unit
                  
                val entry :
                  Format.formatter -> Structure.internal_entry -> unit
                  
              end
              
          end =
          struct
            module Make (Structure : Structure.S) =
              struct
                open Structure
                  
                open Format
                  
                open Sig.Grammar
                  
                let rec flatten_tree =
                  function
                  | DeadEnd -> []
                  | LocAct (_, _) -> [ [] ]
                  | Node { node = n; brother = b; son = s } ->
                      (List.map (fun l -> n :: l) (flatten_tree s)) @
                        (flatten_tree b)
                  
                let rec print_symbol ppf =
                  function
                  | Smeta (n, sl, _) -> print_meta ppf n sl
                  | Slist0 s -> fprintf ppf "LIST0 %a" print_symbol1 s
                  | Slist0sep (s, t) ->
                      fprintf ppf "LIST0 %a SEP %a" print_symbol1 s
                        print_symbol1 t
                  | Slist1 s -> fprintf ppf "LIST1 %a" print_symbol1 s
                  | Slist1sep (s, t) ->
                      fprintf ppf "LIST1 %a SEP %a" print_symbol1 s
                        print_symbol1 t
                  | Sopt s -> fprintf ppf "OPT %a" print_symbol1 s
                  | Stry s -> fprintf ppf "TRY %a" print_symbol1 s
                  | Snterml (e, l) -> fprintf ppf "%s@ LEVEL@ %S" e.ename l
                  | (Snterm _ | Snext | Sself | Stree _ | Stoken _ |
                       Skeyword _
                     as s) -> print_symbol1 ppf s
                and print_meta ppf n sl =
                  let rec loop i =
                    function
                    | [] -> ()
                    | s :: sl ->
                        let j =
                          (try String.index_from n i ' '
                           with | Not_found -> String.length n)
                        in
                          (fprintf ppf "%s %a" (String.sub n i (j - i))
                             print_symbol1 s;
                           if sl = []
                           then ()
                           else
                             (fprintf ppf " ";
                              loop (min (j + 1) (String.length n)) sl))
                  in loop 0 sl
                and print_symbol1 ppf =
                  function
                  | Snterm e -> pp_print_string ppf e.ename
                  | Sself -> pp_print_string ppf "SELF"
                  | Snext -> pp_print_string ppf "NEXT"
                  | Stoken ((_, descr)) -> pp_print_string ppf descr
                  | Skeyword s -> fprintf ppf "%S" s
                  | Stree t ->
                      print_level ppf pp_print_space (flatten_tree t)
                  | (Smeta (_, _, _) | Snterml (_, _) | Slist0 _ |
                       Slist0sep (_, _) | Slist1 _ | Slist1sep (_, _) |
                       Sopt _ | Stry _
                     as s) -> fprintf ppf "(%a)" print_symbol s
                and print_rule ppf symbols =
                  (fprintf ppf "@[<hov 0>";
                   let _ =
                     List.fold_left
                       (fun sep symbol ->
                          (fprintf ppf "%t%a" sep print_symbol symbol;
                           fun ppf -> fprintf ppf ";@ "))
                       (fun _ -> ()) symbols
                   in fprintf ppf "@]")
                and print_level ppf pp_print_space rules =
                  (fprintf ppf "@[<hov 0>[ ";
                   let _ =
                     List.fold_left
                       (fun sep rule ->
                          (fprintf ppf "%t%a" sep print_rule rule;
                           fun ppf -> fprintf ppf "%a| " pp_print_space ()))
                       (fun _ -> ()) rules
                   in fprintf ppf " ]@]")
                  
                let levels ppf elev =
                  let _ =
                    List.fold_left
                      (fun sep lev ->
                         let rules =
                           (List.map (fun t -> Sself :: t)
                              (flatten_tree lev.lsuffix))
                             @ (flatten_tree lev.lprefix)
                         in
                           (fprintf ppf "%t@[<hov 2>" sep;
                            (match lev.lname with
                             | Some n -> fprintf ppf "%S@;<1 2>" n
                             | None -> ());
                            (match lev.assoc with
                             | LeftA -> fprintf ppf "LEFTA"
                             | RightA -> fprintf ppf "RIGHTA"
                             | NonA -> fprintf ppf "NONA");
                            fprintf ppf "@]@;<1 2>";
                            print_level ppf pp_force_newline rules;
                            fun ppf -> fprintf ppf "@,| "))
                      (fun _ -> ()) elev
                  in ()
                  
                let entry ppf e =
                  (fprintf ppf "@[<v 0>%s: [ " e.ename;
                   (match e.edesc with
                    | Dlevels elev -> levels ppf elev
                    | Dparser _ -> fprintf ppf "<parser>");
                   fprintf ppf " ]@]")
                  
              end
              
            module MakeDump (Structure : Structure.S) =
              struct
                open Structure
                  
                open Format
                  
                open Sig.Grammar
                  
                type brothers = | Bro of symbol * brothers list
                
                let rec print_tree ppf tree =
                  let rec get_brothers acc =
                    function
                    | DeadEnd -> List.rev acc
                    | LocAct (_, _) -> List.rev acc
                    | Node { node = n; brother = b; son = s } ->
                        get_brothers ((Bro (n, (get_brothers [] s))) :: acc)
                          b
                  and print_brothers ppf brothers =
                    if brothers = []
                    then fprintf ppf "@ []"
                    else
                      List.iter
                        (function
                         | Bro (n, xs) ->
                             (fprintf ppf "@ @[<hv2>- %a" print_symbol n;
                              (match xs with
                               | [] -> ()
                               | [ _ ] ->
                                   (try
                                      print_children ppf (get_children [] xs)
                                    with
                                    | Exit ->
                                        fprintf ppf ":%a" print_brothers xs)
                               | _ -> fprintf ppf ":%a" print_brothers xs);
                              fprintf ppf "@]"))
                        brothers
                  and print_children ppf =
                    List.iter (fprintf ppf ";@ %a" print_symbol)
                  and get_children acc =
                    function
                    | [] -> List.rev acc
                    | [ Bro (n, x) ] -> get_children (n :: acc) x
                    | _ -> raise Exit
                  in print_brothers ppf (get_brothers [] tree)
                and print_symbol ppf =
                  function
                  | Smeta (n, sl, _) -> print_meta ppf n sl
                  | Slist0 s -> fprintf ppf "LIST0 %a" print_symbol1 s
                  | Slist0sep (s, t) ->
                      fprintf ppf "LIST0 %a SEP %a" print_symbol1 s
                        print_symbol1 t
                  | Slist1 s -> fprintf ppf "LIST1 %a" print_symbol1 s
                  | Slist1sep (s, t) ->
                      fprintf ppf "LIST1 %a SEP %a" print_symbol1 s
                        print_symbol1 t
                  | Sopt s -> fprintf ppf "OPT %a" print_symbol1 s
                  | Stry s -> fprintf ppf "TRY %a" print_symbol1 s
                  | Snterml (e, l) -> fprintf ppf "%s@ LEVEL@ %S" e.ename l
                  | (Snterm _ | Snext | Sself | Stree _ | Stoken _ |
                       Skeyword _
                     as s) -> print_symbol1 ppf s
                and print_meta ppf n sl =
                  let rec loop i =
                    function
                    | [] -> ()
                    | s :: sl ->
                        let j =
                          (try String.index_from n i ' '
                           with | Not_found -> String.length n)
                        in
                          (fprintf ppf "%s %a" (String.sub n i (j - i))
                             print_symbol1 s;
                           if sl = []
                           then ()
                           else
                             (fprintf ppf " ";
                              loop (min (j + 1) (String.length n)) sl))
                  in loop 0 sl
                and print_symbol1 ppf =
                  function
                  | Snterm e -> pp_print_string ppf e.ename
                  | Sself -> pp_print_string ppf "SELF"
                  | Snext -> pp_print_string ppf "NEXT"
                  | Stoken ((_, descr)) -> pp_print_string ppf descr
                  | Skeyword s -> fprintf ppf "%S" s
                  | Stree t -> print_tree ppf t
                  | (Smeta (_, _, _) | Snterml (_, _) | Slist0 _ |
                       Slist0sep (_, _) | Slist1 _ | Slist1sep (_, _) |
                       Sopt _ | Stry _
                     as s) -> fprintf ppf "(%a)" print_symbol s
                and print_rule ppf symbols =
                  (fprintf ppf "@[<hov 0>";
                   let _ =
                     List.fold_left
                       (fun sep symbol ->
                          (fprintf ppf "%t%a" sep print_symbol symbol;
                           fun ppf -> fprintf ppf ";@ "))
                       (fun _ -> ()) symbols
                   in fprintf ppf "@]")
                and print_level ppf pp_print_space rules =
                  (fprintf ppf "@[<hov 0>[ ";
                   let _ =
                     List.fold_left
                       (fun sep rule ->
                          (fprintf ppf "%t%a" sep print_rule rule;
                           fun ppf -> fprintf ppf "%a| " pp_print_space ()))
                       (fun _ -> ()) rules
                   in fprintf ppf " ]@]")
                  
                let levels ppf elev =
                  let _ =
                    List.fold_left
                      (fun sep lev ->
                         (fprintf ppf "%t@[<v2>" sep;
                          (match lev.lname with
                           | Some n -> fprintf ppf "%S@;<1 2>" n
                           | None -> ());
                          (match lev.assoc with
                           | LeftA -> fprintf ppf "LEFTA"
                           | RightA -> fprintf ppf "RIGHTA"
                           | NonA -> fprintf ppf "NONA");
                          fprintf ppf "@]@;<1 2>";
                          fprintf ppf "@[<v2>suffix:@ ";
                          print_tree ppf lev.lsuffix;
                          fprintf ppf "@]@ @[<v2>prefix:@ ";
                          print_tree ppf lev.lprefix;
                          fprintf ppf "@]";
                          fun ppf -> fprintf ppf "@,| "))
                      (fun _ -> ()) elev
                  in ()
                  
                let entry ppf e =
                  (fprintf ppf "@[<v 0>%s: [ " e.ename;
                   (match e.edesc with
                    | Dlevels elev -> levels ppf elev
                    | Dparser _ -> fprintf ppf "<parser>");
                   fprintf ppf " ]@]")
                  
              end
              
          end
          
        module Failed =
          struct
            module Make (Structure : Structure.S) =
              struct
                module Tools = Tools.Make(Structure)
                  
                module Search = Search.Make(Structure)
                  
                module Print = Print.Make(Structure)
                  
                open Structure
                  
                open Format
                  
                let rec name_of_symbol entry =
                  function
                  | Snterm e -> "[" ^ (e.ename ^ "]")
                  | Snterml (e, l) ->
                      "[" ^ (e.ename ^ (" level " ^ (l ^ "]")))
                  | Sself | Snext -> "[" ^ (entry.ename ^ "]")
                  | Stoken ((_, descr)) -> descr
                  | Skeyword kwd -> "\"" ^ (kwd ^ "\"")
                  | _ -> "???"
                  
                let rec name_of_symbol_failed entry =
                  function
                  | Slist0 s | Slist0sep (s, _) | Slist1 s | Slist1sep (s, _)
                      | Sopt s | Stry s -> name_of_symbol_failed entry s
                  | Stree t -> name_of_tree_failed entry t
                  | s -> name_of_symbol entry s
                and name_of_tree_failed entry =
                  function
                  | Node { node = s; brother = bro; son = son } ->
                      let tokl =
                        (match s with
                         | Stoken _ | Skeyword _ ->
                             Tools.get_token_list entry [] s son
                         | _ -> None)
                      in
                        (match tokl with
                         | None ->
                             let txt = name_of_symbol_failed entry s in
                             let txt =
                               (match (s, son) with
                                | (Sopt _, Node _) ->
                                    txt ^
                                      (" or " ^
                                         (name_of_tree_failed entry son))
                                | _ -> txt) in
                             let txt =
                               (match bro with
                                | DeadEnd | LocAct (_, _) -> txt
                                | Node _ ->
                                    txt ^
                                      (" or " ^
                                         (name_of_tree_failed entry bro)))
                             in txt
                         | Some ((tokl, _, _)) ->
                             List.fold_left
                               (fun s tok ->
                                  (if s = "" then "" else s ^ " then ") ^
                                    (match tok with
                                     | Stoken ((_, descr)) -> descr
                                     | Skeyword kwd -> kwd
                                     | _ -> assert false))
                               "" tokl)
                  | DeadEnd | LocAct (_, _) -> "???"
                  
                let magic _s x = Obj.magic x
                  
                let tree_failed entry prev_symb_result prev_symb tree =
                  let txt = name_of_tree_failed entry tree in
                  let txt =
                    match prev_symb with
                    | Slist0 s ->
                        let txt1 = name_of_symbol_failed entry s
                        in txt1 ^ (" or " ^ (txt ^ " expected"))
                    | Slist1 s ->
                        let txt1 = name_of_symbol_failed entry s
                        in txt1 ^ (" or " ^ (txt ^ " expected"))
                    | Slist0sep (s, sep) ->
                        (match magic "tree_failed: 'a -> list 'b"
                                 prev_symb_result
                         with
                         | [] ->
                             let txt1 = name_of_symbol_failed entry s
                             in txt1 ^ (" or " ^ (txt ^ " expected"))
                         | _ ->
                             let txt1 = name_of_symbol_failed entry sep
                             in txt1 ^ (" or " ^ (txt ^ " expected")))
                    | Slist1sep (s, sep) ->
                        (match magic "tree_failed: 'a -> list 'b"
                                 prev_symb_result
                         with
                         | [] ->
                             let txt1 = name_of_symbol_failed entry s
                             in txt1 ^ (" or " ^ (txt ^ " expected"))
                         | _ ->
                             let txt1 = name_of_symbol_failed entry sep
                             in txt1 ^ (" or " ^ (txt ^ " expected")))
                    | Stry _ | Sopt _ | Stree _ -> txt ^ " expected"
                    | _ ->
                        txt ^
                          (" expected after " ^
                             (name_of_symbol entry prev_symb))
                  in
                    (if !(entry.egram.error_verbose)
                     then
                       (let tree =
                          Search.tree_in_entry prev_symb tree entry.edesc in
                        let ppf = err_formatter
                        in
                          (fprintf ppf "@[<v 0>@,";
                           fprintf ppf "----------------------------------@,";
                           fprintf ppf
                             "Parse error in entry [%s], rule:@;<0 2>"
                             entry.ename;
                           fprintf ppf "@[";
                           Print.print_level ppf pp_force_newline
                             (Print.flatten_tree tree);
                           fprintf ppf "@]@,";
                           fprintf ppf "----------------------------------@,";
                           fprintf ppf "@]@."))
                     else ();
                     txt ^ (" (in [" ^ (entry.ename ^ "])")))
                  
                let symb_failed entry prev_symb_result prev_symb symb =
                  let tree =
                    Node { node = symb; brother = DeadEnd; son = DeadEnd; }
                  in tree_failed entry prev_symb_result prev_symb tree
                  
                let symb_failed_txt e s1 s2 = symb_failed e 0 s1 s2
                  
              end
              
          end
          
        module Parser =
          struct
            module Make (Structure : Structure.S) =
              struct
                module Tools = Tools.Make(Structure)
                  
                module Failed = Failed.Make(Structure)
                  
                module Print = Print.Make(Structure)
                  
                open Structure
                  
                open Sig.Grammar
                  
                module StreamOrig = Stream
                  
                let njunk strm n = for i = 1 to n do Stream.junk strm done
                  
                let loc_bp = Tools.get_cur_loc
                  
                let loc_ep = Tools.get_prev_loc
                  
                let drop_prev_loc = Tools.drop_prev_loc
                  
                let add_loc bp parse_fun strm =
                  let x = parse_fun strm in
                  let ep = loc_ep strm in
                  let loc =
                    if (Loc.start_off bp) > (Loc.stop_off ep)
                    then Loc.join bp
                    else Loc.merge bp ep
                  in (x, loc)
                  
                let stream_peek_nth strm n =
                  let rec loop i =
                    function
                    | x :: xs -> if i = 1 then Some x else loop (i - 1) xs
                    | [] -> None
                  in loop n (Stream.npeek n strm)
                  
                module Stream =
                  struct
                    type 'a t = 'a StreamOrig.t
                    
                    exception Failure = StreamOrig.Failure
                      
                    exception Error = StreamOrig.Error
                      
                    let peek = StreamOrig.peek
                      
                    let junk = StreamOrig.junk
                      
                    let dup strm =
                      let peek_nth n =
                        let rec loop n =
                          function
                          | [] -> None
                          | [ x ] -> if n = 0 then Some x else None
                          | _ :: l -> loop (n - 1) l
                        in loop n (Stream.npeek (n + 1) strm)
                      in Stream.from peek_nth
                      
                  end
                  
                let try_parser ps strm =
                  let strm' = Stream.dup strm in
                  let r =
                    try ps strm'
                    with
                    | Stream.Error _ | Loc.Exc_located (_, (Stream.Error _))
                        -> raise Stream.Failure
                    | exc -> raise exc
                  in (njunk strm (StreamOrig.count strm'); r)
                  
                let level_number entry lab =
                  let rec lookup levn =
                    function
                    | [] -> failwith ("unknown level " ^ lab)
                    | lev :: levs ->
                        if Tools.is_level_labelled lab lev
                        then levn
                        else lookup (succ levn) levs
                  in
                    match entry.edesc with
                    | Dlevels elev -> lookup 0 elev
                    | Dparser _ -> raise Not_found
                  
                let strict_parsing = ref false
                  
                let strict_parsing_warning = ref false
                  
                let rec top_symb entry =
                  function
                  | Sself | Snext -> Snterm entry
                  | Snterml (e, _) -> Snterm e
                  | Slist1sep (s, sep) -> Slist1sep ((top_symb entry s), sep)
                  | _ -> raise Stream.Failure
                  
                let top_tree entry =
                  function
                  | Node { node = s; brother = bro; son = son } ->
                      Node
                        { node = top_symb entry s; brother = bro; son = son;
                        }
                  | LocAct (_, _) | DeadEnd -> raise Stream.Failure
                  
                let entry_of_symb entry =
                  function
                  | Sself | Snext -> entry
                  | Snterm e -> e
                  | Snterml (e, _) -> e
                  | _ -> raise Stream.Failure
                  
                let continue entry loc a s son p1 (__strm : _ Stream.t) =
                  let a = (entry_of_symb entry s).econtinue 0 loc a __strm in
                  let act =
                    try p1 __strm
                    with
                    | Stream.Failure ->
                        raise
                          (Stream.Error (Failed.tree_failed entry a s son))
                  in Action.mk (fun _ -> Action.getf act a)
                  
                let skip_if_empty bp strm =
                  if (loc_bp strm) = bp
                  then Action.mk (fun _ -> raise Stream.Failure)
                  else raise Stream.Failure
                  
                let do_recover parser_of_tree entry nlevn alevn loc a s son
                               (__strm : _ Stream.t) =
                  try
                    parser_of_tree entry nlevn alevn (top_tree entry son)
                      __strm
                  with
                  | Stream.Failure ->
                      (try skip_if_empty loc __strm
                       with
                       | Stream.Failure ->
                           continue entry loc a s son
                             (parser_of_tree entry nlevn alevn son) __strm)
                  
                let recover parser_of_tree entry nlevn alevn loc a s son strm
                            =
                  if !strict_parsing
                  then
                    raise (Stream.Error (Failed.tree_failed entry a s son))
                  else
                    (let _ =
                       if !strict_parsing_warning
                       then
                         (let msg = Failed.tree_failed entry a s son
                          in
                            (Format.eprintf
                               "Warning: trying to recover from syntax error";
                             if entry.ename <> ""
                             then Format.eprintf " in [%s]" entry.ename
                             else ();
                             Format.eprintf "\n%s%a@." msg Loc.print loc))
                       else ()
                     in
                       do_recover parser_of_tree entry nlevn alevn loc a s
                         son strm)
                  
                let rec parser_of_tree entry nlevn alevn =
                  function
                  | DeadEnd ->
                      (fun (__strm : _ Stream.t) -> raise Stream.Failure)
                  | LocAct (act, _) -> (fun (__strm : _ Stream.t) -> act)
                  | Node
                      {
                        node = Sself;
                        son = LocAct (act, _);
                        brother = DeadEnd
                      } ->
                      (fun (__strm : _ Stream.t) ->
                         let a = entry.estart alevn __strm
                         in Action.getf act a)
                  | Node { node = Sself; son = LocAct (act, _); brother = bro
                      } ->
                      let p2 = parser_of_tree entry nlevn alevn bro
                      in
                        (fun (__strm : _ Stream.t) ->
                           match try Some (entry.estart alevn __strm)
                                 with | Stream.Failure -> None
                           with
                           | Some a -> Action.getf act a
                           | _ -> p2 __strm)
                  | Node { node = s; son = son; brother = DeadEnd } ->
                      let tokl =
                        (match s with
                         | Stoken _ | Skeyword _ ->
                             Tools.get_token_list entry [] s son
                         | _ -> None)
                      in
                        (match tokl with
                         | None ->
                             let ps = parser_of_symbol entry nlevn s in
                             let p1 = parser_of_tree entry nlevn alevn son in
                             let p1 = parser_cont p1 entry nlevn alevn s son
                             in
                               (fun strm ->
                                  let bp = loc_bp strm in
                                  let (__strm : _ Stream.t) = strm in
                                  let a = ps __strm in
                                  let act =
                                    try p1 bp a __strm
                                    with
                                    | Stream.Failure ->
                                        raise (Stream.Error "")
                                  in Action.getf act a)
                         | Some ((tokl, last_tok, son)) ->
                             let p1 = parser_of_tree entry nlevn alevn son in
                             let p1 =
                               parser_cont p1 entry nlevn alevn last_tok son
                             in parser_of_token_list p1 tokl)
                  | Node { node = s; son = son; brother = bro } ->
                      let tokl =
                        (match s with
                         | Stoken _ | Skeyword _ ->
                             Tools.get_token_list entry [] s son
                         | _ -> None)
                      in
                        (match tokl with
                         | None ->
                             let ps = parser_of_symbol entry nlevn s in
                             let p1 = parser_of_tree entry nlevn alevn son in
                             let p1 =
                               parser_cont p1 entry nlevn alevn s son in
                             let p2 = parser_of_tree entry nlevn alevn bro
                             in
                               (fun strm ->
                                  let bp = loc_bp strm in
                                  let (__strm : _ Stream.t) = strm
                                  in
                                    match try Some (ps __strm)
                                          with | Stream.Failure -> None
                                    with
                                    | Some a ->
                                        let act =
                                          (try p1 bp a __strm
                                           with
                                           | Stream.Failure ->
                                               raise (Stream.Error ""))
                                        in Action.getf act a
                                    | _ -> p2 __strm)
                         | Some ((tokl, last_tok, son)) ->
                             let p1 = parser_of_tree entry nlevn alevn son in
                             let p1 =
                               parser_cont p1 entry nlevn alevn last_tok son in
                             let p1 = parser_of_token_list p1 tokl in
                             let p2 = parser_of_tree entry nlevn alevn bro
                             in
                               (fun (__strm : _ Stream.t) ->
                                  try p1 __strm
                                  with | Stream.Failure -> p2 __strm))
                and
                  parser_cont p1 entry nlevn alevn s son loc a
                              (__strm : _ Stream.t) =
                  try p1 __strm
                  with
                  | Stream.Failure ->
                      (try
                         recover parser_of_tree entry nlevn alevn loc a s son
                           __strm
                       with
                       | Stream.Failure ->
                           raise
                             (Stream.Error (Failed.tree_failed entry a s son)))
                and parser_of_token_list p1 tokl =
                  let rec loop n =
                    function
                    | Stoken ((tematch, _)) :: tokl ->
                        (match tokl with
                         | [] ->
                             let ps strm =
                               (match stream_peek_nth strm n with
                                | Some ((tok, _)) when tematch tok ->
                                    (njunk strm n; Action.mk tok)
                                | _ -> raise Stream.Failure)
                             in
                               (fun strm ->
                                  let bp = loc_bp strm in
                                  let (__strm : _ Stream.t) = strm in
                                  let a = ps __strm in
                                  let act =
                                    try p1 bp a __strm
                                    with
                                    | Stream.Failure ->
                                        raise (Stream.Error "")
                                  in Action.getf act a)
                         | _ ->
                             let ps strm =
                               (match stream_peek_nth strm n with
                                | Some ((tok, _)) when tematch tok -> tok
                                | _ -> raise Stream.Failure) in
                             let p1 = loop (n + 1) tokl
                             in
                               (fun (__strm : _ Stream.t) ->
                                  let tok = ps __strm in
                                  let s = __strm in
                                  let act = p1 s in Action.getf act tok))
                    | Skeyword kwd :: tokl ->
                        (match tokl with
                         | [] ->
                             let ps strm =
                               (match stream_peek_nth strm n with
                                | Some ((tok, _)) when
                                    Token.match_keyword kwd tok ->
                                    (njunk strm n; Action.mk tok)
                                | _ -> raise Stream.Failure)
                             in
                               (fun strm ->
                                  let bp = loc_bp strm in
                                  let (__strm : _ Stream.t) = strm in
                                  let a = ps __strm in
                                  let act =
                                    try p1 bp a __strm
                                    with
                                    | Stream.Failure ->
                                        raise (Stream.Error "")
                                  in Action.getf act a)
                         | _ ->
                             let ps strm =
                               (match stream_peek_nth strm n with
                                | Some ((tok, _)) when
                                    Token.match_keyword kwd tok -> tok
                                | _ -> raise Stream.Failure) in
                             let p1 = loop (n + 1) tokl
                             in
                               (fun (__strm : _ Stream.t) ->
                                  let tok = ps __strm in
                                  let s = __strm in
                                  let act = p1 s in Action.getf act tok))
                    | _ -> invalid_arg "parser_of_token_list"
                  in loop 1 tokl
                and parser_of_symbol entry nlevn =
                  function
                  | Smeta (_, symbl, act) ->
                      let act = Obj.magic act entry symbl in
                      let pl = List.map (parser_of_symbol entry nlevn) symbl
                      in
                        Obj.magic
                          (List.fold_left (fun act p -> Obj.magic act p) act
                             pl)
                  | Slist0 s ->
                      let ps = parser_of_symbol entry nlevn s in
                      let rec loop al (__strm : _ Stream.t) =
                        (match try Some (ps __strm)
                               with | Stream.Failure -> None
                         with
                         | Some a -> loop (a :: al) __strm
                         | _ -> al)
                      in
                        (fun (__strm : _ Stream.t) ->
                           let a = loop [] __strm in Action.mk (List.rev a))
                  | Slist0sep (symb, sep) ->
                      let ps = parser_of_symbol entry nlevn symb in
                      let pt = parser_of_symbol entry nlevn sep in
                      let rec kont al (__strm : _ Stream.t) =
                        (match try Some (pt __strm)
                               with | Stream.Failure -> None
                         with
                         | Some v ->
                             let a =
                               (try ps __strm
                                with
                                | Stream.Failure ->
                                    raise
                                      (Stream.Error
                                         (Failed.symb_failed entry v sep symb)))
                             in kont (a :: al) __strm
                         | _ -> al)
                      in
                        (fun (__strm : _ Stream.t) ->
                           match try Some (ps __strm)
                                 with | Stream.Failure -> None
                           with
                           | Some a ->
                               let s = __strm
                               in Action.mk (List.rev (kont [ a ] s))
                           | _ -> Action.mk [])
                  | Slist1 s ->
                      let ps = parser_of_symbol entry nlevn s in
                      let rec loop al (__strm : _ Stream.t) =
                        (match try Some (ps __strm)
                               with | Stream.Failure -> None
                         with
                         | Some a -> loop (a :: al) __strm
                         | _ -> al)
                      in
                        (fun (__strm : _ Stream.t) ->
                           let a = ps __strm in
                           let s = __strm
                           in Action.mk (List.rev (loop [ a ] s)))
                  | Slist1sep (symb, sep) ->
                      let ps = parser_of_symbol entry nlevn symb in
                      let pt = parser_of_symbol entry nlevn sep in
                      let rec kont al (__strm : _ Stream.t) =
                        (match try Some (pt __strm)
                               with | Stream.Failure -> None
                         with
                         | Some v ->
                             let a =
                               (try ps __strm
                                with
                                | Stream.Failure ->
                                    (try parse_top_symb entry symb __strm
                                     with
                                     | Stream.Failure ->
                                         raise
                                           (Stream.Error
                                              (Failed.symb_failed entry v sep
                                                 symb))))
                             in kont (a :: al) __strm
                         | _ -> al)
                      in
                        (fun (__strm : _ Stream.t) ->
                           let a = ps __strm in
                           let s = __strm
                           in Action.mk (List.rev (kont [ a ] s)))
                  | Sopt s ->
                      let ps = parser_of_symbol entry nlevn s
                      in
                        (fun (__strm : _ Stream.t) ->
                           match try Some (ps __strm)
                                 with | Stream.Failure -> None
                           with
                           | Some a -> Action.mk (Some a)
                           | _ -> Action.mk None)
                  | Stry s ->
                      let ps = parser_of_symbol entry nlevn s
                      in try_parser ps
                  | Stree t ->
                      let pt = parser_of_tree entry 1 0 t
                      in
                        (fun strm ->
                           let bp = loc_bp strm in
                           let (__strm : _ Stream.t) = strm in
                           let (act, loc) = add_loc bp pt __strm
                           in Action.getf act loc)
                  | Snterm e ->
                      (fun (__strm : _ Stream.t) -> e.estart 0 __strm)
                  | Snterml (e, l) ->
                      (fun (__strm : _ Stream.t) ->
                         e.estart (level_number e l) __strm)
                  | Sself ->
                      (fun (__strm : _ Stream.t) -> entry.estart 0 __strm)
                  | Snext ->
                      (fun (__strm : _ Stream.t) -> entry.estart nlevn __strm)
                  | Skeyword kwd ->
                      (fun (__strm : _ Stream.t) ->
                         match Stream.peek __strm with
                         | Some ((tok, _)) when Token.match_keyword kwd tok
                             -> (Stream.junk __strm; Action.mk tok)
                         | _ -> raise Stream.Failure)
                  | Stoken ((f, _)) ->
                      (fun (__strm : _ Stream.t) ->
                         match Stream.peek __strm with
                         | Some ((tok, _)) when f tok ->
                             (Stream.junk __strm; Action.mk tok)
                         | _ -> raise Stream.Failure)
                and parse_top_symb entry symb strm =
                  parser_of_symbol entry 0 (top_symb entry symb) strm
                  
                let rec start_parser_of_levels entry clevn =
                  function
                  | [] ->
                      (fun _ (__strm : _ Stream.t) -> raise Stream.Failure)
                  | lev :: levs ->
                      let p1 = start_parser_of_levels entry (succ clevn) levs
                      in
                        (match lev.lprefix with
                         | DeadEnd -> p1
                         | tree ->
                             let alevn =
                               (match lev.assoc with
                                | LeftA | NonA -> succ clevn
                                | RightA -> clevn) in
                             let p2 =
                               parser_of_tree entry (succ clevn) alevn tree
                             in
                               (match levs with
                                | [] ->
                                    (fun levn strm ->
                                       let bp = loc_bp strm in
                                       let (__strm : _ Stream.t) = strm in
                                       let (act, loc) =
                                         add_loc bp p2 __strm in
                                       let strm = __strm in
                                       let a = Action.getf act loc
                                       in entry.econtinue levn loc a strm)
                                | _ ->
                                    (fun levn strm ->
                                       if levn > clevn
                                       then p1 levn strm
                                       else
                                         (let bp = loc_bp strm in
                                          let (__strm : _ Stream.t) = strm
                                          in
                                            match try
                                                    Some
                                                      (add_loc bp p2 __strm)
                                                  with
                                                  | Stream.Failure -> None
                                            with
                                            | Some ((act, loc)) ->
                                                let a = Action.getf act loc
                                                in
                                                  entry.econtinue levn loc a
                                                    strm
                                            | _ -> p1 levn __strm))))
                  
                let start_parser_of_entry entry =
                  match entry.edesc with
                  | Dlevels [] -> Tools.empty_entry entry.ename
                  | Dlevels elev -> start_parser_of_levels entry 0 elev
                  | Dparser p -> (fun _ -> p)
                  
                let rec continue_parser_of_levels entry clevn =
                  function
                  | [] ->
                      (fun _ _ _ (__strm : _ Stream.t) ->
                         raise Stream.Failure)
                  | lev :: levs ->
                      let p1 =
                        continue_parser_of_levels entry (succ clevn) levs
                      in
                        (match lev.lsuffix with
                         | DeadEnd -> p1
                         | tree ->
                             let alevn =
                               (match lev.assoc with
                                | LeftA | NonA -> succ clevn
                                | RightA -> clevn) in
                             let p2 =
                               parser_of_tree entry (succ clevn) alevn tree
                             in
                               (fun levn bp a strm ->
                                  if levn > clevn
                                  then p1 levn bp a strm
                                  else
                                    (let (__strm : _ Stream.t) = strm
                                     in
                                       try p1 levn bp a __strm
                                       with
                                       | Stream.Failure ->
                                           let (act, loc) =
                                             add_loc bp p2 __strm in
                                           let a = Action.getf2 act a loc
                                           in entry.econtinue levn loc a strm)))
                  
                let continue_parser_of_entry entry =
                  match entry.edesc with
                  | Dlevels elev ->
                      let p = continue_parser_of_levels entry 0 elev
                      in
                        (fun levn bp a (__strm : _ Stream.t) ->
                           try p levn bp a __strm with | Stream.Failure -> a)
                  | Dparser _ ->
                      (fun _ _ _ (__strm : _ Stream.t) ->
                         raise Stream.Failure)
                  
              end
              
          end
          
        module Insert =
          struct
            module Make (Structure : Structure.S) =
              struct
                module Tools = Tools.Make(Structure)
                  
                module Parser = Parser.Make(Structure)
                  
                open Structure
                  
                open Format
                  
                open Sig.Grammar
                  
                let is_before s1 s2 =
                  match (s1, s2) with
                  | ((Skeyword _ | Stoken _), (Skeyword _ | Stoken _)) ->
                      false
                  | ((Skeyword _ | Stoken _), _) -> true
                  | _ -> false
                  
                let rec derive_eps =
                  function
                  | Slist0 _ | Slist0sep (_, _) | Sopt _ -> true
                  | Stry s -> derive_eps s
                  | Stree t -> tree_derive_eps t
                  | Slist1 _ | Slist1sep (_, _) | Stoken _ | Skeyword _ ->
                      false
                  | Smeta (_, _, _) | Snterm _ | Snterml (_, _) | Snext |
                      Sself -> false
                and tree_derive_eps =
                  function
                  | LocAct (_, _) -> true
                  | Node { node = s; brother = bro; son = son } ->
                      ((derive_eps s) && (tree_derive_eps son)) ||
                        (tree_derive_eps bro)
                  | DeadEnd -> false
                  
                let empty_lev lname assoc =
                  let assoc = match assoc with | Some a -> a | None -> LeftA
                  in
                    {
                      assoc = assoc;
                      lname = lname;
                      lsuffix = DeadEnd;
                      lprefix = DeadEnd;
                    }
                  
                let change_lev entry lev n lname assoc =
                  let a =
                    match assoc with
                    | None -> lev.assoc
                    | Some a ->
                        (if
                           (a <> lev.assoc) && !(entry.egram.warning_verbose)
                         then
                           (eprintf
                              "<W> Changing associativity of level \"%s\"\n"
                              n;
                            flush Pervasives.stderr)
                         else ();
                         a)
                  in
                    ((match lname with
                      | Some n ->
                          if
                            (lname <> lev.lname) &&
                              !(entry.egram.warning_verbose)
                          then
                            (eprintf "<W> Level label \"%s\" ignored\n" n;
                             flush Pervasives.stderr)
                          else ()
                      | None -> ());
                     {
                       assoc = a;
                       lname = lev.lname;
                       lsuffix = lev.lsuffix;
                       lprefix = lev.lprefix;
                     })
                  
                let change_to_self entry =
                  function | Snterm e when e == entry -> Sself | x -> x
                  
                let get_level entry position levs =
                  match position with
                  | Some First -> ([], empty_lev, levs)
                  | Some Last -> (levs, empty_lev, [])
                  | Some (Level n) ->
                      let rec get =
                        (function
                         | [] ->
                             (eprintf
                                "No level labelled \"%s\" in entry \"%s\"\n"
                                n entry.ename;
                              flush Pervasives.stderr;
                              failwith "Grammar.extend")
                         | lev :: levs ->
                             if Tools.is_level_labelled n lev
                             then ([], (change_lev entry lev n), levs)
                             else
                               (let (levs1, rlev, levs2) = get levs
                                in ((lev :: levs1), rlev, levs2)))
                      in get levs
                  | Some (Before n) ->
                      let rec get =
                        (function
                         | [] ->
                             (eprintf
                                "No level labelled \"%s\" in entry \"%s\"\n"
                                n entry.ename;
                              flush Pervasives.stderr;
                              failwith "Grammar.extend")
                         | lev :: levs ->
                             if Tools.is_level_labelled n lev
                             then ([], empty_lev, (lev :: levs))
                             else
                               (let (levs1, rlev, levs2) = get levs
                                in ((lev :: levs1), rlev, levs2)))
                      in get levs
                  | Some (After n) ->
                      let rec get =
                        (function
                         | [] ->
                             (eprintf
                                "No level labelled \"%s\" in entry \"%s\"\n"
                                n entry.ename;
                              flush Pervasives.stderr;
                              failwith "Grammar.extend")
                         | lev :: levs ->
                             if Tools.is_level_labelled n lev
                             then ([ lev ], empty_lev, levs)
                             else
                               (let (levs1, rlev, levs2) = get levs
                                in ((lev :: levs1), rlev, levs2)))
                      in get levs
                  | None ->
                      (match levs with
                       | lev :: levs ->
                           ([], (change_lev entry lev "<top>"), levs)
                       | [] -> ([], empty_lev, []))
                  
                let rec check_gram entry =
                  function
                  | Snterm e ->
                      if ( != ) e.egram entry.egram
                      then
                        (eprintf
                           "\
  Error: entries \"%s\" and \"%s\" do not belong to the same grammar.\n"
                           entry.ename e.ename;
                         flush Pervasives.stderr;
                         failwith "Grammar.extend error")
                      else ()
                  | Snterml (e, _) ->
                      if ( != ) e.egram entry.egram
                      then
                        (eprintf
                           "\
  Error: entries \"%s\" and \"%s\" do not belong to the same grammar.\n"
                           entry.ename e.ename;
                         flush Pervasives.stderr;
                         failwith "Grammar.extend error")
                      else ()
                  | Smeta (_, sl, _) -> List.iter (check_gram entry) sl
                  | Slist0sep (s, t) ->
                      (check_gram entry t; check_gram entry s)
                  | Slist1sep (s, t) ->
                      (check_gram entry t; check_gram entry s)
                  | Slist0 s | Slist1 s | Sopt s | Stry s ->
                      check_gram entry s
                  | Stree t -> tree_check_gram entry t
                  | Snext | Sself | Stoken _ | Skeyword _ -> ()
                and tree_check_gram entry =
                  function
                  | Node { node = n; brother = bro; son = son } ->
                      (check_gram entry n;
                       tree_check_gram entry bro;
                       tree_check_gram entry son)
                  | LocAct (_, _) | DeadEnd -> ()
                  
                let get_initial =
                  function
                  | Sself :: symbols -> (true, symbols)
                  | symbols -> (false, symbols)
                  
                let insert_tokens gram symbols =
                  let rec insert =
                    function
                    | Smeta (_, sl, _) -> List.iter insert sl
                    | Slist0 s | Slist1 s | Sopt s | Stry s -> insert s
                    | Slist0sep (s, t) -> (insert s; insert t)
                    | Slist1sep (s, t) -> (insert s; insert t)
                    | Stree t -> tinsert t
                    | Skeyword kwd -> using gram kwd
                    | Snterm _ | Snterml (_, _) | Snext | Sself | Stoken _ ->
                        ()
                  and tinsert =
                    function
                    | Node { node = s; brother = bro; son = son } ->
                        (insert s; tinsert bro; tinsert son)
                    | LocAct (_, _) | DeadEnd -> ()
                  in List.iter insert symbols
                  
                let insert_tree entry gsymbols action tree =
                  let rec insert symbols tree =
                    match symbols with
                    | s :: sl -> insert_in_tree s sl tree
                    | [] ->
                        (match tree with
                         | Node { node = s; son = son; brother = bro } ->
                             Node
                               {
                                 node = s;
                                 son = son;
                                 brother = insert [] bro;
                               }
                         | LocAct (old_action, action_list) ->
                             let () =
                               if !(entry.egram.warning_verbose)
                               then
                                 eprintf
                                   "<W> Grammar extension: in [%s] some rule has been masked@."
                                   entry.ename
                               else ()
                             in LocAct (action, (old_action :: action_list))
                         | DeadEnd -> LocAct (action, []))
                  and insert_in_tree s sl tree =
                    match try_insert s sl tree with
                    | Some t -> t
                    | None ->
                        Node
                          {
                            node = s;
                            son = insert sl DeadEnd;
                            brother = tree;
                          }
                  and try_insert s sl tree =
                    match tree with
                    | Node { node = s1; son = son; brother = bro } ->
                        if Tools.eq_symbol s s1
                        then
                          (let t =
                             Node
                               {
                                 node = s1;
                                 son = insert sl son;
                                 brother = bro;
                               }
                           in Some t)
                        else
                          if
                            (is_before s1 s) ||
                              ((derive_eps s) && (not (derive_eps s1)))
                          then
                            (let bro =
                               match try_insert s sl bro with
                               | Some bro -> bro
                               | None ->
                                   Node
                                     {
                                       node = s;
                                       son = insert sl DeadEnd;
                                       brother = bro;
                                     } in
                             let t =
                               Node { node = s1; son = son; brother = bro; }
                             in Some t)
                          else
                            (match try_insert s sl bro with
                             | Some bro ->
                                 let t =
                                   Node
                                     { node = s1; son = son; brother = bro; }
                                 in Some t
                             | None -> None)
                    | LocAct (_, _) | DeadEnd -> None
                  in insert gsymbols tree
                  
                let insert_level entry e1 symbols action slev =
                  match e1 with
                  | true ->
                      {
                        assoc = slev.assoc;
                        lname = slev.lname;
                        lsuffix =
                          insert_tree entry symbols action slev.lsuffix;
                        lprefix = slev.lprefix;
                      }
                  | false ->
                      {
                        assoc = slev.assoc;
                        lname = slev.lname;
                        lsuffix = slev.lsuffix;
                        lprefix =
                          insert_tree entry symbols action slev.lprefix;
                      }
                  
                let levels_of_rules entry position rules =
                  let elev =
                    match entry.edesc with
                    | Dlevels elev -> elev
                    | Dparser _ ->
                        (eprintf "Error: entry not extensible: \"%s\"\n"
                           entry.ename;
                         flush Pervasives.stderr;
                         failwith "Grammar.extend")
                  in
                    if rules = []
                    then elev
                    else
                      (let (levs1, make_lev, levs2) =
                         get_level entry position elev in
                       let (levs, _) =
                         List.fold_left
                           (fun (levs, make_lev) (lname, assoc, level) ->
                              let lev = make_lev lname assoc in
                              let lev =
                                List.fold_left
                                  (fun lev (symbols, action) ->
                                     let symbols =
                                       List.map (change_to_self entry)
                                         symbols
                                     in
                                       (List.iter (check_gram entry) symbols;
                                        let (e1, symbols) =
                                          get_initial symbols
                                        in
                                          (insert_tokens entry.egram symbols;
                                           insert_level entry e1 symbols
                                             action lev)))
                                  lev level
                              in ((lev :: levs), empty_lev))
                           ([], make_lev) rules
                       in levs1 @ ((List.rev levs) @ levs2))
                  
                let extend entry (position, rules) =
                  let elev = levels_of_rules entry position rules
                  in
                    (entry.edesc <- Dlevels elev;
                     entry.estart <-
                       (fun lev strm ->
                          let f = Parser.start_parser_of_entry entry
                          in (entry.estart <- f; f lev strm));
                     entry.econtinue <-
                       fun lev bp a strm ->
                         let f = Parser.continue_parser_of_entry entry
                         in (entry.econtinue <- f; f lev bp a strm))
                  
              end
              
          end
          
        module Delete =
          struct
            exception Rule_not_found of (string * string)
              
            let _ =
              let () =
                Printexc.register_printer
                  (function
                   | Rule_not_found ((symbols, entry)) ->
                       let msg =
                         Printf.sprintf
                           "rule %S cannot be found in entry\n%s" symbols
                           entry
                       in Some msg
                   | _ -> None)
              in ()
              
            module Make (Structure : Structure.S) =
              struct
                module Tools = Tools.Make(Structure)
                  
                module Parser = Parser.Make(Structure)
                  
                module Print = Print.Make(Structure)
                  
                open Structure
                  
                let raise_rule_not_found entry symbols =
                  let to_string f x =
                    let buff = Buffer.create 128 in
                    let ppf = Format.formatter_of_buffer buff
                    in
                      (f ppf x;
                       Format.pp_print_flush ppf ();
                       Buffer.contents buff) in
                  let entry = to_string Print.entry entry in
                  let symbols = to_string Print.print_rule symbols
                  in raise (Rule_not_found ((symbols, entry)))
                  
                let delete_rule_in_tree entry =
                  let rec delete_in_tree symbols tree =
                    match (symbols, tree) with
                    | (s :: sl, Node n) ->
                        if Tools.logically_eq_symbols entry s n.node
                        then delete_son sl n
                        else
                          (match delete_in_tree symbols n.brother with
                           | Some ((dsl, t)) ->
                               Some
                                 ((dsl,
                                   (Node
                                      {
                                        node = n.node;
                                        son = n.son;
                                        brother = t;
                                      })))
                           | None -> None)
                    | (_ :: _, _) -> None
                    | ([], Node n) ->
                        (match delete_in_tree [] n.brother with
                         | Some ((dsl, t)) ->
                             Some
                               ((dsl,
                                 (Node
                                    {
                                      node = n.node;
                                      son = n.son;
                                      brother = t;
                                    })))
                         | None -> None)
                    | ([], DeadEnd) -> None
                    | ([], LocAct (_, [])) -> Some (((Some []), DeadEnd))
                    | ([], LocAct (_, (action :: list))) ->
                        Some ((None, (LocAct (action, list))))
                  and delete_son sl n =
                    match delete_in_tree sl n.son with
                    | Some ((Some dsl, DeadEnd)) ->
                        Some (((Some (n.node :: dsl)), (n.brother)))
                    | Some ((Some dsl, t)) ->
                        let t =
                          Node
                            { node = n.node; son = t; brother = n.brother; }
                        in Some (((Some (n.node :: dsl)), t))
                    | Some ((None, t)) ->
                        let t =
                          Node
                            { node = n.node; son = t; brother = n.brother; }
                        in Some ((None, t))
                    | None -> None
                  in delete_in_tree
                  
                let rec decr_keyw_use gram =
                  function
                  | Skeyword kwd -> removing gram kwd
                  | Smeta (_, sl, _) -> List.iter (decr_keyw_use gram) sl
                  | Slist0 s | Slist1 s | Sopt s | Stry s ->
                      decr_keyw_use gram s
                  | Slist0sep (s1, s2) ->
                      (decr_keyw_use gram s1; decr_keyw_use gram s2)
                  | Slist1sep (s1, s2) ->
                      (decr_keyw_use gram s1; decr_keyw_use gram s2)
                  | Stree t -> decr_keyw_use_in_tree gram t
                  | Sself | Snext | Snterm _ | Snterml (_, _) | Stoken _ ->
                      ()
                and decr_keyw_use_in_tree gram =
                  function
                  | DeadEnd | LocAct (_, _) -> ()
                  | Node n ->
                      (decr_keyw_use gram n.node;
                       decr_keyw_use_in_tree gram n.son;
                       decr_keyw_use_in_tree gram n.brother)
                  
                let rec delete_rule_in_suffix entry symbols =
                  function
                  | lev :: levs ->
                      (match delete_rule_in_tree entry symbols lev.lsuffix
                       with
                       | Some ((dsl, t)) ->
                           ((match dsl with
                             | Some dsl ->
                                 List.iter (decr_keyw_use entry.egram) dsl
                             | None -> ());
                            (match t with
                             | DeadEnd when lev.lprefix == DeadEnd -> levs
                             | _ ->
                                 let lev =
                                   {
                                     assoc = lev.assoc;
                                     lname = lev.lname;
                                     lsuffix = t;
                                     lprefix = lev.lprefix;
                                   }
                                 in lev :: levs))
                       | None ->
                           let levs =
                             delete_rule_in_suffix entry symbols levs
                           in lev :: levs)
                  | [] -> raise_rule_not_found entry symbols
                  
                let rec delete_rule_in_prefix entry symbols =
                  function
                  | lev :: levs ->
                      (match delete_rule_in_tree entry symbols lev.lprefix
                       with
                       | Some ((dsl, t)) ->
                           ((match dsl with
                             | Some dsl ->
                                 List.iter (decr_keyw_use entry.egram) dsl
                             | None -> ());
                            (match t with
                             | DeadEnd when lev.lsuffix == DeadEnd -> levs
                             | _ ->
                    