// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception


// Copyright 2018 Ulf Adams
// Copyright (c) Microsoft Corporation. All rights reserved.

// Boost Software License - Version 1.0 - August 17th, 2003

// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:

// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.


// This file contains test cases derived from:
// https://github.com/ulfjack/ryu
// See xcharconv_ryu.h for the exact commit.
// (Keep the cgmanifest.json commitHash in sync.)


#ifndef FLOAT_SCIENTIFIC_PRECISION_TO_CHARS_TEST_CASES_HPP
#define FLOAT_SCIENTIFIC_PRECISION_TO_CHARS_TEST_CASES_HPP

#include <charconv>

#include "test.hpp"
using namespace std;

inline constexpr FloatPrecisionToCharsTestCase float_scientific_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::scientific, 4, "0.0000e+00"},
    {-0.0f, chars_format::scientific, 4, "-0.0000e+00"},
    {float_inf, chars_format::scientific, 4, "inf"},
    {-float_inf, chars_format::scientific, 4, "-inf"},
    {float_nan, chars_format::scientific, 4, "nan"},
    {-float_nan, chars_format::scientific, 4, "-nan(ind)"},
    {float_nan_payload, chars_format::scientific, 4, "nan"},
    {-float_nan_payload, chars_format::scientific, 4, "-nan"},
    {1.729f, chars_format::scientific, 4, "1.7290e+00"},
    {-1.729f, chars_format::scientific, 4, "-1.7290e+00"},

    // Ryu Printf Zero
    {0.0f, chars_format::scientific, 4, "0.0000e+00"},
    {0.0f, chars_format::scientific, 3, "0.000e+00"},
    {0.0f, chars_format::scientific, 2, "0.00e+00"},
    {0.0f, chars_format::scientific, 1, "0.0e+00"},
    {0.0f, chars_format::scientific, 0, "0e+00"},

    // Test corner cases.
    {0x0.000002p-126f, chars_format::scientific, 104, // min subnormal
        "1."
        "40129846432481707092372958328991613128026194187651577175706828388979108268586060148663818836212158203125e-"
        "45"},
    {0x0.fffffep-126f, chars_format::scientific, 111, // max subnormal
        "1."
        "1754942106924410754870294448492873488270524287458933338571745305715888704756189042655023513361811637878417"
        "96875e-38"},
    {0x1p-126f, chars_format::scientific, 88, // min normal
        "1.1754943508222875079687365372222456778186655567720875215087517062784172594547271728515625e-38"},
    {0x1.fffffep+127f, chars_format::scientific, 37, // max normal
        "3.4028234663852885981170418348451692544e+38"},

    // Ryu Printf AllPowersOfTen
    // These values test every power of ten that's within the range of floats.
    {1e-44f, chars_format::scientific, 104,
        "9."
        "80908925027371949646610708302941291896183359313561040229947798722853757880102421040646731853485107421875e-"
        "45"},
    {1e-43f, chars_format::scientific, 105,
        "9."
        "949219096706201203558480041358404532089859787323261979475184815617516687069610270555131137371063232421875e"
        "-44"},
    {1e-42f, chars_format::scientific, 106,
        "1."
        "0005271035279193886395429224690001177341070264998322610345467546973108330377044694614596664905548095703125"
        "e-42"},
    {1e-41f, chars_format::scientific, 102,
        "9.999665841421894618111734306356841512815949217230816547258439273837549166046301252208650112152099609375e-"
        "42"},
    {1e-40f, chars_format::scientific, 107,
        "9."
        "9999461011147595815259190522734994960422052696191918504127906874943271242628384243289474397897720336914062"
        "5e-41"},
    {1e-39f, chars_format::scientific, 107,
        "1."
        "0000002153053332574208756001456831092687456480096866911043660970225682715906145858753006905317306518554687"
        "5e-39"},
    {1e-38f, chars_format::scientific, 109,
        "9."
        "9999993504564039245746141539976645128551939195729831580121174560891149363239804870318039320409297943115234"
        "375e-39"},
    {1e-37f, chars_format::scientific, 107,
        "9."
        "9999999109757896545014425234894978288216464316777599086184261589164284922404135613760445266962051391601562"
        "5e-38"},
    {1e-36f, chars_format::scientific, 107,
        "1."
        "0000000359391298238442905219082964481594808441361581309103473121178279336973560020851437002420425415039062"
        "5e-36"},
    {1e-35f, chars_format::scientific, 104,
        "1."
        "00000001800250948048663201408455778204855436374880527489094543362735389990803014370612800121307373046875e-"
        "35"},
    {1e-34f, chars_format::scientific, 102,
        "1.000000046701102029858885626602539647826036732368569844521988439212112353970951517112553119659423828125e-"
        "34"},
    {1e-33f, chars_format::scientific, 98,
        "1.00000002374222799036108273658815415520405083747275818881715403474430559072061441838741302490234375e-33"},
    {1e-32f, chars_format::scientific, 98,
        "1.00000002374222799036108273658815415520405083747275818881715403474430559072061441838741302490234375e-32"},
    {1e-31f, chars_format::scientific, 94,
        "9.9999997966118983452530118776053400936983791927279980986387197816611660527996718883514404296875e-32"},
    {1e-30f, chars_format::scientific, 88,
        "1.0000000031710768509710513471352647538147514756461109453056224083411507308483123779296875e-30"},
    {1e-29f, chars_format::scientific, 88,
        "1.0000000031710768509710513471352647538147514756461109453056224083411507308483123779296875e-29"},
    {1e-28f, chars_format::scientific, 88,
        "1.0000000031710768509710513471352647538147514756461109453056224083411507308483123779296875e-28"},
    {1e-27f, chars_format::scientific, 85,
        "1.0000000272452011558114995103349890361263429573723815479979748488403856754302978515625e-27"},
    {1e-26f, chars_format::scientific, 82,
        "9.9999998872660226806678244921543018442779658661034858369021094404160976409912109375e-27"},
    {1e-25f, chars_format::scientific, 79,
        "1.0000000195414813782625560981110772657866336832199749551364220678806304931640625e-25"},
    {1e-24f, chars_format::scientific, 79,
        "1.0000000195414813782625560981110772657866336832199749551364220678806304931640625e-24"},
    {1e-23f, chars_format::scientific, 75,
        "9.999999998199587477372609628178631337169779413898140774108469486236572265625e-24"},
    {1e-22f, chars_format::scientific, 75,
        "1.000000031374394956577733179287005745028427128318071481771767139434814453125e-22"},
    {1e-21f, chars_format::scientific, 66, "9.999999682655225388967887463487205224055287544615566730499267578125e-22"},
    {1e-20f, chars_format::scientific, 66, "9.999999682655225388967887463487205224055287544615566730499267578125e-21"},
    {1e-19f, chars_format::scientific, 66, "9.999999682655225388967887463487205224055287544615566730499267578125e-20"},
    {1e-18f, chars_format::scientific, 65, "1.00000004581370496574313326554328540396454627625644207000732421875e-18"},
    {1e-17f, chars_format::scientific, 61, "9.9999998377515902426605765018763349871733225882053375244140625e-18"},
    {1e-16f, chars_format::scientific, 61, "1.0000000168623835263871646450439811815158464014530181884765625e-16"},
    {1e-15f, chars_format::scientific, 58, "1.0000000036274937255387218471014421083964407444000244140625e-15"},
    {1e-14f, chars_format::scientific, 53, "9.99999982451670044181213370393379591405391693115234375e-15"},
    {1e-13f, chars_format::scientific, 53, "9.99999982451670044181213370393379591405391693115234375e-14"},
    {1e-12f, chars_format::scientific, 48, "9.999999960041972002500187954865396022796630859375e-13"},
    {1e-11f, chars_format::scientific, 48, "9.999999960041972002500187954865396022796630859375e-12"},
    {1e-10f, chars_format::scientific, 47, "1.00000001335143196001808973960578441619873046875e-10"},
    {1e-09f, chars_format::scientific, 43, "9.9999997171806853657471947371959686279296875e-10"},
    {1e-08f, chars_format::scientific, 41, "9.99999993922529029077850282192230224609375e-09"},
    {1e-07f, chars_format::scientific, 40, "1.0000000116860974230803549289703369140625e-07"},
    {1e-06f, chars_format::scientific, 36, "9.999999974752427078783512115478515625e-07"},
    {1e-05f, chars_format::scientific, 32, "9.99999974737875163555145263671875e-06"},
    {1e-04f, chars_format::scientific, 32, "9.99999974737875163555145263671875e-05"},
    {1e-03f, chars_format::scientific, 30, "1.000000047497451305389404296875e-03"},
    {1e-02f, chars_format::scientific, 26, "9.99999977648258209228515625e-03"},
    {1e-01f, chars_format::scientific, 26, "1.00000001490116119384765625e-01"},
    {1e+00f, chars_format::scientific, 0, "1e+00"},
    {1e+01f, chars_format::scientific, 0, "1e+01"},
    {1e+02f, chars_format::scientific, 0, "1e+02"},
    {1e+03f, chars_format::scientific, 0, "1e+03"},
    {1e+04f, chars_format::scientific, 0, "1e+04"},
    {1e+05f, chars_format::scientific, 0, "1e+05"},
    {1e+06f, chars_format::scientific, 0, "1e+06"},
    {1e+07f, chars_format::scientific, 0, "1e+07"},
    {1e+08f, chars_format::scientific, 0, "1e+08"},
    {1e+09f, chars_format::scientific, 0, "1e+09"},
    {1e+10f, chars_format::scientific, 0, "1e+10"},
    {1e+11f, chars_format::scientific, 10, "9.9999997952e+10"},
    {1e+12f, chars_format::scientific, 11, "9.99999995904e+11"},
    {1e+13f, chars_format::scientific, 12, "9.999999827968e+12"},
    {1e+14f, chars_format::scientific, 14, "1.00000000376832e+14"},
    {1e+15f, chars_format::scientific, 14, "9.99999986991104e+14"},
    {1e+16f, chars_format::scientific, 16, "1.0000000272564224e+16"},
    {1e+17f, chars_format::scientific, 16, "9.9999998430674944e+16"},
    {1e+18f, chars_format::scientific, 16, "9.9999998430674944e+17"},
    {1e+19f, chars_format::scientific, 18, "9.999999980506447872e+18"},
    {1e+20f, chars_format::scientific, 20, "1.00000002004087734272e+20"},
    {1e+21f, chars_format::scientific, 20, "1.00000002004087734272e+21"},
    {1e+22f, chars_format::scientific, 21, "9.999999778196308361216e+21"},
    {1e+23f, chars_format::scientific, 21, "9.999999778196308361216e+22"},
    {1e+24f, chars_format::scientific, 24, "1.000000013848427855085568e+24"},
    {1e+25f, chars_format::scientific, 24, "9.999999562023526247432192e+24"},
    {1e+26f, chars_format::scientific, 26, "1.00000002537764290115403776e+26"},
    {1e+27f, chars_format::scientific, 26, "9.99999988484154753734934528e+26"},
    {1e+28f, chars_format::scientific, 27, "9.999999442119689768320106496e+27"},
    {1e+29f, chars_format::scientific, 29, "1.00000001504746621987668885504e+29"},
    {1e+30f, chars_format::scientific, 29, "1.00000001504746621987668885504e+30"},
    {1e+31f, chars_format::scientific, 30, "9.999999848243207295109594873856e+30"},
    {1e+32f, chars_format::scientific, 32, "1.00000003318135351409612647563264e+32"},
    {1e+33f, chars_format::scientific, 32, "9.99999994495727286427992885035008e+32"},
    {1e+34f, chars_format::scientific, 33, "9.999999790214767953607394487959552e+33"},
    {1e+35f, chars_format::scientific, 34, "1.0000000409184787596297531937521664e+35"},
    {1e+36f, chars_format::scientific, 35, "9.99999961690316245365415600208216064e+35"},
    {1e+37f, chars_format::scientific, 36, "9.999999933815812510711506376257961984e+36"},
    {1e+38f, chars_format::scientific, 37, "9.9999996802856924650656260769173209088e+37"},

    // Ryu Printf RoundToEven
    {0.125f, chars_format::scientific, 2, "1.25e-01"},
    {0.125f, chars_format::scientific, 1, "1.2e-01"},
    {0.375f, chars_format::scientific, 2, "3.75e-01"},
    {0.375f, chars_format::scientific, 1, "3.8e-01"},

    // Ryu Printf RoundToEvenInteger
    {2.5f, chars_format::scientific, 1, "2.5e+00"},
    {2.5f, chars_format::scientific, 0, "2e+00"},
    {3.5f, chars_format::scientific, 1, "3.5e+00"},
    {3.5f, chars_format::scientific, 0, "4e+00"},

    // Ryu Printf NonRoundToEvenScenarios
    {0.748046875f, chars_format::scientific, 2, "7.48e-01"},
    {0.748046875f, chars_format::scientific, 1, "7.5e-01"},
    {0.748046875f, chars_format::scientific, 0, "7e-01"}, // 0.75 would round to "8e-01", but this is smaller

    {0.2509765625f, chars_format::scientific, 2, "2.51e-01"},
    {0.2509765625f, chars_format::scientific, 1, "2.5e-01"},
    {0.2509765625f, chars_format::scientific, 0, "3e-01"}, // 0.25 would round to "2e-01", but this is larger

    {0x1.000002p-2f, chars_format::scientific, 24, "2.500000298023223876953125e-01"},
    {0x1.000002p-2f, chars_format::scientific, 2, "2.50e-01"},
    {0x1.000002p-2f, chars_format::scientific, 1, "2.5e-01"},
    {0x1.000002p-2f, chars_format::scientific, 0, "3e-01"}, // 0.25 would round to "2e-01", but this is larger (again)

    // More rounding tests.
    {9.5f, chars_format::scientific, 1, "9.5e+00"},
    {9.5f, chars_format::scientific, 0, "1e+01"},
    {10.5f, chars_format::scientific, 2, "1.05e+01"},
    {10.5f, chars_format::scientific, 1, "1.0e+01"},

    {1.241f, chars_format::scientific, 3, "1.241e+00"},
    {1.241f, chars_format::scientific, 1, "1.2e+00"},
    {1.251f, chars_format::scientific, 3, "1.251e+00"},
    {1.251f, chars_format::scientific, 1, "1.3e+00"},
    {1.261f, chars_format::scientific, 3, "1.261e+00"},
    {1.261f, chars_format::scientific, 1, "1.3e+00"},
    {1.341f, chars_format::scientific, 3, "1.341e+00"},
    {1.341f, chars_format::scientific, 1, "1.3e+00"},
    {1.351f, chars_format::scientific, 3, "1.351e+00"},
    {1.351f, chars_format::scientific, 1, "1.4e+00"},
    {1.361f, chars_format::scientific, 3, "1.361e+00"},
    {1.361f, chars_format::scientific, 1, "1.4e+00"},

    {2.41f, chars_format::scientific, 2, "2.41e+00"},
    {2.41f, chars_format::scientific, 0, "2e+00"},
    {2.51f, chars_format::scientific, 2, "2.51e+00"},
    {2.51f, chars_format::scientific, 0, "3e+00"},
    {2.61f, chars_format::scientific, 2, "2.61e+00"},
    {2.61f, chars_format::scientific, 0, "3e+00"},
    {3.41f, chars_format::scientific, 2, "3.41e+00"},
    {3.41f, chars_format::scientific, 0, "3e+00"},
    {3.51f, chars_format::scientific, 2, "3.51e+00"},
    {3.51f, chars_format::scientific, 0, "4e+00"},
    {3.61f, chars_format::scientific, 2, "3.61e+00"},
    {3.61f, chars_format::scientific, 0, "4e+00"},

    // Ryu Printf VaryingPrecision
    {1.142857f, chars_format::scientific, 28, "1.1428569555282592773437500000e+00"},
    {1.142857f, chars_format::scientific, 27, "1.142856955528259277343750000e+00"},
    {1.142857f, chars_format::scientific, 26, "1.14285695552825927734375000e+00"},
    {1.142857f, chars_format::scientific, 25, "1.1428569555282592773437500e+00"},
    {1.142857f, chars_format::scientific, 24, "1.142856955528259277343750e+00"},
    {1.142857f, chars_format::scientific, 23, "1.14285695552825927734375e+00"},
    {1.142857f, chars_format::scientific, 22, "1.1428569555282592773438e+00"},
    {1.142857f, chars_format::scientific, 21, "1.142856955528259277344e+00"},
    {1.142857f, chars_format::scientific, 20, "1.14285695552825927734e+00"},
    {1.142857f, chars_format::scientific, 19, "1.1428569555282592773e+00"},
    {1.142857f, chars_format::scientific, 18, "1.142856955528259277e+00"},
    {1.142857f, chars_format::scientific, 17, "1.14285695552825928e+00"},
    {1.142857f, chars_format::scientific, 16, "1.1428569555282593e+00"},
    {1.142857f, chars_format::scientific, 15, "1.142856955528259e+00"},
    {1.142857f, chars_format::scientific, 14, "1.14285695552826e+00"},
    {1.142857f, chars_format::scientific, 13, "1.1428569555283e+00"},
    {1.142857f, chars_format::scientific, 12, "1.142856955528e+00"},
    {1.142857f, chars_format::scientific, 11, "1.14285695553e+00"},
    {1.142857f, chars_format::scientific, 10, "1.1428569555e+00"},
    {1.142857f, chars_format::scientific, 9, "1.142856956e+00"},
    {1.142857f, chars_format::scientific, 8, "1.14285696e+00"},
    {1.142857f, chars_format::scientific, 7, "1.1428570e+00"},
    {1.142857f, chars_format::scientific, 6, "1.142857e+00"},
    {1.142857f, chars_format::scientific, 5, "1.14286e+00"},
    {1.142857f, chars_format::scientific, 4, "1.1429e+00"},
    {1.142857f, chars_format::scientific, 3, "1.143e+00"},
    {1.142857f, chars_format::scientific, 2, "1.14e+00"},
    {1.142857f, chars_format::scientific, 1, "1.1e+00"},
    {1.142857f, chars_format::scientific, 0, "1e+00"},

    // Negative precision requests 6 digits of precision.
    {1.142857f, chars_format::scientific, -1, "1.142857e+00"},
    {1.142857f, chars_format::scientific, -2, "1.142857e+00"},
    {1.142857f, chars_format::scientific, -3, "1.142857e+00"},

    // Ryu Printf Carrying
    {2.0009f, chars_format::scientific, 4, "2.0009e+00"},
    {2.0009f, chars_format::scientific, 3, "2.001e+00"},
    {2.0029f, chars_format::scientific, 4, "2.0029e+00"},
    {2.0029f, chars_format::scientific, 3, "2.003e+00"},
    {2.0099f, chars_format::scientific, 4, "2.0099e+00"},
    {2.0099f, chars_format::scientific, 3, "2.010e+00"},
    {2.0299f, chars_format::scientific, 4, "2.0299e+00"},
    {2.0299f, chars_format::scientific, 3, "2.030e+00"},
    {2.0999f, chars_format::scientific, 4, "2.0999e+00"},
    {2.0999f, chars_format::scientific, 3, "2.100e+00"},
    {2.2999f, chars_format::scientific, 4, "2.2999e+00"},
    {2.2999f, chars_format::scientific, 3, "2.300e+00"},
    {2.9999f, chars_format::scientific, 4, "2.9999e+00"},
    {2.9999f, chars_format::scientific, 3, "3.000e+00"},
    {9.9999f, chars_format::scientific, 4, "9.9999e+00"},
    {9.9999f, chars_format::scientific, 3, "1.000e+01"},

    {2.09f, chars_format::scientific, 2, "2.09e+00"},
    {2.09f, chars_format::scientific, 1, "2.1e+00"},
    {2.29f, chars_format::scientific, 2, "2.29e+00"},
    {2.29f, chars_format::scientific, 1, "2.3e+00"},
    {2.99f, chars_format::scientific, 2, "2.99e+00"},
    {2.99f, chars_format::scientific, 1, "3.0e+00"},
    {9.99f, chars_format::scientific, 2, "9.99e+00"},
    {9.99f, chars_format::scientific, 1, "1.0e+01"},

    {2.9f, chars_format::scientific, 1, "2.9e+00"},
    {2.9f, chars_format::scientific, 0, "3e+00"},
    {9.9f, chars_format::scientific, 1, "9.9e+00"},
    {9.9f, chars_format::scientific, 0, "1e+01"},

    // Ryu Printf Exponents
    {9.99e-10f, chars_format::scientific, 2, "9.99e-10"},
    {9.99e-09f, chars_format::scientific, 2, "9.99e-09"},
    {9.99e-01f, chars_format::scientific, 2, "9.99e-01"},
    {9.99e+00f, chars_format::scientific, 2, "9.99e+00"},
    {9.99e+01f, chars_format::scientific, 2, "9.99e+01"},
    {9.99e+09f, chars_format::scientific, 2, "9.99e+09"},
    {9.99e+10f, chars_format::scientific, 2, "9.99e+10"},

    {9.99e-10f, chars_format::scientific, 1, "1.0e-09"},
    {9.99e-09f, chars_format::scientific, 1, "1.0e-08"},
    {9.99e-01f, chars_format::scientific, 1, "1.0e+00"},
    {9.99e+00f, chars_format::scientific, 1, "1.0e+01"},
    {9.99e+01f, chars_format::scientific, 1, "1.0e+02"},
    {9.99e+09f, chars_format::scientific, 1, "1.0e+10"},
    {9.99e+10f, chars_format::scientific, 1, "1.0e+11"},

    // Ryu Printf AllBinaryExponents
    // These values test every binary exponent.
    // The mantissas were randomly generated.
    {0x0.bafab0p-126f, chars_format::scientific, 107,
        "8."
        "5856660078164374052571520381239855817217629811131320365461649230225810169869760102301370352506637573242187"
        "5e-39"},
    {0x1.2c4906p-126f, chars_format::scientific, 111,
        "1."
        "3788422360444725799170555395988202383016563601793620435599297375199720136484948795896343654021620750427246"
        "09375e-38"},
    {0x1.da6c8cp-125f, chars_format::scientific, 109,
        "4."
        "3568964460614492296234103491671745446178474818057230651136547036595368653788540314053534530103206634521484"
        "375e-38"},
    {0x1.094fd8p-124f, chars_format::scientific, 107,
        "4."
        "8730098044956940648689859309723501868625857818278160902221179289536129308757494982273783534765243530273437"
        "5e-38"},
    {0x1.1fba2ap-123f, chars_format::scientific, 109,
        "1."
        "0569428191822507939881039060004188252129978601750040353991776209794002661102041429330711252987384796142578"
        "125e-37"},
    {0x1.05c066p-122f, chars_format::scientific, 108,
        "1."
        "9230467241438048605749036664954171842721044312192826420869210321670447760844524509593611583113670349121093"
        "75e-37"},
    {0x1.aa97aep-121f, chars_format::scientific, 107,
        "6."
        "2682134052278382034317570078635218253248121012038564692208751497667224006349329101794864982366561889648437"
        "5e-37"},
    {0x1.dd39a8p-120f, chars_format::scientific, 105,
        "1."
        "402438874646247750851657016594307463985884969771101485328502708228859408023936339304782450199127197265625e"
        "-36"},
    {0x1.d47ee4p-119f, chars_format::scientific, 105,
        "2."
        "753570046800320919541684228734204622774621916697559471583583415484064449429979504202492535114288330078125e"
        "-36"},
    {0x1.3d3c36p-118f, chars_format::scientific, 105,
        "3."
        "729081842766376549743290774072513136892856721536744273788293430570150999159295679419301450252532958984375e"
        "-36"},
    {0x1.1661f4p-117f, chars_format::scientific, 103,
        "6."
        "5447441644065192772010189083715139205202243608571574700187600294454259852727773250080645084381103515625e-"
        "36"},
    {0x1.b68df4p-116f, chars_format::scientific, 103,
        "2."
        "0620733697737581832019236718658803118114911610758483179403903647053386549714559805579483509063720703125e-"
        "35"},
    {0x1.d99cbcp-115f, chars_format::scientific, 102,
        "4.453828135148790991673442414370889923378622963993538561236896870798585013062620419077575206756591796875e-"
        "35"},
    {0x1.fd046ep-114f, chars_format::scientific, 102,
        "9.573551435136219346253356313052326716805272442482100391194957227092299234527672524563968181610107421875e-"
        "35"},
    {0x1.89834cp-113f, chars_format::scientific, 101,
        "1.48023092977964777039833097802364118989950189925000614428441357561805347131667076610028743743896484375e-"
        "34"},
    {0x1.44f9f6p-112f, chars_format::scientific, 101,
        "2.44485077761403269855863218622241304191185681346184344636825701291282797456005937419831752777099609375e-"
        "34"},
    {0x1.610156p-111f, chars_format::scientific, 100,
        "5.3114321941046389584918230189833796378476189953948627677188069895475308612731168977916240692138671875e-"
        "34"},
    {0x1.1c4ce0p-110f, chars_format::scientific, 95,
        "8.55535074104030543315341178235051098966105705542347053919882693406862017582170665264129638671875e-34"},
    {0x1.c8846ap-109f, chars_format::scientific, 99,
        "2.747563210400574676694231398741237029228255561600948508123483382536988983702030964195728302001953125e-"
        "33"},
    {0x1.49aaa6p-108f, chars_format::scientific, 98,
        "3.96821729911656973098277999199730227135040686156682094762847279323381144422455690801143646240234375e-33"},
    {0x1.f5603cp-107f, chars_format::scientific, 97,
        "1.2070186113858457615715036175854047096256968023404067681496332209434285687166266143321990966796875e-32"},
    {0x1.b7bbf8p-106f, chars_format::scientific, 95,
        "2.11724341322508937840915176712265363597160619557838059749677039889093066449277102947235107421875e-32"},
    {0x1.6d305cp-105f, chars_format::scientific, 95,
        "3.51664122601460292174574136500884378151845989294218826175242309517443572985939681529998779296875e-32"},
    {0x1.dd9944p-104f, chars_format::scientific, 94,
        "9.1982162588143308372426801228237163172804681524337790977929874003393706516362726688385009765625e-32"},
    {0x1.0f4254p-103f, chars_format::scientific, 94,
        "1.0448520245617299544506055022011322932281191403904399041258077573957052663899958133697509765625e-31"},
    {0x1.049450p-102f, chars_format::scientific, 91,
        "2.0074302591139273483884727591728498336691676235299559849512007758676190860569477081298828125e-31"},
    {0x1.28d030p-101f, chars_format::scientific, 90,
        "4.573131937693259427041496124538667251427229422876784281637441154089174233376979827880859375e-31"},
    {0x1.28a2bep-100f, chars_format::scientific, 92,
        "9.14079359487553225693590893672771706036816190371857397678478918123801122419536113739013671875e-31"},
    {0x1.e674d2p-99f, chars_format::scientific, 92,
        "2.99801859623548450508972193909865244412972328427583780519061207314734929241240024566650390625e-30"},
    {0x1.227314p-98f, chars_format::scientific, 90,
        "3.580066786954745677669456497979475880433682721056161402106710056614247150719165802001953125e-30"},
    {0x1.735b6cp-97f, chars_format::scientific, 89,
        "9.15465972623783196037990815292079067038371088262625752118850641636527143418788909912109375e-30"},
    {0x1.ef60b4p-96f, chars_format::scientific, 89,
        "2.44240085996903849216356078751341022854748745722804070812372856380534358322620391845703125e-29"},
    {0x1.f58d34p-95f, chars_format::scientific, 88,
        "4.9456803654801575096422210377919300096339643075561698370989915929385460913181304931640625e-29"},
    {0x1.a9fa8ap-94f, chars_format::scientific, 88,
        "8.4009479452815486408032573050798399509585053673323129519445728874416090548038482666015625e-29"},
    {0x1.2ebd9ap-93f, chars_format::scientific, 88,
        "1.1941012414974986903540736041200443068748917913209084407100135649670846760272979736328125e-28"},
    {0x1.1c25bep-92f, chars_format::scientific, 87,
        "2.241527991772840369420073304083191365256388471842441401093992681126110255718231201171875e-28"},
    {0x1.80d526p-91f, chars_format::scientific, 86,
        "6.07158803876554990450680694292368438689814417845436178566842500003986060619354248046875e-28"},
    {0x1.16cdd0p-90f, chars_format::scientific, 82,
        "8.7975016152851199468348749874043115366058394333226289063532021827995777130126953125e-28"},
    {0x1.be00c0p-89f, chars_format::scientific, 80,
        "2.81467419875603623917323685831723008277625852624481694874702952802181243896484375e-27"},
    {0x1.dbe376p-88f, chars_format::scientific, 84,
        "6.006557569745856595501482055708847377292216272726133041715002036653459072113037109375e-27"},
    {0x1.75b358p-87f, chars_format::scientific, 81,
        "9.433528423839338263891493367075328273989136274035871565502020530402660369873046875e-27"},
    {0x1.5e56fap-86f, chars_format::scientific, 83,
        "1.76876373794073549186243776242822499413496518949617808402763330377638339996337890625e-26"},
    {0x1.1542e6p-85f, chars_format::scientific, 82,
        "2.7996239036498255213653797353262236131499034186287389047720353119075298309326171875e-26"},
    {0x1.37b7a6p-84f, chars_format::scientific, 81,
        "6.295082290272475030989309944874260889838800403506269276476814411580562591552734375e-26"},
    {0x1.31f62ap-83f, chars_format::scientific, 81,
        "1.235768973696664669858567539321145118649987459935601918914471752941608428955078125e-25"},
    {0x1.ac3560p-82f, chars_format::scientific, 76,
        "3.4590406845628797200186450581230516131137076030199750675819814205169677734375e-25"},
    {0x1.a7db5cp-81f, chars_format::scientific, 78,
        "6.847777099176331341674240101847394713956151957034990118700079619884490966796875e-25"},
    {0x1.40189cp-80f, chars_format::scientific, 78,
        "1.034286379672715366987641733210033664035198963659922810620628297328948974609375e-24"},
    {0x1.aad1eep-79f, chars_format::scientific, 78,
        "2.758259846499093682487211692864773559218105614121441249153576791286468505859375e-24"},
    {0x1.49824cp-78f, chars_format::scientific, 76,
        "4.2588036474940459085811637121780459484809977510622047702781856060028076171875e-24"},
    {0x1.955292p-77f, chars_format::scientific, 77,
        "1.04773420985315373838626628182169411331037256474019159213639795780181884765625e-23"},
    {0x1.d8ca0cp-76f, chars_format::scientific, 75,
        "2.444263111177596802332967266437459101513507420122550684027373790740966796875e-23"},
    {0x1.28b5aap-75f, chars_format::scientific, 75,
        "3.067905619497844072028707730390270809472941238027487997896969318389892578125e-23"},
    {0x1.e5fda8p-74f, chars_format::scientific, 73,
        "1.0050055115902033206854316793794380802129495577901252545416355133056640625e-22"},
    {0x1.fd929cp-73f, chars_format::scientific, 73,
        "2.1075432611470358337541921610390309449467594049565377645194530487060546875e-22"},
    {0x1.c0b84cp-72f, chars_format::scientific, 72,
        "3.711724097438896357340602997067040280665395357573288492858409881591796875e-22"},
    {0x1.5cfeaep-71f, chars_format::scientific, 72,
        "5.773635352424624465965559338086719731730767080080113373696804046630859375e-22"},
    {0x1.bcce4ap-70f, chars_format::scientific, 72,
        "1.471738991536079335112024613790339400143380998997599817812442779541015625e-21"},
    {0x1.edf106p-69f, chars_format::scientific, 71,
        "3.26863064574260634910627773444362353938430487687583081424236297607421875e-21"},
    {0x1.30b422p-68f, chars_format::scientific, 70,
        "4.0327191475944672156035895296995186232180685692583210766315460205078125e-21"},
    {0x1.7aa8d8p-67f, chars_format::scientific, 68,
        "1.00230347240102665385544432156972316505516573670320212841033935546875e-20"},
    {0x1.4ad4e0p-66f, chars_format::scientific, 65,
        "1.75140760553442509348562143578487138029231573455035686492919921875e-20"},
    {0x1.dde636p-65f, chars_format::scientific, 68,
        "5.05995524921864861016965251964971894693690046551637351512908935546875e-20"},
    {0x1.5df870p-64f, chars_format::scientific, 64,
        "7.4109127331368847396687003781234892585416673682630062103271484375e-20"},
    {0x1.c346fap-63f, chars_format::scientific, 67,
        "1.9112335047873604296656620434025075638828639057464897632598876953125e-19"},
    {0x1.58d2eap-62f, chars_format::scientific, 66,
        "2.920771899491385068938311721231659845443573431111872196197509765625e-19"},
    {0x1.0d4824p-61f, chars_format::scientific, 64,
        "4.5618111223383324851561766710705825289551285095512866973876953125e-19"},
    {0x1.04585cp-60f, chars_format::scientific, 63,
        "8.820836917354691955064048547452415505176759324967861175537109375e-19"},
    {0x1.55cf7ap-59f, chars_format::scientific, 64,
        "2.3161977389916240139687737820128887733517331071197986602783203125e-18"},
    {0x1.1fd8ecp-58f, chars_format::scientific, 62,
        "3.90105904223582084021197668999292318403604440391063690185546875e-18"},
    {0x1.0bc866p-57f, chars_format::scientific, 62,
        "7.25826751123333980988787395016714754092390649020671844482421875e-18"},
    {0x1.4dfa86p-56f, chars_format::scientific, 62,
        "1.81050165732891247031242920595417444928898476064205169677734375e-17"},
    {0x1.335daep-55f, chars_format::scientific, 61,
        "3.3324681586205479543426333233213654239079914987087249755859375e-17"},
    {0x1.5bc756p-54f, chars_format::scientific, 60,
        "7.541247487712833172911197632259927559061907231807708740234375e-17"},
    {0x1.9eb052p-53f, chars_format::scientific, 60,
        "1.798425779915148827771409489884035792783834040164947509765625e-16"},
    {0x1.13b6d2p-52f, chars_format::scientific, 59,
        "2.39143897259270284301468922905087310937233269214630126953125e-16"},
    {0x1.260438p-51f, chars_format::scientific, 56, "5.10037289299151118393549353413618518970906734466552734375e-16"},
    {0x1.9e6b44p-50f, chars_format::scientific, 57, "1.437804758404521467129999479084290214814245700836181640625e-15"},
    {0x1.89c0bcp-49f, chars_format::scientific, 56, "2.73220937993773164975674916377101908437907695770263671875e-15"},
    {0x1.e30610p-48f, chars_format::scientific, 53, "6.70330015995791728133923470522859133780002593994140625e-15"},
    {0x1.48b6e8p-47f, chars_format::scientific, 53, "9.12365953728740131101204724473063834011554718017578125e-15"},
    {0x1.41382ep-46f, chars_format::scientific, 55, "1.7831261573081173821275768887062440626323223114013671875e-14"},
    {0x1.383b8ep-45f, chars_format::scientific, 54, "3.466478609693256218715617933412431739270687103271484375e-14"},
    {0x1.1e6564p-44f, chars_format::scientific, 52, "6.3592699357274684590635160930105485022068023681640625e-14"},
    {0x1.c35e62p-43f, chars_format::scientific, 53, "2.00447961722950707130763703389675356447696685791015625e-13"},
    {0x1.2a2f4ep-42f, chars_format::scientific, 52, "2.6484129017449731247069166784058324992656707763671875e-13"},
    {0x1.69fae2p-41f, chars_format::scientific, 51, "6.430056682417417679431537180789746344089508056640625e-13"},
    {0x1.4ccefep-40f, chars_format::scientific, 51, "1.182373535017766652543969030375592410564422607421875e-12"},
    {0x1.aa9bf6p-39f, chars_format::scientific, 50, "3.03124083993189241681420753593556582927703857421875e-12"},
    {0x1.3b9744p-38f, chars_format::scientific, 48, "4.484816164274096905728583806194365024566650390625e-12"},
    {0x1.b2fc6ap-37f, chars_format::scientific, 49, "1.2363045483188006556929394719190895557403564453125e-11"},
    {0x1.7bc418p-36f, chars_format::scientific, 46, "2.1587197307493255493682227097451686859130859375e-11"},
    {0x1.f4a74cp-35f, chars_format::scientific, 46, "5.6917713597837149563929415307939052581787109375e-11"},
    {0x1.89f248p-34f, chars_format::scientific, 44, "8.95730434269381703416001982986927032470703125e-11"},
    {0x1.60ac54p-33f, chars_format::scientific, 45, "1.603771837555001411601551808416843414306640625e-10"},
    {0x1.2f6d0ep-32f, chars_format::scientific, 45, "2.759643347172158200919511727988719940185546875e-10"},
    {0x1.748684p-31f, chars_format::scientific, 43, "6.7761984912095840627443976700305938720703125e-10"},
    {0x1.b4fa00p-30f, chars_format::scientific, 36, "1.589711473570787347853183746337890625e-09"},
    {0x1.c204d8p-29f, chars_format::scientific, 41, "3.27431859403759517590515315532684326171875e-09"},
    {0x1.50029ep-28f, chars_format::scientific, 42, "4.889592286616561978007666766643524169921875e-09"},
    {0x1.56cf38p-27f, chars_format::scientific, 39, "9.977068060607052757404744625091552734375e-09"},
    {0x1.0b5a5cp-26f, chars_format::scientific, 40, "1.5561990807100301026366651058197021484375e-08"},
    {0x1.fc8250p-25f, chars_format::scientific, 37, "5.9198242752245278097689151763916015625e-08"},
    {0x1.c66674p-24f, chars_format::scientific, 39, "1.057982927932243910618126392364501953125e-07"},
    {0x1.4da57ep-23f, chars_format::scientific, 39, "1.553662372089092968963086605072021484375e-07"},
    {0x1.4fcdacp-22f, chars_format::scientific, 37, "3.1274129241865011863410472869873046875e-07"},
    {0x1.5eaff4p-21f, chars_format::scientific, 36, "6.532060297104180790483951568603515625e-07"},
    {0x1.d2f696p-20f, chars_format::scientific, 37, "1.7395735767422593198716640472412109375e-06"},
    {0x1.e4400cp-19f, chars_format::scientific, 35, "3.60794501830241642892360687255859375e-06"},
    {0x1.03e624p-18f, chars_format::scientific, 34, "3.8727966966689564287662506103515625e-06"},
    {0x1.bdb65ep-17f, chars_format::scientific, 35, "1.32832637973478995263576507568359375e-05"},
    {0x1.57fb84p-16f, chars_format::scientific, 33, "2.050295370281673967838287353515625e-05"},
    {0x1.fd2d62p-15f, chars_format::scientific, 33, "6.069866140023805201053619384765625e-05"},
    {0x1.ca0c58p-14f, chars_format::scientific, 31, "1.0920720524154603481292724609375e-04"},
    {0x1.988f70p-13f, chars_format::scientific, 29, "1.94816733710467815399169921875e-04"},
    {0x1.032dd6p-12f, chars_format::scientific, 31, "2.4717240012250840663909912109375e-04"},
    {0x1.571b08p-11f, chars_format::scientific, 28, "6.5442197956144809722900390625e-04"},
    {0x1.53bedap-10f, chars_format::scientific, 30, "1.296026282943785190582275390625e-03"},
    {0x1.ab2f36p-9f, chars_format::scientific, 29, "3.25915846042335033416748046875e-03"},
    {0x1.7293dap-8f, chars_format::scientific, 28, "5.6545645929872989654541015625e-03"},
    {0x1.825eb6p-7f, chars_format::scientific, 28, "1.1791075579822063446044921875e-02"},
    {0x1.f45aa0p-6f, chars_format::scientific, 23, "3.05391848087310791015625e-02"},
    {0x1.854d96p-5f, chars_format::scientific, 26, "4.75223474204540252685546875e-02"},
    {0x1.5650cep-4f, chars_format::scientific, 25, "8.3573155105113983154296875e-02"},
    {0x1.03acdap-3f, chars_format::scientific, 25, "1.2679453194141387939453125e-01"},
    {0x1.6b9416p-2f, chars_format::scientific, 24, "3.550570905208587646484375e-01"},
    {0x1.a8544ap-1f, chars_format::scientific, 23, "8.28768074512481689453125e-01"},
    {0x1.0693f6p+0f, chars_format::scientific, 23, "1.02569520473480224609375e+00"},
    {0x1.b9476ep+1f, chars_format::scientific, 22, "3.4474923610687255859375e+00"},
    {0x1.3cb752p+2f, chars_format::scientific, 21, "4.948688983917236328125e+00"},
    {0x1.bb8a64p+3f, chars_format::scientific, 20, "1.38606433868408203125e+01"},
    {0x1.1de906p+4f, chars_format::scientific, 20, "1.78693904876708984375e+01"},
    {0x1.d8e834p+5f, chars_format::scientific, 18, "5.911338043212890625e+01"},
    {0x1.27cd38p+6f, chars_format::scientific, 16, "7.3950408935546875e+01"},
    {0x1.3cdcd6p+7f, chars_format::scientific, 18, "1.584313201904296875e+02"},
    {0x1.392656p+8f, chars_format::scientific, 17, "3.13149749755859375e+02"},
    {0x1.c96aa8p+9f, chars_format::scientific, 14, "9.14833251953125e+02"},
    {0x1.28b6b2p+10f, chars_format::scientific, 16, "1.1868546142578125e+03"},
    {0x1.786090p+11f, chars_format::scientific, 12, "3.011017578125e+03"},
    {0x1.79c6f6p+12f, chars_format::scientific, 14, "6.04443505859375e+03"},
    {0x1.ef1840p+13f, chars_format::scientific, 9, "1.584303125e+04"},
    {0x1.539fd0p+14f, chars_format::scientific, 10, "2.1735953125e+04"},
    {0x1.b31804p+15f, chars_format::scientific, 11, "5.56920078125e+04"},
    {0x1.ad4a9cp+16f, chars_format::scientific, 11, "1.09898609375e+05"},
    {0x1.4c43a6p+17f, chars_format::scientific, 11, "1.70119296875e+05"},
    {0x1.5598c6p+18f, chars_format::scientific, 10, "3.4979509375e+05"},
    {0x1.73695ep+19f, chars_format::scientific, 9, "7.606509375e+05"},
    {0x1.234f2ap+20f, chars_format::scientific, 9, "1.193202625e+06"},
    {0x1.0a4cc8p+21f, chars_format::scientific, 6, "2.181529e+06"},
    {0x1.90abd2p+22f, chars_format::scientific, 7, "6.5645965e+06"},
    {0x1.62dde8p+23f, chars_format::scientific, 7, "1.1628276e+07"},
    {0x1.9e3a8cp+24f, chars_format::scientific, 7, "2.7146892e+07"},
    {0x1.53a3eap+25f, chars_format::scientific, 7, "4.4517332e+07"},
    {0x1.41a1cep+26f, chars_format::scientific, 7, "8.4313912e+07"},
    {0x1.8fdda4p+27f, chars_format::scientific, 8, "2.09644832e+08"},
    {0x1.d0322ap+28f, chars_format::scientific, 8, "4.86744736e+08"},
    {0x1.cdb764p+29f, chars_format::scientific, 8, "9.68289408e+08"},
    {0x1.7620d8p+30f, chars_format::scientific, 9, "1.569207808e+09"},
    {0x1.c18df6p+31f, chars_format::scientific, 9, "3.771136768e+09"},
    {0x1.240cf8p+32f, chars_format::scientific, 9, "4.899796992e+09"},
    {0x1.81669ap+33f, chars_format::scientific, 10, "1.2931904512e+10"},
    {0x1.3be30cp+34f, chars_format::scientific, 10, "2.1198811136e+10"},
    {0x1.d1e6e4p+35f, chars_format::scientific, 10, "6.2532296704e+10"},
    {0x1.06b274p+36f, chars_format::scientific, 10, "7.0517211136e+10"},
    {0x1.a74284p+37f, chars_format::scientific, 11, "2.27235889152e+11"},
    {0x1.9fd3e6p+38f, chars_format::scientific, 11, "4.46491623424e+11"},
    {0x1.e2cec4p+39f, chars_format::scientific, 12, "1.036821594112e+12"},
    {0x1.3d5d32p+40f, chars_format::scientific, 11, "1.36306819072e+12"},
    {0x1.accccap+41f, chars_format::scientific, 12, "3.683363586048e+12"},
    {0x1.a120ccp+42f, chars_format::scientific, 12, "7.166206410752e+12"},
    {0x1.55a028p+43f, chars_format::scientific, 13, "1.1738166591488e+13"},
    {0x1.035296p+44f, chars_format::scientific, 13, "1.7820513468416e+13"},
    {0x1.22d1aap+45f, chars_format::scientific, 13, "3.9969859043328e+13"},
    {0x1.eb8eaep+46f, chars_format::scientific, 14, "1.35118253457408e+14"},
    {0x1.490d0ep+47f, chars_format::scientific, 14, "1.80897697497088e+14"},
    {0x1.9da088p+48f, chars_format::scientific, 14, "4.54787778740224e+14"},
    {0x1.e7fab4p+49f, chars_format::scientific, 15, "1.073077848899584e+15"},
    {0x1.98a534p+50f, chars_format::scientific, 14, "1.79724114460672e+15"},
    {0x1.93aeeap+51f, chars_format::scientific, 15, "3.550835489374208e+15"},
    {0x1.3df680p+52f, chars_format::scientific, 15, "5.593662327095296e+15"},
    {0x1.c763f6p+53f, chars_format::scientific, 15, "1.602262782705664e+16"},
    {0x1.8b669ep+54f, chars_format::scientific, 15, "2.782386114789376e+16"},
    {0x1.73e5b6p+55f, chars_format::scientific, 16, "5.2339893103230976e+16"},
    {0x1.a13d18p+56f, chars_format::scientific, 17, "1.17442238576852992e+17"},
    {0x1.a0797ep+57f, chars_format::scientific, 17, "2.34454344768946176e+17"},
    {0x1.c07a80p+58f, chars_format::scientific, 17, "5.04941918963105792e+17"},
    {0x1.729388p+59f, chars_format::scientific, 17, "8.34463629662224384e+17"},
    {0x1.edfb70p+60f, chars_format::scientific, 18, "2.224697951572197376e+18"},
    {0x1.3d6782p+61f, chars_format::scientific, 17, "2.85892402114199552e+18"},
    {0x1.b121e8p+62f, chars_format::scientific, 18, "7.802620494837972992e+18"},
    {0x1.0efc5ap+63f, chars_format::scientific, 18, "9.763290520209063936e+18"},
    {0x1.b7dba0p+64f, chars_format::scientific, 19, "3.1695102724410441728e+19"},
    {0x1.ec2306p+65f, chars_format::scientific, 19, "7.0924388975830368256e+19"},
    {0x1.2e2d28p+66f, chars_format::scientific, 19, "8.7096415015485308928e+19"},
    {0x1.e02208p+67f, chars_format::scientific, 20, "2.76777792668052750336e+20"},
    {0x1.402636p+68f, chars_format::scientific, 20, "3.69106968238077509632e+20"},
    {0x1.11f97cp+69f, chars_format::scientific, 20, "6.31742296991907971072e+20"},
    {0x1.74db2ap+70f, chars_format::scientific, 21, "1.719495307615820316672e+21"},
    {0x1.94a32ap+71f, chars_format::scientific, 21, "3.732120907777931476992e+21"},
    {0x1.c272dcp+72f, chars_format::scientific, 21, "8.309311323384498356224e+21"},
    {0x1.36ca40p+73f, chars_format::scientific, 22, "1.1466128622488263852032e+22"},
    {0x1.5f6fbep+74f, chars_format::scientific, 22, "2.5931436172223350571008e+22"},
    {0x1.95ec4ep+75f, chars_format::scientific, 22, "5.9903671176748022628352e+22"},
    {0x1.6b3912p+76f, chars_format::scientific, 23, "1.07204487170660958732288e+23"},
    {0x1.10992ap+77f, chars_format::scientific, 23, "1.60913632700346331561984e+23"},
    {0x1.74a25ep+78f, chars_format::scientific, 23, "4.39928869395322133020672e+23"},
    {0x1.43f462p+79f, chars_format::scientific, 22, "7.6491622058254812577792e+23"},
    {0x1.f12ca2p+80f, chars_format::scientific, 24, "2.347839472055691035803648e+24"},
    {0x1.2b7f18p+81f, chars_format::scientific, 22, "2.8286640885152838844416e+24"},
    {0x1.a40704p+82f, chars_format::scientific, 24, "7.934093352976572433301504e+24"},
    {0x1.35d5f8p+83f, chars_format::scientific, 23, "1.17052661598219352932352e+25"},
    {0x1.c2c9d2p+84f, chars_format::scientific, 25, "3.4060605519118462894473216e+25"},
    {0x1.47bf20p+85f, chars_format::scientific, 23, "4.95276631635027751337984e+25"},
    {0x1.60b728p+86f, chars_format::scientific, 25, "1.0660170486011939073818624e+26"},
    {0x1.3354c8p+87f, chars_format::scientific, 26, "1.85770297377533474371534848e+26"},
    {0x1.e9e512p+88f, chars_format::scientific, 26, "5.92246479757524141957185536e+26"},
    {0x1.c4b6cap+89f, chars_format::scientific, 27, "1.094595334815995103451021312e+27"},
    {0x1.799cb8p+90f, chars_format::scientific, 27, "1.826020469467809704300249088e+27"},
    {0x1.1afa36p+91f, chars_format::scientific, 27, "2.736789351009782551090823168e+27"},
    {0x1.80c214p+92f, chars_format::scientific, 27, "7.442304364233212615194574848e+27"},
    {0x1.657890p+93f, chars_format::scientific, 28, "1.3828987453168434783077793792e+28"},
    {0x1.5ce17cp+94f, chars_format::scientific, 28, "2.6993344325171312829134798848e+28"},
    {0x1.3f1e9ap+95f, chars_format::scientific, 28, "4.9381356576017938861904625664e+28"},
    {0x1.874612p+96f, chars_format::scientific, 29, "1.21093348650115637567232671744e+29"},
    {0x1.5f4d5ep+97f, chars_format::scientific, 29, "2.17445539275703670631001227264e+29"},
    {0x1.45b1bep+98f, chars_format::scientific, 29, "4.03190021246562727728269754368e+29"},
    {0x1.a570f4p+99f, chars_format::scientific, 30, "1.043437928672039460753056464896e+30"},
    {0x1.f5106ep+100f, chars_format::scientific, 30, "2.481149635102733266542145830912e+30"},
    {0x1.d84424p+101f, chars_format::scientific, 30, "4.677097651091265616934539886592e+30"},
    {0x1.3d6c56p+102f, chars_format::scientific, 30, "6.287213966425746785671183335424e+30"},
    {0x1.9d8cf0p+103f, chars_format::scientific, 31, "1.6382424580981433623378525159424e+31"},
    {0x1.e2e73ep+104f, chars_format::scientific, 29, "3.82595403225449575379724402688e+31"},
    {0x1.2d6594p+105f, chars_format::scientific, 30, "4.775822764761364886543157690368e+31"},
    {0x1.ce43bap+106f, chars_format::scientific, 31, "1.4649748574980240963539344293888e+32"},
    {0x1.b3ea00p+107f, chars_format::scientific, 32, "2.76293361488025452794185737306112e+32"},
    {0x1.03a052p+108f, chars_format::scientific, 32, "3.29115373194929392757058784198656e+32"},
    {0x1.6f59e0p+109f, chars_format::scientific, 31, "9.3134561945576656911623262306304e+32"},
    {0x1.05adacp+110f, chars_format::scientific, 33, "1.326867152522435745601434087849984e+33"},
    {0x1.2cdef0p+111f, chars_format::scientific, 33, "3.051192904788012466473218045116416e+33"},
    {0x1.e81552p+112f, chars_format::scientific, 33, "9.899505055765620068271358482579456e+33"},
    {0x1.bfa8f4p+113f, chars_format::scientific, 34, "1.8159245876954178992833811110166528e+34"},
    {0x1.a14810p+114f, chars_format::scientific, 33, "3.385389673673572296245535418875904e+34"},
    {0x1.f18b10p+115f, chars_format::scientific, 34, "8.0731001914916160681187088757948416e+34"},
    {0x1.8d6e30p+116f, chars_format::scientific, 32, "1.28973545052908058560090358153216e+35"},
    {0x1.9480c2p+117f, chars_format::scientific, 35, "2.62537431192608192877759864086986752e+35"},
    {0x1.60975cp+118f, chars_format::scientific, 34, "4.5768960676134050994895233721827328e+35"},
    {0x1.ab1bb2p+119f, chars_format::scientific, 36, "1.108836243133298765768030079592431616e+36"},
    {0x1.6a0c80p+120f, chars_format::scientific, 36, "1.879864992909653247408339011818749952e+36"},
    {0x1.2cac2cp+121f, chars_format::scientific, 36, "3.122362236102854007005843883842076672e+36"},
    {0x1.0baaf6p+122f, chars_format::scientific, 36, "5.559243043957593079267046257728684032e+36"},
    {0x1.098282p+123f, chars_format::scientific, 35, "1.10288454433706471446366546449924096e+37"},
    {0x1.122f8ap+124f, chars_format::scientific, 37, "2.2778456735621461875293910785310326784e+37"},
    {0x1.57f4c6p+125f, chars_format::scientific, 36, "5.714951736310127067226390054203031552e+37"},
    {0x1.05e028p+126f, chars_format::scientific, 36, "8.702309817313974757087535795024166912e+37"},
    {0x1.9d8424p+127f, chars_format::scientific, 38, "2.74828637805621292108186801756142829568e+38"},

    // Ryu Printf PrintDecimalPoint
    // These values exercise each codepath.
    {0x1.59bc8cp+92f, chars_format::scientific, 0, "7e+27"},
    {0x1.59bc8cp+92f, chars_format::scientific, 1, "6.7e+27"},
    {0x1.37da7cp-30f, chars_format::scientific, 0, "1e-09"},
    {0x1.37da7cp-30f, chars_format::scientific, 1, "1.1e-09"},
    {0x1.834c98p+29f, chars_format::scientific, 0, "8e+08"},
    {0x1.834c98p+29f, chars_format::scientific, 1, "8.1e+08"},

    // Test the maximum mantissa, which generates the most digits for each exponent.
    {0x0.fffffep-126f, chars_format::scientific, 111,
        "1."
        "1754942106924410754870294448492873488270524287458933338571745305715888704756189042655023513361811637878417"
        "96875e-38"},
    {0x1.fffffep-126f, chars_format::scientific, 111,
        "2."
        "3509885615147285834557659820715330266457179855179808553659262368500061299303460771170648513361811637878417"
        "96875e-38"},
    {0x1.fffffep-125f, chars_format::scientific, 110,
        "4."
        "7019771230294571669115319641430660532914359710359617107318524737000122598606921542341297026723623275756835"
        "9375e-38"},
    {0x1.fffffep-124f, chars_format::scientific, 109,
        "9."
        "4039542460589143338230639282861321065828719420719234214637049474000245197213843084682594053447246551513671"
        "875e-38"},
    {0x1.fffffep-123f, chars_format::scientific, 109,
        "1."
        "8807908492117828667646127856572264213165743884143846842927409894800049039442768616936518810689449310302734"
        "375e-37"},
    {0x1.fffffep-122f, chars_format::scientific, 108,
        "3."
        "7615816984235657335292255713144528426331487768287693685854819789600098078885537233873037621378898620605468"
        "75e-37"},
    {0x1.fffffep-121f, chars_format::scientific, 107,
        "7."
        "5231633968471314670584511426289056852662975536575387371709639579200196157771074467746075242757797241210937"
        "5e-37"},
    {0x1.fffffep-120f, chars_format::scientific, 107,
        "1."
        "5046326793694262934116902285257811370532595107315077474341927915840039231554214893549215048551559448242187"
        "5e-36"},
    {0x1.fffffep-119f, chars_format::scientific, 106,
        "3."
        "0092653587388525868233804570515622741065190214630154948683855831680078463108429787098430097103118896484375"
        "e-36"},
    {0x1.fffffep-118f, chars_format::scientific, 105,
        "6."
        "018530717477705173646760914103124548213038042926030989736771166336015692621685957419686019420623779296875e"
        "-36"},
    {0x1.fffffep-117f, chars_format::scientific, 105,
        "1."
        "203706143495541034729352182820624909642607608585206197947354233267203138524337191483937203884124755859375e"
        "-35"},
    {0x1.fffffep-116f, chars_format::scientific, 104,
        "2."
        "40741228699108206945870436564124981928521521717041239589470846653440627704867438296787440776824951171875e-"
        "35"},
    {0x1.fffffep-115f, chars_format::scientific, 103,
        "4."
        "8148245739821641389174087312824996385704304343408247917894169330688125540973487659357488155364990234375e-"
        "35"},
    {0x1.fffffep-114f, chars_format::scientific, 102,
        "9.629649147964328277834817462564999277140860868681649583578833866137625108194697531871497631072998046875e-"
        "35"},
    {0x1.fffffep-113f, chars_format::scientific, 102,
        "1.925929829592865655566963492512999855428172173736329916715766773227525021638939506374299526214599609375e-"
        "34"},
    {0x1.fffffep-112f, chars_format::scientific, 101,
        "3.85185965918573131113392698502599971085634434747265983343153354645505004327787901274859905242919921875e-"
        "34"},
    {0x1.fffffep-111f, chars_format::scientific, 100,
        "7.7037193183714626222678539700519994217126886949453196668630670929101000865557580254971981048583984375e-"
        "34"},
    {0x1.fffffep-110f, chars_format::scientific, 100,
        "1.5407438636742925244535707940103998843425377389890639333726134185820200173111516050994396209716796875e-"
        "33"},
    {0x1.fffffep-109f, chars_format::scientific, 99,
        "3.081487727348585048907141588020799768685075477978127866745226837164040034622303210198879241943359375e-"
        "33"},
    {0x1.fffffep-108f, chars_format::scientific, 98,
        "6.16297545469717009781428317604159953737015095595625573349045367432808006924460642039775848388671875e-33"},
    {0x1.fffffep-107f, chars_format::scientific, 98,
        "1.23259509093943401956285663520831990747403019119125114669809073486561601384892128407955169677734375e-32"},
    {0x1.fffffep-106f, chars_format::scientific, 97,
        "2.4651901818788680391257132704166398149480603823825022933961814697312320276978425681591033935546875e-32"},
    {0x1.fffffep-105f, chars_format::scientific, 96,
        "4.930380363757736078251426540833279629896120764765004586792362939462464055395685136318206787109375e-32"},
    {0x1.fffffep-104f, chars_format::scientific, 95,
        "9.86076072751547215650285308166655925979224152953000917358472587892492811079137027263641357421875e-32"},
    {0x1.fffffep-103f, chars_format::scientific, 95,
        "1.97215214550309443130057061633331185195844830590600183471694517578498562215827405452728271484375e-31"},
    {0x1.fffffep-102f, chars_format::scientific, 94,
        "3.9443042910061888626011412326666237039168966118120036694338903515699712443165481090545654296875e-31"},
    {0x1.fffffep-101f, chars_format::scientific, 93,
        "7.888608582012377725202282465333247407833793223624007338867780703139942488633096218109130859375e-31"},
    {0x1.fffffep-100f, chars_format::scientific, 93,
        "1.577721716402475545040456493066649481566758644724801467773556140627988497726619243621826171875e-30"},
    {0x1.fffffep-99f, chars_format::scientific, 92,
        "3.15544343280495109008091298613329896313351728944960293554711228125597699545323848724365234375e-30"},
    {0x1.fffffep-98f, chars_format::scientific, 91,
        "6.3108868656099021801618259722665979262670345788992058710942245625119539909064769744873046875e-30"},
    {0x1.fffffep-97f, chars_format::scientific, 91,
        "1.2621773731219804360323651944533195852534069157798411742188449125023907981812953948974609375e-29"},
    {0x1.fffffep-96f, chars_format::scientific, 90,
        "2.524354746243960872064730388906639170506813831559682348437689825004781596362590789794921875e-29"},
    {0x1.fffffep-95f, chars_format::scientific, 89,
        "5.04870949248792174412946077781327834101362766311936469687537965000956319272518157958984375e-29"},
    {0x1.fffffep-94f, chars_format::scientific, 89,
        "1.00974189849758434882589215556265566820272553262387293937507593000191263854503631591796875e-28"},
    {0x1.fffffep-93f, chars_format::scientific, 88,
        "2.0194837969951686976517843111253113364054510652477458787501518600038252770900726318359375e-28"},
    {0x1.fffffep-92f, chars_format::scientific, 87,
        "4.038967593990337395303568622250622672810902130495491757500303720007650554180145263671875e-28"},
    {0x1.fffffep-91f, chars_format::scientific, 86,
        "8.07793518798067479060713724450124534562180426099098351500060744001530110836029052734375e-28"},
    {0x1.fffffep-90f, chars_format::scientific, 86,
        "1.61558703759613495812142744890024906912436085219819670300012148800306022167205810546875e-27"},
    {0x1.fffffep-89f, chars_format::scientific, 85,
        "3.2311740751922699162428548978004981382487217043963934060002429760061204433441162109375e-27"},
    {0x1.fffffep-88f, chars_format::scientific, 84,
        "6.462348150384539832485709795600996276497443408792786812000485952012240886688232421875e-27"},
    {0x1.fffffep-87f, chars_format::scientific, 84,
        "1.292469630076907966497141959120199255299488681758557362400097190402448177337646484375e-26"},
    {0x1.fffffep-86f, chars_format::scientific, 83,
        "2.58493926015381593299428391824039851059897736351711472480019438080489635467529296875e-26"},
    {0x1.fffffep-85f, chars_format::scientific, 82,
        "5.1698785203076318659885678364807970211979547270342294496003887616097927093505859375e-26"},
    {0x1.fffffep-84f, chars_format::scientific, 82,
        "1.0339757040615263731977135672961594042395909454068458899200777523219585418701171875e-25"},
    {0x1.fffffep-83f, chars_format::scientific, 81,
        "2.067951408123052746395427134592318808479181890813691779840155504643917083740234375e-25"},
    {0x1.fffffep-82f, chars_format::scientific, 80,
        "4.13590281624610549279085426918463761695836378162738355968031100928783416748046875e-25"},
    {0x1.fffffep-81f, chars_format::scientific, 79,
        "8.2718056324922109855817085383692752339167275632547671193606220185756683349609375e-25"},
    {0x1.fffffep-80f, chars_format::scientific, 79,
        "1.6543611264984421971163417076738550467833455126509534238721244037151336669921875e-24"},
    {0x1.fffffep-79f, chars_format::scientific, 78,
        "3.308722252996884394232683415347710093566691025301906847744248807430267333984375e-24"},
    {0x1.fffffep-78f, chars_format::scientific, 77,
        "6.61744450599376878846536683069542018713338205060381369548849761486053466796875e-24"},
    {0x1.fffffep-77f, chars_format::scientific, 77,
        "1.32348890119875375769307336613908403742667641012076273909769952297210693359375e-23"},
    {0x1.fffffep-76f, chars_format::scientific, 76,
        "2.6469778023975075153861467322781680748533528202415254781953990459442138671875e-23"},
    {0x1.fffffep-75f, chars_format::scientific, 75,
        "5.293955604795015030772293464556336149706705640483050956390798091888427734375e-23"},
    {0x1.fffffep-74f, chars_format::scientific, 75,
        "1.058791120959003006154458692911267229941341128096610191278159618377685546875e-22"},
    {0x1.fffffep-73f, chars_format::scientific, 74,
        "2.11758224191800601230891738582253445988268225619322038255631923675537109375e-22"},
    {0x1.fffffep-72f, chars_format::scientific, 73,
        "4.2351644838360120246178347716450689197653645123864407651126384735107421875e-22"},
    {0x1.fffffep-71f, chars_format::scientific, 72,
        "8.470328967672024049235669543290137839530729024772881530225276947021484375e-22"},
    {0x1.fffffep-70f, chars_format::scientific, 72,
        "1.694065793534404809847133908658027567906145804954576306045055389404296875e-21"},
    {0x1.fffffep-69f, chars_format::scientific, 71,
        "3.38813158706880961969426781731605513581229160990915261209011077880859375e-21"},
    {0x1.fffffep-68f, chars_format::scientific, 70,
        "6.7762631741376192393885356346321102716245832198183052241802215576171875e-21"},
    {0x1.fffffep-67f, chars_format::scientific, 70,
        "1.3552526348275238478777071269264220543249166439636610448360443115234375e-20"},
    {0x1.fffffep-66f, chars_format::scientific, 69,
        "2.710505269655047695755414253852844108649833287927322089672088623046875e-20"},
    {0x1.fffffep-65f, chars_format::scientific, 68,
        "5.42101053931009539151082850770568821729966657585464417934417724609375e-20"},
    {0x1.fffffep-64f, chars_format::scientific, 68,
        "1.08420210786201907830216570154113764345993331517092883586883544921875e-19"},
    {0x1.fffffep-63f, chars_format::scientific, 67,
        "2.1684042157240381566043314030822752869198666303418576717376708984375e-19"},
    {0x1.fffffep-62f, chars_format::scientific, 66,
        "4.336808431448076313208662806164550573839733260683715343475341796875e-19"},
    {0x1.fffffep-61f, chars_format::scientific, 65,
        "8.67361686289615262641732561232910114767946652136743068695068359375e-19"},
    {0x1.fffffep-60f, chars_format::scientific, 65,
        "1.73472337257923052528346512246582022953589330427348613739013671875e-18"},
    {0x1.fffffep-59f, chars_format::scientific, 64,
        "3.4694467451584610505669302449316404590717866085469722747802734375e-18"},
    {0x1.fffffep-58f, chars_format::scientific, 63,
        "6.938893490316922101133860489863280918143573217093944549560546875e-18"},
    {0x1.fffffep-57f, chars_format::scientific, 63,
        "1.387778698063384420226772097972656183628714643418788909912109375e-17"},
    {0x1.fffffep-56f, chars_format::scientific, 62,
        "2.77555739612676884045354419594531236725742928683757781982421875e-17"},
    {0x1.fffffep-55f, chars_format::scientific, 61,
        "5.5511147922535376809070883918906247345148585736751556396484375e-17"},
    {0x1.fffffep-54f, chars_format::scientific, 61,
        "1.1102229584507075361814176783781249469029717147350311279296875e-16"},
    {0x1.fffffep-53f, chars_format::scientific, 60,
        "2.220445916901415072362835356756249893805943429470062255859375e-16"},
    {0x1.fffffep-52f, chars_format::scientific, 59,
        "4.44089183380283014472567071351249978761188685894012451171875e-16"},
    {0x1.fffffep-51f, chars_format::scientific, 58, "8.8817836676056602894513414270249995752237737178802490234375e-16"},
    {0x1.fffffep-50f, chars_format::scientific, 58, "1.7763567335211320578902682854049999150447547435760498046875e-15"},
    {0x1.fffffep-49f, chars_format::scientific, 57, "3.552713467042264115780536570809999830089509487152099609375e-15"},
    {0x1.fffffep-48f, chars_format::scientific, 56, "7.10542693408452823156107314161999966017901897430419921875e-15"},
    {0x1.fffffep-47f, chars_format::scientific, 56, "1.42108538681690564631221462832399993203580379486083984375e-14"},
    {0x1.fffffep-46f, chars_format::scientific, 55, "2.8421707736338112926244292566479998640716075897216796875e-14"},
    {0x1.fffffep-45f, chars_format::scientific, 54, "5.684341547267622585248858513295999728143215179443359375e-14"},
    {0x1.fffffep-44f, chars_format::scientific, 54, "1.136868309453524517049771702659199945628643035888671875e-13"},
    {0x1.fffffep-43f, chars_format::scientific, 53, "2.27373661890704903409954340531839989125728607177734375e-13"},
    {0x1.fffffep-42f, chars_format::scientific, 52, "4.5474732378140980681990868106367997825145721435546875e-13"},
    {0x1.fffffep-41f, chars_format::scientific, 51, "9.094946475628196136398173621273599565029144287109375e-13"},
    {0x1.fffffep-40f, chars_format::scientific, 51, "1.818989295125639227279634724254719913005828857421875e-12"},
    {0x1.fffffep-39f, chars_format::scientific, 50, "3.63797859025127845455926944850943982601165771484375e-12"},
    {0x1.fffffep-38f, chars_format::scientific, 49, "7.2759571805025569091185388970188796520233154296875e-12"},
    {0x1.fffffep-37f, chars_format::scientific, 49, "1.4551914361005113818237077794037759304046630859375e-11"},
    {0x1.fffffep-36f, chars_format::scientific, 48, "2.910382872201022763647415558807551860809326171875e-11"},
    {0x1.fffffep-35f, chars_format::scientific, 47, "5.82076574440204552729483111761510372161865234375e-11"},
    {0x1.fffffep-34f, chars_format::scientific, 47, "1.16415314888040910545896622352302074432373046875e-10"},
    {0x1.fffffep-33f, chars_format::scientific, 46, "2.3283062977608182109179324470460414886474609375e-10"},
    {0x1.fffffep-32f, chars_format::scientific, 45, "4.656612595521636421835864894092082977294921875e-10"},
    {0x1.fffffep-31f, chars_format::scientific, 44, "9.31322519104327284367172978818416595458984375e-10"},
    {0x1.fffffep-30f, chars_format::scientific, 44, "1.86264503820865456873434595763683319091796875e-09"},
    {0x1.fffffep-29f, chars_format::scientific, 43, "3.7252900764173091374686919152736663818359375e-09"},
    {0x1.fffffep-28f, chars_format::scientific, 42, "7.450580152834618274937383830547332763671875e-09"},
    {0x1.fffffep-27f, chars_format::scientific, 42, "1.490116030566923654987476766109466552734375e-08"},
    {0x1.fffffep-26f, chars_format::scientific, 41, "2.98023206113384730997495353221893310546875e-08"},
    {0x1.fffffep-25f, chars_format::scientific, 40, "5.9604641222676946199499070644378662109375e-08"},
    {0x1.fffffep-24f, chars_format::scientific, 40, "1.1920928244535389239899814128875732421875e-07"},
    {0x1.fffffep-23f, chars_format::scientific, 39, "2.384185648907077847979962825775146484375e-07"},
    {0x1.fffffep-22f, chars_format::scientific, 38, "4.76837129781415569595992565155029296875e-07"},
    {0x1.fffffep-21f, chars_format::scientific, 37, "9.5367425956283113919198513031005859375e-07"},
    {0x1.fffffep-20f, chars_format::scientific, 37, "1.9073485191256622783839702606201171875e-06"},
    {0x1.fffffep-19f, chars_format::scientific, 36, "3.814697038251324556767940521240234375e-06"},
    {0x1.fffffep-18f, chars_format::scientific, 35, "7.62939407650264911353588104248046875e-06"},
    {0x1.fffffep-17f, chars_format::scientific, 35, "1.52587881530052982270717620849609375e-05"},
    {0x1.fffffep-16f, chars_format::scientific, 34, "3.0517576306010596454143524169921875e-05"},
    {0x1.fffffep-15f, chars_format::scientific, 33, "6.103515261202119290828704833984375e-05"},
    {0x1.fffffep-14f, chars_format::scientific, 33, "1.220703052240423858165740966796875e-04"},
    {0x1.fffffep-13f, chars_format::scientific, 32, "2.44140610448084771633148193359375e-04"},
    {0x1.fffffep-12f, chars_format::scientific, 31, "4.8828122089616954326629638671875e-04"},
    {0x1.fffffep-11f, chars_format::scientific, 30, "9.765624417923390865325927734375e-04"},
    {0x1.fffffep-10f, chars_format::scientific, 30, "1.953124883584678173065185546875e-03"},
    {0x1.fffffep-9f, chars_format::scientific, 29, "3.90624976716935634613037109375e-03"},
    {0x1.fffffep-8f, chars_format::scientific, 28, "7.8124995343387126922607421875e-03"},
    {0x1.fffffep-7f, chars_format::scientific, 28, "1.5624999068677425384521484375e-02"},
    {0x1.fffffep-6f, chars_format::scientific, 27, "3.124999813735485076904296875e-02"},
    {0x1.fffffep-5f, chars_format::scientific, 26, "6.24999962747097015380859375e-02"},
    {0x1.fffffep-4f, chars_format::scientific, 26, "1.24999992549419403076171875e-01"},
    {0x1.fffffep-3f, chars_format::scientific, 25, "2.4999998509883880615234375e-01"},
    {0x1.fffffep-2f, chars_format::scientific, 24, "4.999999701976776123046875e-01"},
    {0x1.fffffep-1f, chars_format::scientific, 23, "9.99999940395355224609375e-01"},
    {0x1.fffffep+0f, chars_format::scientific, 23, "1.99999988079071044921875e+00"},
    {0x1.fffffep+1f, chars_format::scientific, 22, "3.9999997615814208984375e+00"},
    {0x1.fffffep+2f, chars_format::scientific, 21, "7.999999523162841796875e+00"},
    {0x1.fffffep+3f, chars_format::scientific, 21, "1.599999904632568359375e+01"},
    {0x1.fffffep+4f, chars_format::scientific, 20, "3.19999980926513671875e+01"},
    {0x1.fffffep+5f, chars_format::scientific, 19, "6.3999996185302734375e+01"},
    {0x1.fffffep+6f, chars_format::scientific, 19, "1.2799999237060546875e+02"},
    {0x1.fffffep+7f, chars_format::scientific, 18, "2.559999847412109375e+02"},
    {0x1.fffffep+8f, chars_format::scientific, 17, "5.11999969482421875e+02"},
    {0x1.fffffep+9f, chars_format::scientific, 17, "1.02399993896484375e+03"},
    {0x1.fffffep+10f, chars_format::scientific, 16, "2.0479998779296875e+03"},
    {0x1.fffffep+11f, chars_format::scientific, 15, "4.095999755859375e+03"},
    {0x1.fffffep+12f, chars_format::scientific, 14, "8.19199951171875e+03"},
    {0x1.fffffep+13f, chars_format::scientific, 14, "1.63839990234375e+04"},
    {0x1.fffffep+14f, chars_format::scientific, 13, "3.2767998046875e+04"},
    {0x1.fffffep+15f, chars_format::scientific, 12, "6.553599609375e+04"},
    {0x1.fffffep+16f, chars_format::scientific, 12, "1.310719921875e+05"},
    {0x1.fffffep+17f, chars_format::scientific, 11, "2.62143984375e+05"},
    {0x1.fffffep+18f, chars_format::scientific, 10, "5.2428796875e+05"},
    {0x1.fffffep+19f, chars_format::scientific, 10, "1.0485759375e+06"},
    {0x1.fffffep+20f, chars_format::scientific, 9, "2.097151875e+06"},
    {0x1.fffffep+21f, chars_format::scientific, 8, "4.19430375e+06"},
    {0x1.fffffep+22f, chars_format::scientific, 7, "8.3886075e+06"},
    {0x1.fffffep+23f, chars_format::scientific, 7, "1.6777215e+07"},
    {0x1.fffffep+24f, chars_format::scientific, 6, "3.355443e+07"},
    {0x1.fffffep+25f, chars_format::scientific, 6, "6.710886e+07"},
    {0x1.fffffep+26f, chars_format::scientific, 7, "1.3421772e+08"},
    {0x1.fffffep+27f, chars_format::scientific, 7, "2.6843544e+08"},
    {0x1.fffffep+28f, chars_format::scientific, 7, "5.3687088e+08"},
    {0x1.fffffep+29f, chars_format::scientific, 8, "1.07374176e+09"},
    {0x1.fffffep+30f, chars_format::scientific, 8, "2.14748352e+09"},
    {0x1.fffffep+31f, chars_format::scientific, 8, "4.29496704e+09"},
    {0x1.fffffep+32f, chars_format::scientific, 8, "8.58993408e+09"},
    {0x1.fffffep+33f, chars_format::scientific, 9, "1.717986816e+10"},
    {0x1.fffffep+34f, chars_format::scientific, 9, "3.435973632e+10"},
    {0x1.fffffep+35f, chars_format::scientific, 9, "6.871947264e+10"},
    {0x1.fffffep+36f, chars_format::scientific, 10, "1.3743894528e+11"},
    {0x1.fffffep+37f, chars_format::scientific, 10, "2.7487789056e+11"},
    {0x1.fffffep+38f, chars_format::scientific, 10, "5.4975578112e+11"},
    {0x1.fffffep+39f, chars_format::scientific, 11, "1.09951156224e+12"},
    {0x1.fffffep+40f, chars_format::scientific, 11, "2.19902312448e+12"},
    {0x1.fffffep+41f, chars_format::scientific, 11, "4.39804624896e+12"},
    {0x1.fffffep+42f, chars_format::scientific, 11, "8.79609249792e+12"},
    {0x1.fffffep+43f, chars_format::scientific, 12, "1.759218499584e+13"},
    {0x1.fffffep+44f, chars_format::scientific, 12, "3.518436999168e+13"},
    {0x1.fffffep+45f, chars_format::scientific, 12, "7.036873998336e+13"},
    {0x1.fffffep+46f, chars_format::scientific, 13, "1.4073747996672e+14"},
    {0x1.fffffep+47f, chars_format::scientific, 13, "2.8147495993344e+14"},
    {0x1.fffffep+48f, chars_format::scientific, 13, "5.6294991986688e+14"},
    {0x1.fffffep+49f, chars_format::scientific, 14, "1.12589983973376e+15"},
    {0x1.fffffep+50f, chars_format::scientific, 14, "2.25179967946752e+15"},
    {0x1.fffffep+51f, chars_format::scientific, 14, "4.50359935893504e+15"},
    {0x1.fffffep+52f, chars_format::scientific, 14, "9.00719871787008e+15"},
    {0x1.fffffep+53f, chars_format::scientific, 15, "1.801439743574016e+16"},
    {0x1.fffffep+54f, chars_format::scientific, 15, "3.602879487148032e+16"},
    {0x1.fffffep+55f, chars_format::scientific, 15, "7.205758974296064e+16"},
    {0x1.fffffep+56f, chars_format::scientific, 16, "1.4411517948592128e+17"},
    {0x1.fffffep+57f, chars_format::scientific, 16, "2.8823035897184256e+17"},
    {0x1.fffffep+58f, chars_format::scientific, 16, "5.7646071794368512e+17"},
    {0x1.fffffep+59f, chars_format::scientific, 17, "1.15292143588737024e+18"},
    {0x1.fffffep+60f, chars_format::scientific, 17, "2.30584287177474048e+18"},
    {0x1.fffffep+61f, chars_format::scientific, 17, "4.61168574354948096e+18"},
    {0x1.fffffep+62f, chars_format::scientific, 17, "9.22337148709896192e+18"},
    {0x1.fffffep+63f, chars_format::scientific, 18, "1.844674297419792384e+19"},
    {0x1.fffffep+64f, chars_format::scientific, 18, "3.689348594839584768e+19"},
    {0x1.fffffep+65f, chars_format::scientific, 18, "7.378697189679169536e+19"},
    {0x1.fffffep+66f, chars_format::scientific, 19, "1.4757394379358339072e+20"},
    {0x1.fffffep+67f, chars_format::scientific, 19, "2.9514788758716678144e+20"},
    {0x1.fffffep+68f, chars_format::scientific, 19, "5.9029577517433356288e+20"},
    {0x1.fffffep+69f, chars_format::scientific, 20, "1.18059155034866712576e+21"},
    {0x1.fffffep+70f, chars_format::scientific, 20, "2.36118310069733425152e+21"},
    {0x1.fffffep+71f, chars_format::scientific, 20, "4.72236620139466850304e+21"},
    {0x1.fffffep+72f, chars_format::scientific, 20, "9.44473240278933700608e+21"},
    {0x1.fffffep+73f, chars_format::scientific, 21, "1.888946480557867401216e+22"},
    {0x1.fffffep+74f, chars_format::scientific, 21, "3.777892961115734802432e+22"},
    {0x1.fffffep+75f, chars_format::scientific, 21, "7.555785922231469604864e+22"},
    {0x1.fffffep+76f, chars_format::scientific, 22, "1.5111571844462939209728e+23"},
    {0x1.fffffep+77f, chars_format::scientific, 22, "3.0223143688925878419456e+23"},
    {0x1.fffffep+78f, chars_format::scientific, 22, "6.0446287377851756838912e+23"},
    {0x1.fffffep+79f, chars_format::scientific, 23, "1.20892574755703513677824e+24"},
    {0x1.fffffep+80f, chars_format::scientific, 23, "2.41785149511407027355648e+24"},
    {0x1.fffffep+81f, chars_format::scientific, 23, "4.83570299022814054711296e+24"},
    {0x1.fffffep+82f, chars_format::scientific, 23, "9.67140598045628109422592e+24"},
    {0x1.fffffep+83f, chars_format::scientific, 24, "1.934281196091256218845184e+25"},
    {0x1.fffffep+84f, chars_format::scientific, 24, "3.868562392182512437690368e+25"},
    {0x1.fffffep+85f, chars_format::scientific, 24, "7.737124784365024875380736e+25"},
    {0x1.fffffep+86f, chars_format::scientific, 25, "1.5474249568730049750761472e+26"},
    {0x1.fffffep+87f, chars_format::scientific, 25, "3.0948499137460099501522944e+26"},
    {0x1.fffffep+88f, chars_format::scientific, 25, "6.1896998274920199003045888e+26"},
    {0x1.fffffep+89f, chars_format::scientific, 26, "1.23793996549840398006091776e+27"},
    {0x1.fffffep+90f, chars_format::scientific, 26, "2.47587993099680796012183552e+27"},
    {0x1.fffffep+91f, chars_format::scientific, 26, "4.95175986199361592024367104e+27"},
    {0x1.fffffep+92f, chars_format::scientific, 26, "9.90351972398723184048734208e+27"},
    {0x1.fffffep+93f, chars_format::scientific, 27, "1.980703944797446368097468416e+28"},
    {0x1.fffffep+94f, chars_format::scientific, 27, "3.961407889594892736194936832e+28"},
    {0x1.fffffep+95f, chars_format::scientific, 27, "7.922815779189785472389873664e+28"},
    {0x1.fffffep+96f, chars_format::scientific, 28, "1.5845631558379570944779747328e+29"},
    {0x1.fffffep+97f, chars_format::scientific, 28, "3.1691263116759141889559494656e+29"},
    {0x1.fffffep+98f, chars_format::scientific, 28, "6.3382526233518283779118989312e+29"},
    {0x1.fffffep+99f, chars_format::scientific, 29, "1.26765052467036567558237978624e+30"},
    {0x1.fffffep+100f, chars_format::scientific, 29, "2.53530104934073135116475957248e+30"},
    {0x1.fffffep+101f, chars_format::scientific, 29, "5.07060209868146270232951914496e+30"},
    {0x1.fffffep+102f, chars_format::scientific, 30, "1.014120419736292540465903828992e+31"},
    {0x1.fffffep+103f, chars_format::scientific, 30, "2.028240839472585080931807657984e+31"},
    {0x1.fffffep+104f, chars_format::scientific, 30, "4.056481678945170161863615315968e+31"},
    {0x1.fffffep+105f, chars_format::scientific, 30, "8.112963357890340323727230631936e+31"},
    {0x1.fffffep+106f, chars_format::scientific, 31, "1.6225926715780680647454461263872e+32"},
    {0x1.fffffep+107f, chars_format::scientific, 31, "3.2451853431561361294908922527744e+32"},
    {0x1.fffffep+108f, chars_format::scientific, 31, "6.4903706863122722589817845055488e+32"},
    {0x1.fffffep+109f, chars_format::scientific, 32, "1.29807413726245445179635690110976e+33"},
    {0x1.fffffep+110f, chars_format::scientific, 32, "2.59614827452490890359271380221952e+33"},
    {0x1.fffffep+111f, chars_format::scientific, 32, "5.19229654904981780718542760443904e+33"},
    {0x1.fffffep+112f, chars_format::scientific, 33, "1.038459309809963561437085520887808e+34"},
    {0x1.fffffep+113f, chars_format::scientific, 33, "2.076918619619927122874171041775616e+34"},
    {0x1.fffffep+114f, chars_format::scientific, 33, "4.153837239239854245748342083551232e+34"},
    {0x1.fffffep+115f, chars_format::scientific, 33, "8.307674478479708491496684167102464e+34"},
    {0x1.fffffep+116f, chars_format::scientific, 34, "1.6615348956959416982993368334204928e+35"},
    {0x1.fffffep+117f, chars_format::scientific, 34, "3.3230697913918833965986736668409856e+35"},
    {0x1.fffffep+118f, chars_format::scientific, 34, "6.6461395827837667931973473336819712e+35"},
    {0x1.fffffep+119f, chars_format::scientific, 35, "1.32922791655675335863946946673639424e+36"},
    {0x1.fffffep+120f, chars_format::scientific, 35, "2.65845583311350671727893893347278848e+36"},
    {0x1.fffffep+121f, chars_format::scientific, 35, "5.31691166622701343455787786694557696e+36"},
    {0x1.fffffep+122f, chars_format::scientific, 36, "1.063382333245402686911575573389115392e+37"},
    {0x1.fffffep+123f, chars_format::scientific, 36, "2.126764666490805373823151146778230784e+37"},
    {0x1.fffffep+124f, chars_format::scientific, 36, "4.253529332981610747646302293556461568e+37"},
    {0x1.fffffep+125f, chars_format::scientific, 36, "8.507058665963221495292604587112923136e+37"},
    {0x1.fffffep+126f, chars_format::scientific, 37, "1.7014117331926442990585209174225846272e+38"},
    {0x1.fffffep+127f, chars_format::scientific, 37, "3.4028234663852885981170418348451692544e+38"},
};

#endif // FLOAT_SCIENTIFIC_PRECISION_TO_CHARS_TEST_CASES_HPP
