/*
    Bear Engine - Editor library

    Copyright (C) 20052011 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/accordion.cpp
 * \brief Implementation of the accordion class.
 * \author Julien Jorge
 */
#include "bf/accordion.hpp"

#include <wx/sizer.h>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The parent control of the accordion.
 * \param orient The orientation of the accordion: wxVERTICAL or wxHORIZONTAL.
 */
bf::accordion::accordion( wxWindow* parent, int orient )
  : wxPanel(parent), m_current_button(NULL)
{
  m_content = new wxBoxSizer(orient);
  SetSizer(m_content);
} // accordion::accordion()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add something in the accordion.
 * \param c The control to add.
 * \param label The label associated to this control.
 */
void bf::accordion::add( wxWindow* c, const wxString& label )
{
  wxSizer* s = new wxBoxSizer( wxHORIZONTAL );
  s->Add( c, 0, wxEXPAND );
  add( s, label );
} // accordion::add()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add something in the accordion.
 * \param c The control to add.
 * \param label The label associated to this control.
 */
void bf::accordion::add( wxSizer* s, const wxString& label )
{
  wxToggleButton* btn = new wxToggleButton( this, wxID_ANY, label );
  Connect( btn->GetId(), wxEVT_COMMAND_TOGGLEBUTTON_CLICKED,
           wxCommandEventHandler(accordion::on_button) );
  m_sizers[btn] = s;

  m_content->Add( btn, 0, wxEXPAND );
  m_content->Add( s, 0, wxEXPAND );
  m_content->Show(s, false, true);
  m_content->Layout();
} // accordion::add()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on a button of the accordion.
 * \param event The event generated by the click.
 */
void bf::accordion::on_button( wxCommandEvent& event )
{
  wxToggleButton* btn = dynamic_cast<wxToggleButton*>(event.GetEventObject());
  CLAW_PRECOND( btn != NULL );

  if ( btn == m_current_button )
    {
      m_content->Show(m_sizers[m_current_button], false, true);
      m_current_button = NULL;
    }
  else
    {
      if ( m_current_button != NULL )
        {
          m_content->Show(m_sizers[m_current_button], false, true);
          m_current_button->SetValue(false);
        }

      m_content->Show(m_sizers[btn], true, true);
      m_current_button = btn;
    }

  m_content->Layout();
} // accordion::on_button()
