/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "_xitk.h"
#include "inputtext.h"
#include "menu.h"
#include "window.h"
#include "_backend.h"

#define NEW_CLIPBOARD

#ifndef NEW_CLIPBOARD
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "xitk_x11.h"
#endif

#define DIRTY_CHANGED     1
#define DIRTY_BEFORE      2
#define DIRTY_AFTER       4
#define DIRTY_MOVE_LEFT   8
#define DIRTY_MOVE_RIGHT 16

/** string buf overread safety. */
#define SBUF_PAD 4
#define SBUF_PLUG "\x00\x00 \x80"
/** string buf block size. */
#define SBUF_MASK 127
/** built in size */
#define SBUF_BUILTIN_SIZE 128

typedef struct _inputtext_menu_s _inputtext_menu_t;

typedef enum {
  _IT_NORMAL = 0,
  _IT_FOCUS,
  _IT_END
} _it_state_t;

typedef struct {
  xitk_widget_t           w;

  xitk_short_string_t     fontname;
  uint32_t                color[_IT_END];

  xitk_hv_t               img_state_list[XITK_IMG_STATE_LIST_SIZE];
  xitk_part_image_t       skin; /** background image. */

  int                     cursor_focus;  /** << mouse pointer gfx. */
  uint32_t                auto_callback; /** << XITK_WIDGET_STATE_FOCUS to fire callback on focus loss with changed text. */

  xitk_string_callback_t  callback;

  _inputtext_menu_t      *menu;

  struct {
    char                 *buf;
    xitk_part_image_t     temp_img;
    int                   size; /** text bytes _without_ trailing 0. */
    int                   used; /** text bytes _without_ trailing 0. */
    int                   draw_start; /** text bytes. */
    int                   draw_stop;  /** text bytes. */
    int                   cursor_pos; /** text bytes. */
    int                   dirty; /** see DIRTY_*. */
    int                   box_start; /** pixels. */
    int                   box_width; /** pixels. */
    int                   shift;     /** pixels. */
    int                   width;     /** pixels. */
  } text;

  int                     max_length; /** text bytes. */

  char                    builtin_buf[SBUF_BUILTIN_SIZE];
} _inputtext_private_t;

struct _inputtext_menu_s {
  _inputtext_private_t *wp;
  int refs;
};

typedef struct {
  int bytes;
  int pixels;
} _inputtext_pos_t;

/* exact text width may be fractional, dont try to assemble it from text fragments.
 * xitk_font_get_text_width () may be slow, try to minimize its use by phonebook search.
 * negative values count from end of text.
 * input: pos.bytes = text length, pos.pixels = wanted pos.
 * output: pos = found pos. */
static int _inputtext_find_text_pos (const char *text, _inputtext_pos_t *pos, xitk_font_t *fs, int round_up) {
  const uint8_t *btext = (const uint8_t *)text;
  _inputtext_pos_t want, ref, try, best;
  int tries;

  if ((pos->bytes == 0) || (pos->pixels == 0)) {
    pos->bytes = 0;
    pos->pixels = 0;
    return 1;
  }

  tries = 12;
  ref.bytes = 0;
  ref.pixels = 0;
  best.bytes = 0;
  best.pixels = 0;
  want.pixels = pos->pixels;
  if (want.pixels < 0) {
    /* start with 200 bytes, or the entire text if less. */
    int best_diff = -pos->pixels /* + best.pixels */, last_bytes = 1;
    if (pos->bytes < 0) {
      want.bytes = pos->bytes;
    } else {
      btext += pos->bytes;
      want.bytes = -pos->bytes;
    }
    try.bytes = -200;
    try.pixels = -1000000;
    do {
      int diff;
      if (try.bytes >= 0) {
        try.bytes = 0;
      } else if (try.bytes <= want.bytes) {
        try.bytes = want.bytes;
      } else {
        while ((btext[try.bytes] & 0xc0) == 0x80)
          try.bytes += 1;
        if (try.bytes > 0)
          try.bytes = 0;
      }
      if (try.bytes == last_bytes)
        break;
      last_bytes = try.bytes;
      /* naive optimization: do it from the near side. */
      tries -= 1;
      if (try.bytes <= ref.bytes) {
        try.pixels = -xitk_font_get_text_width (fs, (char *)btext + try.bytes, -try.bytes);
        ref = try;
      } else if (try.bytes > (ref.bytes >> 1)) {
        try.pixels = -xitk_font_get_text_width (fs, (char *)btext + try.bytes, -try.bytes);
      } else {
        try.pixels = ref.pixels + xitk_font_get_text_width (fs, (char *)btext + ref.bytes, -ref.bytes + try.bytes);
      }
      diff = want.pixels - try.pixels;
      diff = diff < 0 ? -diff : diff;
      if (diff < best_diff) {
        best_diff = diff;
        best = try;
      } else if (try.bytes == best.bytes) {
        break;
      }
      diff = try.pixels;
      if (diff >= 0)
        diff = 1;
      try.bytes = (try.bytes * want.pixels + (diff >> 1)) / diff;
    } while (tries > 0);
    if (round_up && (want.pixels < best.pixels)) {
      int b = best.bytes;
      while ((btext[--b] & 0xc0) == 0x80) ;
      if (b >= want.bytes) {
        best.bytes = b;
        best.pixels = -xitk_font_get_text_width (fs, (char *)btext + best.bytes, -best.bytes);
        tries -= 1;
      }
    }
    if (pos->bytes >= 0)
      best.bytes += pos->bytes;
  } else {
    int best_diff = pos->pixels /* - best.pixels */, last_bytes = -1;
    if (pos->bytes < 0) {
      btext += pos->bytes;
      want.bytes = -pos->bytes;
    } else {
      want.bytes = pos->bytes;
    }
    try.bytes = 200;
    try.pixels = 1000000;
    do {
      int diff;
      if (try.bytes <= 0) {
        try.bytes = 0;
      } else if (try.bytes >= want.bytes) {
        try.bytes = want.bytes;
      } else {
        while ((btext[try.bytes] & 0xc0) == 0x80)
          try.bytes -= 1;
        if (try.bytes < 0)
          try.bytes = 0;
      }
      if (try.bytes == last_bytes)
        break;
      last_bytes = try.bytes;
      /* naive optimization: do it from the near side. */
      tries -= 1;
      if (try.bytes >= ref.bytes) {
        try.pixels = xitk_font_get_text_width (fs, (char *)btext, try.bytes);
        ref = try;
      } else if (try.bytes < (ref.bytes >> 1)) {
        try.pixels = xitk_font_get_text_width (fs, (char *)btext, try.bytes);
      } else {
        try.pixels = ref.pixels - xitk_font_get_text_width (fs, (char *)btext + try.bytes, ref.bytes - try.bytes);
      }
      diff = want.pixels - try.pixels;
      diff = diff < 0 ? -diff : diff;
      if (diff < best_diff) {
        best_diff = diff;
        best = try;
      } else if (try.bytes == best.bytes) {
        break;
      }
      diff = try.pixels;
      if (diff <= 0)
        diff = 1;
      try.bytes = (try.bytes * want.pixels + (diff >> 1)) / diff;
    } while (tries > 0);
    if (round_up && (want.pixels > best.pixels)) {
      int b = best.bytes;
      while (btext[b] && (btext[++b] & 0xc0) == 0x80) ;
      if (b <= want.bytes) {
        best.bytes = b;
        best.pixels = xitk_font_get_text_width (fs, (char *)btext, best.bytes);
        tries -= 1;
      }
    }
    if (pos->bytes < 0)
      best.bytes += pos->bytes;
  }

#if 0
  printf ("_inputtext_find_text_pos (%d, %d, %d) = (%d, %d) after %d iterations.\n",
    pos->bytes, pos->pixels, round_up, best.bytes, best.pixels, 12 - tries);
#endif
  *pos = best;
  return 1;
}

static void _inputtext_sbuf_init (_inputtext_private_t *wp) {
  memcpy (wp->builtin_buf, SBUF_PLUG, SBUF_PAD);
  memcpy (wp->builtin_buf + SBUF_PAD, SBUF_PLUG, SBUF_PAD);
  wp->text.buf = wp->builtin_buf + SBUF_PAD;
  wp->text.size = SBUF_BUILTIN_SIZE - 2 * SBUF_PAD;
  wp->text.used = 0;
  wp->text.cursor_pos = 0;
}

static uint32_t _inputtext_sbuf_insert (_inputtext_private_t *wp, const char *text, int size) {
  int l;

  if (!text)
    return 0;
  l = wp->max_length - wp->text.used;
  if (l >= size) {
    l = size;
  } else if ((text[l] & 0xc0) == 0x80) {
    const uint8_t *p;
    /* remove incomplete utf8. */
    for (p = (const uint8_t *)text + l; p > (const uint8_t *)text; ) {
      if ((*--p & 0xc0) != 0x80)
        break;
    }
    l = p - (const uint8_t *)text;
  }
  if (l <= 0)
    return 0;

  do {
    int nsize = (wp->text.used + l + 2 * SBUF_PAD + SBUF_MASK) & ~SBUF_MASK;
    char *nbuf;
    if (wp->text.size + 2 * SBUF_PAD >= nsize)
      break;
    if (wp->text.buf == wp->builtin_buf + SBUF_PAD) {
      nbuf = malloc (nsize);
      if (!nbuf)
        return 0;
      memcpy (nbuf, wp->builtin_buf, wp->text.used + 2 * SBUF_PAD);
    } else {
      nbuf = realloc (wp->text.buf - SBUF_PAD, nsize);
      if (!nbuf)
        return 0;
    }
    wp->text.buf = nbuf + SBUF_PAD;
    wp->text.size = nsize - 2 * SBUF_PAD;
  } while (0);

  memmove (wp->text.buf + wp->text.cursor_pos + l, wp->text.buf + wp->text.cursor_pos,
    wp->text.used - wp->text.cursor_pos + SBUF_PAD);
  memcpy (wp->text.buf + wp->text.cursor_pos, text, l);
  wp->text.used += l;
  wp->text.cursor_pos += l;
  return l;
}

static void _inputtext_sbuf_unset (_inputtext_private_t *wp) {
  if (wp->text.buf != wp->builtin_buf + SBUF_PAD) {
    free (wp->text.buf - SBUF_PAD);
    wp->text.buf = NULL;
    wp->text.size = 0;
    wp->text.used = 0;
  }
}

/*
 *
 */
static void _cursor_focus (_inputtext_private_t *wp, int focus) {
  wp->cursor_focus = focus;
  if (wp->w.wl->xwin)
    xitk_window_define_window_cursor (wp->w.wl->xwin, focus ? xitk_cursor_xterm : xitk_cursor_default);
}

/*
 * Paint the input text box.
 */
static void _inputtext_paint (_inputtext_private_t *wp, const widget_event_t *event) {
  xitk_font_t *fs = NULL;
  int xsize, ysize, lbear, rbear, width, asc, des, cursor_x, yoff = 0, fire = 0;
  unsigned int fg = 0, bg_state;
  _it_state_t state;
  widget_event_t full_event;
  XITK_HV_INIT;

  if (!XITK_HV_H (wp->w.size))
    return;

  if (!(wp->w.state & XITK_WIDGET_STATE_VISIBLE)) {
    if (wp->cursor_focus)
      _cursor_focus (wp, 0);
    return;
  }

  {
    uint32_t changed_flags = wp->w.state ^ wp->w.shown_state;
    if (changed_flags & (XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS)) {
      _cursor_focus (wp, !!(wp->w.state & XITK_WIDGET_STATE_MOUSE));
      if (changed_flags & ~wp->w.state & wp->auto_callback) {
        fire = wp->text.dirty & DIRTY_CHANGED;
        wp->text.dirty &= ~DIRTY_CHANGED;
      }
    }
  }

  if (!event) {
    full_event.x = XITK_HV_H (wp->w.pos);
    full_event.y = XITK_HV_V (wp->w.pos);
    full_event.width = XITK_HV_H (wp->w.size);
    full_event.height = XITK_HV_V (wp->w.size);
    event = &full_event;
    wp->w.shown_state = wp->w.state;
  }

#ifdef XITK_PAINT_DEBUG
  printf ("xitk.inputtext.paint (%d, %d, %d, %d).\n", event->x, event->y, event->width, event->height);
#endif

  xsize = XITK_HV_H (wp->img_state_list[0]);
  ysize = XITK_HV_V (wp->img_state_list[0]);

  bg_state = xitk_image_find_state (wp->skin.num_states - 1, wp->w.state);
  state = (wp->w.state & (XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS)) ? _IT_FOCUS : _IT_NORMAL;

  /* Try to load font */

  if (wp->text.buf && wp->text.buf[0]) {
    if (wp->fontname.s[0])
      fs = xitk_font_load_font (wp->w.wl->xitk, wp->fontname.s);
    if (!fs)
      fs = xitk_font_load_font (wp->w.wl->xitk, xitk_get_cfg_string (wp->w.wl->xitk, XITK_SYSTEM_FONT));
    if (!fs)
      XITK_DIE ("%s()@%d: xitk_font_load_font() failed. Exiting\n", __FUNCTION__, __LINE__);
    xitk_font_string_extent (fs, wp->text.buf, &lbear, &rbear, &width, &asc, &des);
  }

  /*  Some colors configurations */
  fg = wp->color[state];
  if (fg & 0x80000000) {
    fg = xitk_get_cfg_num (wp->w.wl->xitk, (fg == XITK_NOSKIN_TEXT_INV)
        ? ((wp->w.state & XITK_WIDGET_STATE_ENABLE) ? XITK_WHITE_COLOR : XITK_DISABLED_WHITE_COLOR)
        : ((wp->w.state & XITK_WIDGET_STATE_ENABLE) ? XITK_BLACK_COLOR : XITK_DISABLED_BLACK_COLOR));
  } else {
    if (!(wp->w.state & XITK_WIDGET_STATE_ENABLE))
      fg = xitk_disabled_color (fg);
    fg = xitk_color_db_get (wp->w.wl->xitk, fg);
  }

  cursor_x = 0;
  if (fs) {
    _inputtext_pos_t pos;
    cursor_x = -1;
    if (wp->text.cursor_pos <= wp->text.draw_start) {
      if (wp->text.dirty & DIRTY_BEFORE) {
        wp->text.draw_stop = wp->text.cursor_pos;
        wp->text.dirty |= DIRTY_MOVE_RIGHT;
      } else {
        wp->text.draw_start = wp->text.cursor_pos;
        wp->text.dirty |= DIRTY_MOVE_LEFT;
      }
    } else if (wp->text.cursor_pos >= wp->text.draw_stop) {
      wp->text.draw_stop = wp->text.cursor_pos;
      wp->text.dirty |= DIRTY_MOVE_RIGHT;
    }
    if (wp->text.cursor_pos >= wp->text.used) {
      wp->text.cursor_pos =
      wp->text.draw_stop  = wp->text.used;
      wp->text.dirty |= DIRTY_MOVE_RIGHT;
    }
    do {
      if (!(wp->text.dirty & ~DIRTY_CHANGED))
        break;
      if (!(wp->text.dirty & (DIRTY_MOVE_LEFT | DIRTY_MOVE_RIGHT))
        && (wp->text.dirty & (DIRTY_BEFORE | DIRTY_AFTER))) {
        /* keep render pos if cursor still visible.
         * pitfall: partial chars both left (inherited) and right (due to editing). */
        pos.bytes = wp->text.used - wp->text.draw_start;
        pos.pixels = wp->text.box_width - wp->text.shift;
        if (_inputtext_find_text_pos (wp->text.buf + wp->text.draw_start, &pos, fs, 1)) {
          int stop;
          pos.bytes += wp->text.draw_start;
          stop = pos.bytes;
          if (pos.pixels > wp->text.box_width - wp->text.shift) {
            while ((wp->text.buf[--stop] & 0xc0) == 0x80) ;
            if (stop < 0)
              stop = 0;
          }
          if (stop >= wp->text.cursor_pos) {
            wp->text.draw_stop = pos.bytes;
            wp->text.width = pos.pixels;
            break;
          }
        }
      }
      if (wp->text.dirty & DIRTY_MOVE_RIGHT) {
        /* right align cursor, with left fallback when leading text is too short.
         * check for right part as well. */
        pos.bytes = wp->text.cursor_pos;
        pos.pixels = -wp->text.box_width;
        if (_inputtext_find_text_pos (wp->text.buf, &pos, fs, 1)) {
          wp->text.draw_start = pos.bytes;
          wp->text.draw_stop = wp->text.cursor_pos;
          wp->text.width = cursor_x = -pos.pixels;
          wp->text.shift = pos.pixels + wp->text.box_width;
          if (wp->text.shift <= 0)
            break;
          wp->text.shift = 0;
          if (wp->text.cursor_pos >= wp->text.used)
            break;
          pos.bytes = wp->text.used - wp->text.draw_start;
          pos.pixels = wp->text.box_width;
          if (_inputtext_find_text_pos (wp->text.buf + wp->text.draw_start, &pos, fs, 1)) {
            pos.bytes += wp->text.draw_start;
            wp->text.draw_stop = pos.bytes;
            wp->text.width = pos.pixels;
          }
          break;
        }
      }
      {
        /* left align. */
        wp->text.shift = 0;
        cursor_x = 0;
        if (wp->text.dirty & (DIRTY_MOVE_LEFT | DIRTY_AFTER)) {
          pos.bytes = wp->text.used - wp->text.draw_start;
          pos.pixels = wp->text.box_width;
          if (_inputtext_find_text_pos (wp->text.buf + wp->text.draw_start, &pos, fs, 1)) {
            pos.bytes += wp->text.draw_start;
            wp->text.draw_stop = pos.bytes;
            wp->text.width = pos.pixels;
          }
        }
      }
    } while (0);
    wp->text.dirty &= DIRTY_CHANGED;
  }

  /*  Put text in the right place */
  wp->text.temp_img.image = xitk_image_temp_get (wp->w.wl->xitk);
  {
    int src_x = XITK_HV_H (wp->img_state_list[1 + bg_state]);
    int src_y = XITK_HV_V (wp->img_state_list[1 + bg_state]);

    xitk_part_image_copy (wp->w.wl, &wp->skin, &wp->text.temp_img,
      src_x, src_y, xsize, ysize, 0, 0);
    if (fs) {
      xitk_image_draw_string (wp->text.temp_img.image, fs,
        ysize + wp->text.box_start + wp->text.shift,
        ((ysize + asc + des + yoff) >> 1) - des,
        wp->text.buf + wp->text.draw_start,
        wp->text.draw_stop - wp->text.draw_start, fg);
      /* with 1 partial char left and/or right, fix borders. */
      if (wp->text.shift < 0)
        xitk_part_image_copy (wp->w.wl, &wp->skin, &wp->text.temp_img,
          src_x, src_y, wp->text.box_start, ysize, 0, 0);
      if (wp->text.shift + wp->text.width > wp->text.box_width)
        xitk_part_image_copy (wp->w.wl, &wp->skin, &wp->text.temp_img,
          src_x + wp->text.box_start + wp->text.box_width, src_y,
          xsize - wp->text.box_start - wp->text.box_width, ysize,
          wp->text.box_start + wp->text.box_width, 0);
    }
  }

  /* Draw cursor pointer */
  if (wp->w.state & XITK_WIDGET_STATE_FOCUS) {
    xitk_be_image_t *beimg;
    xitk_be_line_t xs[3];
    width = cursor_x >= 0
          ? cursor_x
          : xitk_font_get_text_width (fs,
              wp->text.buf + wp->text.draw_start,
              wp->text.cursor_pos - wp->text.draw_start);
#if 0 /* ifdef WITH_XFT */
    if (width)
      width += 2;
#endif
    width += ysize + wp->text.box_start + wp->text.shift;
    xs[0].x1 = width - 1; xs[0].x2 = width + 1; xs[0].y1 = xs[0].y2 = 2;
    xs[1].x1 = width - 1; xs[1].x2 = width + 1; xs[1].y1 = xs[1].y2 = ysize - 3;
    xs[2].x1 = xs[2].x2 = width; xs[2].y1 = 3; xs[2].y2 = ysize - 4;
    beimg = wp->text.temp_img.image->beimg;
    beimg->display->lock (beimg->display);
    beimg->draw_lines (beimg, xs, 3, fg, 0);
    beimg->display->unlock (beimg->display);
  }

  if (fs)
    xitk_font_unload_font (fs);

  xitk_part_image_draw (wp->w.wl, &wp->skin, &wp->text.temp_img,
    event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos), event->width, event->height, event->x, event->y);

  /* notify about _changed_ text on loss of focus (eg TAB) even without ENTER. */
  if (fire && wp->callback)
    wp->callback (&wp->w, wp->w.userdata, wp->text.buf);
}

typedef union {
  char b[4];
  int v;
} _inputtext_key_t;

static void _inputtext_menu_action (xitk_widget_t *w, xitk_menu_entry_t *me, void *data) {
  _inputtext_menu_t *menu = data;

  (void)w;
  /* paranoia */
  if (!menu)
    return;

  if (menu->wp && (menu->wp->menu == menu)) {
    if (me) {
      widget_event_t event = {.type = WIDGET_EVENT_KEY, .pressed = 1};
      _inputtext_key_t key;

      key.v = me->user_id;
      event.string = key.b;
      menu->wp->w.event (&menu->wp->w, &event);
    } else {
      menu->wp->auto_callback = XITK_WIDGET_STATE_FOCUS;
    }
  }
  if (!me && --menu->refs == 0)
    free (menu);
}

static int _inputtext_menu_open (_inputtext_private_t *wp, int x, int y) {
  if (!wp->menu) {
    wp->menu = malloc (sizeof (*wp->menu));
    if (!wp->menu)
      return 0;
    wp->menu->wp = wp;
    wp->menu->refs = 1;
  }

  if (wp->menu) {
    char buf[5][16];
    const char *cstr = _("Ctrl");
    size_t clen = xitk_find_byte (cstr, 0);

    if (clen > 12)
      cstr = "Ctrl", clen = 4;
    memcpy (buf[0], cstr, clen); memcpy (buf[0] + clen, "-C", 3);
    memcpy (buf[1], cstr, clen); memcpy (buf[1] + clen, "-X", 3);
    memcpy (buf[2], cstr, clen); memcpy (buf[2] + clen, "-V", 3);
    memcpy (buf[3], cstr, clen); memcpy (buf[3] + clen, "-T", 3);
    memcpy (buf[4], cstr, clen); memcpy (buf[4] + clen, "-K", 3);

    wp->auto_callback = 0;

    {
      const _inputtext_key_t
        _copy   = {{'C' & 0x1f, 0, 0, 0}},
        _cut    = {{'X' & 0x1f, 0, 0, 0}},
        _insert = {{'V' & 0x1f, 0, 0, 0}},
        _swap   = {{'T' & 0x1f, 0, 0, 0}},
        _clear  = {{'K' & 0x1f, 0, 0, 0}};
      xitk_menu_entry_t menu_entries[] = {
        { XITK_MENU_ENTRY_PLAIN,     _copy.v,   _("Copy"),       buf[0] },
        { XITK_MENU_ENTRY_PLAIN,     _cut.v,    _("Cut"),        buf[1] },
        { XITK_MENU_ENTRY_PLAIN,     _insert.v, _("Insert"),     buf[2] },
        { XITK_MENU_ENTRY_SEPARATOR, 0,         "Sep",           NULL },
        { XITK_MENU_ENTRY_PLAIN,     _swap.v,   _("Swap chars"), buf[3] },
        { XITK_MENU_ENTRY_PLAIN,     _clear.v,  _("Clear"),      buf[4] },
        { XITK_MENU_ENTRY_END,       0,         NULL,        NULL }
      };
      xitk_menu_widget_t menu = {
        .nw = {
          .wl = wp->w.wl,
          .userdata = wp->menu
        },
        .cb = _inputtext_menu_action,
        .menu_tree = &menu_entries[0]
      };
      xitk_widget_t *w = xitk_noskin_menu_create (&menu, wp->w.wl->win.x + x, wp->w.wl->win.y + y);

      if (!w)
        return 0;
      wp->menu->refs++;
      xitk_menu_show_menu (w);
      return 1;
    }
  }
  return 0;
}

/*
 * Handle click events.
 */
static int _inputtext_click (_inputtext_private_t *wp, int button, int bUp, int x, int y) {
  XITK_HV_INIT;

  (void)y;
  if (bUp)
    return 0;
  if (button == 3)
    return _inputtext_menu_open (wp, x, y);
  if (button != 1)
    return 0;

  if (!(wp->w.state & XITK_WIDGET_STATE_FOCUS))
    wp->w.state |= XITK_WIDGET_STATE_FOCUS;

  if ((wp->w.state & XITK_WIDGET_STATE_ENABLE) && !wp->cursor_focus)
    _cursor_focus (wp, 1);

  {
    _inputtext_pos_t pos;
    xitk_font_t *fs = NULL;
    if (wp->fontname.s[0])
      fs = xitk_font_load_font (wp->w.wl->xitk, wp->fontname.s);
    if (!fs)
      fs = xitk_font_load_font (wp->w.wl->xitk, xitk_get_cfg_string (wp->w.wl->xitk, XITK_SYSTEM_FONT));
    if (!fs)
      return 1;
    pos.bytes = wp->text.used - wp->text.draw_start;
    pos.pixels = x - XITK_HV_H (wp->w.pos) - wp->text.box_start - wp->text.shift;
    if (pos.pixels < 0)
      pos.pixels = 0;
    if (_inputtext_find_text_pos (wp->text.buf + wp->text.draw_start, &pos, fs, 0))
      wp->text.cursor_pos = pos.bytes + wp->text.draw_start;
  }

  _inputtext_paint (wp, NULL);
  return 1;
}

/*
 *
 */
static int _inputtext_apply_skin (_inputtext_private_t *wp, xitk_skin_config_t *skonfig) {
  const xitk_skin_element_info_t *s = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
  XITK_HV_INIT;

  if (s) {
    XITK_HV_H (wp->w.pos) = s->x;
    XITK_HV_V (wp->w.pos) = s->y;
    xitk_widget_state_from_info (&wp->w, s);
    xitk_short_string_set (&wp->fontname, s->label_fontname);
    wp->color[_IT_NORMAL] = s->label_color;
    wp->color[_IT_FOCUS] = s->label_color_focus;
    wp->skin = s->pixmap_img;
    return 1;
  }
  wp->skin.image = NULL;
  wp->skin.x = 0;
  wp->skin.y = 0;
  wp->skin.width = 0;
  wp->skin.height = 0;
  wp->w.pos.w = 0;
  return 0;
}

static void _inputtext_change_skin (_inputtext_private_t *wp, xitk_skin_config_t *skonfig) {
  XITK_HV_INIT;

  if (!wp->w.skin_element_name[0]) /* noskin widget. should not happen here. */
    return;

  if (!_inputtext_apply_skin (wp, skonfig)) {
    wp->w.size.w = 0;
    return;
  }

  xitk_part_image_states (&wp->skin, wp->img_state_list, 2);
  wp->w.size.w = wp->img_state_list[0].w;
  wp->text.box_width = XITK_HV_H (wp->w.size) - 2 * 2;

  xitk_image_temp_size (wp->w.wl->xitk, XITK_HV_H (wp->w.size) + 2 * XITK_HV_V (wp->w.size), XITK_HV_V (wp->w.size));
  wp->text.temp_img.x = XITK_HV_V (wp->w.size);
  wp->text.temp_img.y = 0;
  wp->text.temp_img.width = XITK_HV_H (wp->w.size);
  wp->text.temp_img.height = XITK_HV_V (wp->w.size);
  xitk_set_widget_pos (&wp->w, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
}

typedef enum {
  _INPUTTEXT_NONE = 0,
  _INPUTTEXT_MOVE_BOL,   /** << Beginning of line */
  _INPUTTEXT_MOVE_LEFT,  /** << Backward */
  _INPUTTEXT_MOVE_RIGHT, /** << Forward */
  _INPUTTEXT_MOVE_EOL,   /** << End of line */
  _INPUTTEXT_DELETE,     /** << Delete char (right) */
  _INPUTTEXT_BACKSPACE,  /** << Delete char (left) */
  _INPUTTEXT_KILL_LINE,  /** << Kill line (right) */
  _INPUTTEXT_TRANSPOSE,  /** << Swap 2 chars (left) */
  _INPUTTEXT_COPY,       /** << Copy */
  _INPUTTEXT_CUT,        /** << Cut */
  _INPUTTEXT_PASTE,      /** << Paste */
  _INPUTTEXT_MENU,       /** << Context menu */
  _INPUTTEXT_RETURN,     /** << Force update even with same value */
  _INPUTTEXT_CHAR,       /** << insert char just typed */
  _INPUTTEXT_LAST
} _inputtext_action_t;

static const uint8_t _inputtext_ctrl_keys[32] = {
  ['a' & 0x1f] = _INPUTTEXT_MOVE_BOL,
  ['b' & 0x1f] = _INPUTTEXT_MOVE_LEFT,
  ['c' & 0x1f] = _INPUTTEXT_COPY,
  ['d' & 0x1f] = _INPUTTEXT_DELETE,
  ['e' & 0x1f] = _INPUTTEXT_MOVE_EOL,
  ['f' & 0x1f] = _INPUTTEXT_MOVE_RIGHT,
  ['k' & 0x1f] = _INPUTTEXT_KILL_LINE,
  ['m' & 0x1f] = _INPUTTEXT_RETURN,
  ['t' & 0x1f] = _INPUTTEXT_TRANSPOSE,
  ['v' & 0x1f] = _INPUTTEXT_PASTE,
  ['x' & 0x1f] = _INPUTTEXT_CUT,
  ['?' & 0x1f] = _INPUTTEXT_BACKSPACE
};

static const uint8_t _inputtext_xitk_keys[XITK_KEY_LASTCODE] = {
  [XITK_KEY_MENU]      = _INPUTTEXT_MENU,
  [XITK_KEY_DELETE]    = _INPUTTEXT_DELETE,
  [XITK_KEY_BACKSPACE] = _INPUTTEXT_BACKSPACE,
  [XITK_KEY_LEFT]      = _INPUTTEXT_MOVE_LEFT,
  [XITK_KEY_RIGHT]     = _INPUTTEXT_MOVE_RIGHT,
  [XITK_KEY_HOME]      = _INPUTTEXT_MOVE_BOL,
  [XITK_KEY_BEGIN]     = _INPUTTEXT_MOVE_BOL,
  [XITK_KEY_END]       = _INPUTTEXT_MOVE_EOL,
  [XITK_KEY_RETURN]    = _INPUTTEXT_RETURN
};

static const uint8_t tab_utf8_type[256] = {
  /** 0xxxxxxx */
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  /** 10xxxxxx */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  /** 110xxxxx */
  2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
  2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
  /** 1110xxxx */
  3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
  /** 11110xxx */
  4,4,4,4,4,4,4,4,
  /** 111110xx */
                  5,5,5,5,
  /** 1111110x */
                          6,6,
  /** 11111110 */
                              7,
  /** 11111111 */
                                8
};

static int _inputtext_utf8_peek_left (const char *s, int have) {
  const uint8_t *p = (const uint8_t *)s;
  int v;
  do {
    if ((v = tab_utf8_type[*--p]))
      break;
    if ((v = tab_utf8_type[*--p]))
      break;
    if ((v = tab_utf8_type[*--p]))
      break;
    if ((v = tab_utf8_type[*--p]))
      break;
    if ((v = tab_utf8_type[*--p]))
      break;
    if ((v = tab_utf8_type[*--p]))
      break;
    if ((v = tab_utf8_type[*--p]))
      break;
    v = tab_utf8_type[*--p];
  } while (0);
  return (v == (const uint8_t *)s - p) ? v : (have > 0);
}

static int _inputtext_utf8_peek_right (const char *s, int have) {
  const uint8_t *p = (const uint8_t *)s;
  int v = tab_utf8_type[*p], i;
  for (i = 1; (i < v) && !tab_utf8_type[p[i]]; i++) ;
  return (i == v) ? v : (have > 0);
}

/*
 * Handle keyboard event in input text box.
 */
static int _inputtext_key (_inputtext_private_t *wp, const char *s, int modifier) {
  _inputtext_action_t action = _INPUTTEXT_NONE;

  if (!s)
    return 0;
  if (!s[0])
    return 0;

  do {
    const uint8_t *t = (const uint8_t *)s;

    if (t[0] && !(t[0] & 0xe0) && !t[1]) {
      if ((action = _inputtext_ctrl_keys[t[0]]))
        break;
      return 0;
    }
    if (modifier & (MODIFIER_CTRL | MODIFIER_META))
      return 0;
    if (t[0] == XITK_CTRL_KEY_PREFIX) {
      if (t[1] >= XITK_KEY_LASTCODE)
        return 0;
      if ((action = _inputtext_xitk_keys[t[1]]))
        break;
      return 0;
    }
    action = _INPUTTEXT_CHAR;
    if (t[0]) {
      int len = xitk_find_byte (s, 0);

      if (_inputtext_sbuf_insert (wp, s, len) > 0)
        wp->text.dirty |= DIRTY_BEFORE | DIRTY_CHANGED;
    }
  } while (0);

  {
    const uint32_t need_no_buf =
      (1 << _INPUTTEXT_PASTE) |
      (1 << _INPUTTEXT_MENU) |
      (1 << _INPUTTEXT_RETURN) |
      (1 << _INPUTTEXT_CHAR);
    if (!wp->text.buf && !((need_no_buf >> action) & 1))
      return 1;
  }

  switch (action) {

    default: return 0;

    case _INPUTTEXT_CHAR: break;

    case _INPUTTEXT_TRANSPOSE:
      {
        char *p = wp->text.buf + wp->text.cursor_pos;
        int b = _inputtext_utf8_peek_left (p, wp->text.cursor_pos);
        int a = _inputtext_utf8_peek_left (p - b, wp->text.cursor_pos - b);
        if (a) {
          char temp[16];
          int i;
          p -= a + b;
          for (i = 0; i < a + b; i++)
            temp[i] = p[i];
          for (i = 0; i < a; i++)
            p[b + i] = temp[i];
          for (i = 0; i < b; i++)
            p[i] = temp[a + i];
        }
        wp->text.dirty |= DIRTY_BEFORE | DIRTY_CHANGED;
      }
      break;

    case _INPUTTEXT_KILL_LINE:
      /* Erase chars from cursor pos to EOL. */
      if (wp->text.cursor_pos >= wp->text.used)
        return 1;
      wp->text.used = wp->text.cursor_pos;
      memcpy (wp->text.buf + wp->text.used, SBUF_PLUG, SBUF_PAD);
      wp->text.dirty |= DIRTY_AFTER | DIRTY_CHANGED;
      break;

    case _INPUTTEXT_BACKSPACE:
      /* Erase one char from left of cursor. */
      if (wp->text.cursor_pos <= 0)
        return 1;
      {
        char *p = wp->text.buf + wp->text.cursor_pos;
        int rest = wp->text.used - wp->text.cursor_pos;
        int n = _inputtext_utf8_peek_left (p, wp->text.cursor_pos);
        if (rest > 0)
          memmove (p - n, p, rest);
        wp->text.cursor_pos -= n;
        wp->text.used -= n;
        memcpy (wp->text.buf + wp->text.used, SBUF_PLUG, SBUF_PAD);
        wp->text.dirty |= DIRTY_BEFORE | DIRTY_CHANGED;
      }
      break;

    case _INPUTTEXT_DELETE:
      /* Erase one char from right of cursor. */
      if (wp->text.cursor_pos >= wp->text.used)
        return 1;
      {
        char *p = wp->text.buf + wp->text.cursor_pos;
        int rest = wp->text.used - wp->text.cursor_pos;
        int n = _inputtext_utf8_peek_right (p, rest);
        if (n < rest)
          memmove (p, p + n, rest - n);
        wp->text.used -= n;
        memcpy (wp->text.buf + wp->text.used, SBUF_PLUG, SBUF_PAD);
        wp->text.dirty |= DIRTY_AFTER | DIRTY_CHANGED;
      }
      break;

    case _INPUTTEXT_COPY:
      if (wp->text.used > 0) {
#ifndef NEW_CLIPBOARD
        xitk_lock_display (wp->w.wl->xitk);
        XStoreBytes (xitk_x11_get_display(wp->w.wl->xitk), wp->text.buf, wp->text.used);
        xitk_unlock_display (wp->w.wl->xitk);
#else
        xitk_clipboard_set_text (&wp->w, wp->text.buf, wp->text.used);
#endif
      }
      return 1;

    case _INPUTTEXT_CUT:
      if (wp->text.used > 0) {
#ifndef NEW_CLIPBOARD
        xitk_lock_display (wp->w.wl->xitk);
        XStoreBytes (xitk_x11_get_display (wp->w.wl->xitk), wp->text.buf, wp->text.used);
        xitk_unlock_display (wp->w.wl->xitk);
#else
        xitk_clipboard_set_text (&wp->w, wp->text.buf, wp->text.used);
#endif
        wp->text.cursor_pos = 0;
        wp->text.used = 0;
        memcpy (wp->text.buf, SBUF_PLUG, SBUF_PAD);
        wp->text.dirty |= DIRTY_BEFORE | DIRTY_AFTER | DIRTY_CHANGED;
        break;
      }
      return 1;

    case _INPUTTEXT_PASTE:
      {
        char *insert;
        int ilen = 0;

#ifndef NEW_CLIPBOARD
        xitk_lock_display (wp->w.wl->xitk);
        insert = XFetchBytes (xitk_x11_get_display (wp->w.wl->xitk), &ilen);
        xitk_unlock_display (wp->w.wl->xitk);
#else
        insert = NULL;
        ilen = xitk_clipboard_get_text (&wp->w, &insert, wp->max_length - wp->text.used);
#endif
        if (insert && (_inputtext_sbuf_insert (wp, insert, ilen) > 0)) {
          wp->text.dirty |= DIRTY_BEFORE | DIRTY_CHANGED;
#ifndef NEW_CLIPBOARD
          if (insert)
            XFree (insert);
#endif
          break;
        }
#ifndef NEW_CLIPBOARD
        if (insert)
          XFree (insert);
#endif
      }
      return 1;

    case _INPUTTEXT_MOVE_EOL:
      wp->text.cursor_pos = wp->text.used;
      break;

    case _INPUTTEXT_MOVE_LEFT:
      /* Move cursor pos to left. */
      wp->text.cursor_pos -= _inputtext_utf8_peek_left (
        wp->text.buf + wp->text.cursor_pos, wp->text.cursor_pos);
      break;

    case _INPUTTEXT_MOVE_RIGHT:
      /* Move cursor pos to right. */
      wp->text.cursor_pos += _inputtext_utf8_peek_right (
        wp->text.buf + wp->text.cursor_pos, wp->text.used - wp->text.cursor_pos);
      break;

    case _INPUTTEXT_MOVE_BOL:
      wp->text.cursor_pos = 0;
      break;

    case _INPUTTEXT_MENU:
      {
        XITK_HV_INIT;
        xitk_hv_t hv;
        hv.w = wp->w.pos.w + wp->w.size.w - ((4 << 16) + 10);
        return _inputtext_menu_open (wp, XITK_HV_H (hv), XITK_HV_V (hv));
      }

    case _INPUTTEXT_RETURN:
      /* fire callback last inside _inputtext_paint (), it may modify the widget list. */
      wp->text.dirty |= DIRTY_CHANGED;
      /* Remove focus of widget. */
      wp->w.state &= ~(XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS);
      break;
  }
  _inputtext_paint (wp, NULL);
  return 1;
}

static int _inputtext_event (xitk_widget_t *w, const widget_event_t *event) {
  _inputtext_private_t *wp;
  static const widget_event_t paste_event = {
    .type = WIDGET_EVENT_KEY,
    .string = "\x16", /* 'v' & 0x1f */
    .pressed = 1
  };

  xitk_container (wp, w, w);
  if (!wp || !event)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_INPUTTEXT)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      _inputtext_paint (wp, event);
      return 0;
    case WIDGET_EVENT_CLICK:
      return _inputtext_click (wp, event->button, !event->pressed, event->x, event->y);
    case WIDGET_EVENT_CLIP_READY:
      event = &paste_event;
      /* fall through */
    case WIDGET_EVENT_KEY:
      return event->pressed ? _inputtext_key (wp, event->string, event->modifier) : 0;
    case WIDGET_EVENT_INSIDE:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        XITK_HV_INIT;
        return xitk_image_inside (wp->skin.image,
          wp->skin.x + event->x - XITK_HV_H (wp->w.pos),
          wp->skin.y + event->y - XITK_HV_V (wp->w.pos)) ? 1 : 2;
      }
      return 2;
    case WIDGET_EVENT_CHANGE_SKIN:
      _inputtext_change_skin (wp, event->skonfig);
      return 0;
    case WIDGET_EVENT_DESTROY:
      if (wp->menu) {
        wp->menu->wp = NULL;
        if (--wp->menu->refs == 0)
          free (wp->menu);
        wp->menu = NULL;
      }
      if (!wp->w.skin_element_name[0])
        xitk_image_free_image (&wp->skin.image);
      _inputtext_sbuf_unset (wp);
      xitk_short_string_deinit (&wp->fontname);
      return 0;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == BACKGROUND_SKIN) ? wp->skin.image : NULL;
        return 1;
      }
      return 0;
    default: ;
  }
  return 0;
}

int xitk_inputtext_is_editing (xitk_widget_t *w) {
  _inputtext_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_INPUTTEXT)) ? (wp->text.dirty & DIRTY_CHANGED) : 0;
}

/*
 * Return the text of widget.
 */
const char *xitk_inputtext_get_text (xitk_widget_t *w) {
  _inputtext_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_INPUTTEXT)) ? wp->text.buf : "";
}

static void _xitk_inputtext_set_text (_inputtext_private_t *wp, const char *text) {
  wp->text.used = 0;
  wp->text.cursor_pos = 0;
  memcpy (wp->text.buf, SBUF_PLUG, SBUF_PAD);
  wp->text.dirty |= DIRTY_AFTER;
  wp->text.dirty &= DIRTY_CHANGED;
  if (text)
    _inputtext_sbuf_insert (wp, text, xitk_find_byte (text, 0));
  wp->text.draw_start = 0;
  wp->text.draw_stop = 0;
}

/*
 * Change and redisplay the text of widget.
 */
void xitk_inputtext_change_text (xitk_widget_t *w, const char *text) {
  _inputtext_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_INPUTTEXT)) {
    _xitk_inputtext_set_text (wp, text);
    _inputtext_paint (wp, NULL);
  }
}

/*
 * Create input text box
 */
static xitk_widget_t *_xitk_inputtext_create (_inputtext_private_t *wp, const xitk_inputtext_widget_t *it) {
  XITK_HV_INIT;

  ABORT_IF_NULL (wp->w.wl);
  ABORT_IF_NULL (wp->w.wl->xitk);

  wp->cursor_focus    = 0;
  wp->auto_callback   = XITK_WIDGET_STATE_FOCUS;

  wp->menu            = NULL;

  wp->callback        = it->callback;

  wp->w.type          = WIDGET_TYPE_INPUTTEXT | WIDGET_FOCUSABLE | WIDGET_TABABLE
                      | WIDGET_CLICKABLE | WIDGET_KEEP_FOCUS | WIDGET_KEYABLE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event         = _inputtext_event;

  wp->max_length      = it->max_length;

  _inputtext_sbuf_init (wp);
  _xitk_inputtext_set_text (wp, it->text);

  if (!wp->skin.image) {
    wp->w.size.w = 0;
    return &wp->w;
  }

  xitk_part_image_states (&wp->skin, wp->img_state_list, 2);
  wp->w.size.w = wp->img_state_list[0].w;

  wp->text.box_width    = XITK_HV_H (wp->w.size) - 2 * 2;
  wp->text.box_start    = 2;
  if (!wp->w.skin_element_name[0]) {
    wp->text.box_width -= 2 * 1;
    wp->text.box_start += 1;
  }
  xitk_image_temp_size (wp->w.wl->xitk, XITK_HV_H (wp->w.size) + 2 * XITK_HV_V (wp->w.size), XITK_HV_V (wp->w.size));
  wp->text.temp_img.x = XITK_HV_V (wp->w.size);
  wp->text.temp_img.y = 0;
  wp->text.temp_img.width = XITK_HV_H (wp->w.size);
  wp->text.temp_img.height = XITK_HV_V (wp->w.size);

  _xitk_new_widget_apply (&it->nw, &wp->w);

  return &wp->w;
}

xitk_widget_t *xitk_inputtext_create (const xitk_inputtext_widget_t *it, xitk_skin_config_t *skonfig) {
  _inputtext_private_t *wp;

  wp = (_inputtext_private_t *)xitk_widget_new (&it->nw, sizeof (*wp));
  if (!wp)
    return NULL;
  xitk_short_string_init (&wp->fontname);

  _inputtext_apply_skin (wp, skonfig);

  return _xitk_inputtext_create (wp, it);
}

/*
 *
 */
xitk_widget_t *xitk_noskin_inputtext_create (const xitk_inputtext_widget_t *it, int x, int y, int width, int height,
  uint32_t ncolor, uint32_t fcolor, const char *fontname) {
  _inputtext_private_t *wp;
  XITK_HV_INIT;

  wp = (_inputtext_private_t *)xitk_widget_new (&it->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;

  wp->w.state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);

  xitk_short_string_init (&wp->fontname);
  xitk_short_string_set (&wp->fontname, fontname);

  wp->color[_IT_NORMAL] = ncolor;
  wp->color[_IT_FOCUS] = fcolor;

  wp->w.skin_element_name[0] = 0;
  if (xitk_shared_image (wp->w.wl, "xitk_inputtext", width * 2, height, &wp->skin.image) == 1)
    xitk_image_draw_bevel_two_state (wp->skin.image);
  wp->skin.x = 0;
  wp->skin.y = 0;
  wp->skin.width = width * 2;
  wp->skin.height = height;

  return _xitk_inputtext_create (wp, it);
}
