/*
 * @(#)MlinkU.c
 *
 * Copyright 1994 - 2008  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Undo algorithm */

#include "MlinkP.h"

static int startSpace;
int *startPosition = NULL;

static void
newStack(MlinkStack *s)
{
	if (s->lastMove != NULL || s->firstMove != NULL)
		return;
	if (!(s->lastMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	if (!(s->firstMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->firstMove->previous = s->lastMove->next = NULL;
	s->firstMove->next = s->lastMove;
	s->lastMove->previous = s->firstMove;
	s->count = 0;
}

static void
pushStack(MlinkStack *s, MoveRecord **move)
{
	if (!(s->currMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->lastMove->previous->next = s->currMove;
	s->currMove->previous = s->lastMove->previous;
	s->currMove->next = s->lastMove;
	s->lastMove->previous = s->currMove;
	*move = &(s->currMove->move);
	s->count++;
}

static void
popStack(MlinkStack *s)
{
	s->currMove = s->lastMove->previous;
	s->lastMove->previous->previous->next = s->lastMove;
	s->lastMove->previous = s->lastMove->previous->previous;
	free(s->currMove);
	s->count--;
}

static MoveRecord *
topStack(MlinkStack *s)
{
	return &(s->lastMove->previous->move);
}

static int
emptyStack(MlinkStack *s)
{
	return (s->lastMove->previous == s->firstMove);
}

static void
flushStack(MlinkStack *s)
{
	while (s->lastMove->previous != s->firstMove) {
		s->currMove = s->lastMove->previous;
		s->lastMove->previous->previous->next = s->lastMove;
		s->lastMove->previous = s->lastMove->previous->previous;
		free(s->currMove);
	}
	s->count = 0;
}

static void
deleteStack(MlinkStack *s)
{
	flushStack(s);
	if (s->firstMove) {
		free(s->firstMove);
		s->firstMove = NULL;
	}
	if (s->lastMove) {
		free(s->lastMove);
		s->lastMove = NULL;
	}
}

/**********************************/

void
newMoves(MlinkStack *s)
{
	newStack(s);
}

void
deleteMoves(MlinkStack *s)
{
	deleteStack(s);
}

static void
writeMove(MoveRecord *move, int direction, int tile, int control)
{
#if 0
	move->direction = direction;
	move->control = control;
#endif
	move->packed = ((control & 0xF) << 4) + (direction & 0xF);
	move->tile = tile;
}

static void
readMove(MoveRecord *move, int *direction, int *tile, int *control)
{
#if 0
	*direction = move->direction;
	*control = move->control;
#endif
	*direction = (int) (move->packed & 0xF);
	*control = (int) ((move->packed >> 4) & 0xF);
	*tile = move->tile;
}

void
setMove(MlinkStack *s, int direction, int tile, int control)
{
	MoveRecord *move;

	pushStack(s, &move);
	writeMove(move, direction, tile, control);
}

void
getMove(MlinkStack *s, int *direction, int *tile, int *control)
{
	readMove(topStack(s), direction, tile, control);
	popStack(s);
}

int
madeMoves(MlinkStack *s)
{
	return !emptyStack(s);
}

void
flushMoves(MlinkWidget w, MlinkStack *s, Boolean undo)
{
	int i;

	flushStack(s);
	if (undo) {
		startSpace = w->mlink.spacePosition;
		startPosition[w->mlink.tileFaces - 1] = 0;
		for (i = 0; i < w->mlink.tileFaces; i++)
			startPosition[i] = w->mlink.tileOfPosition[i];
	}
}

int
numMoves(MlinkStack *s)
{
	return s->count;
}

void
scanMoves(FILE *fp, MlinkWidget w, int moves)
{
	int direction, tile, shift, l, c;

	for (l = 0; l < moves; l++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d %d %d", &direction, &tile, &shift);
		if (!movePuzzle(w, direction, tile, shift, False, INSTANT))
			(void) fprintf(stderr,
				"%d move in direction %d, tile %d, shift %d, can not be made.\n",
				l, direction, tile, shift);
	}
}

void
printMoves(FILE *fp, MlinkStack *s)
{
	int direction, tile, control, counter = 0;

	s->currMove = s->firstMove->next;
	(void) fprintf(fp, "moves\tdir\ttile\tshift\n");
	while (s->currMove != s->lastMove) {
		readMove(&(s->currMove->move), &direction, &tile, &control);
		(void) fprintf(fp, "%d%c\t%d\t%d\t%d\n",
			++counter, SYMBOL, direction, tile, control);
		s->currMove = s->currMove->next;
	}
}

void
scanStartPosition(FILE *fp, MlinkWidget w)
{
	int i, num, c;

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	for (i = 0; i < w->mlink.tileFaces; i++) {
		(void) fscanf(fp, "%d ", &num);
		startPosition[i] = num;
		if (!num)
			startSpace = i;
	}
}

void
printStartPosition(FILE *fp, MlinkWidget w)
{
	int i;

	(void) fprintf(fp, "\nstartingPosition%c\n", SYMBOL);
	for (i = 0; i < w->mlink.tileFaces; i++) {
		(void) fprintf(fp, "%3d ", startPosition[i]);
		if (!((i + 1) % w->mlink.tiles))
			(void) fprintf(fp, "\n");
	}
	(void) fprintf(fp, "\n");
}

void
setStartPosition(MlinkWidget w)
{
	int i;

	w->mlink.spacePosition = startSpace;
	for (i = 0; i < w->mlink.tileFaces; i++)
		w->mlink.tileOfPosition[i] = startPosition[i];
	drawAllTiles(w);
}
