// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <server.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

typedef struct _cdr
{
	time_t firstsend;
	time_t lastsend;
	struct _cdr *next;
	struct _cdr *prev;
	char data[0];
}	cdr_t;

class CDRAudit : public Audit, public UDPSocket, public Server
{
private:
	cdr_t *first, *last;
	unsigned retry, expires;
	timeout_t interval;

	void remove(cdr_t *cdr);

	void run(void);

	void reportAudit(Trunk *trunk, char *detail)
		{return;};

	void reportCDR(Trunk *trunk, char *detail);

	void reportStats(void)
		{return;};

	char logpath[256];

public:
	CDRAudit();

} mycdrlog;

CDRAudit::CDRAudit() :
Audit(), UDPSocket(), Server(keythreads.priNetwork())
{
	const char *key;

	if(setBroadcast(true))
		slog(Slog::levelWarning) << "audit: broadcast flag failed" << endl;

	first = last = NULL;

	key = keys.getLast("retry");
	if(key)	
		retry = atoi(key);
	else
		retry = 15;

	key = keys.getLast("expires");
	if(key)
		expires = atoi(key);
	else
		expires = 60;

	key = keys.getLast("interval");
	if(key)
		interval = atoi(key) * 1000;
	else
		interval = 5000;

	setPeer(keynetwork.getDBHost(), keynetwork.getDBPort());	
}

void CDRAudit::reportCDR(Trunk *trunk, char *detail)
{
	char buffer[400];
	cdr_t *cdr;
	unsigned len;

	sprintf(buffer, "C%s: port=%s duration=%s date=%s time=%s %s",
		trunk->getSymbol(SYM_GID),
		trunk->getSymbol(SYM_ID),
		trunk->getSymbol(SYM_DURATION),
		trunk->getSymbol(SYM_STARTDATE),
		trunk->getSymbol(SYM_STARTTIME),
		detail);		

	// slog(Slog::levelDebug) << "audit: " << buffer + 1 << endl;
	len = strlen(buffer) + 1;
	send(buffer, len);
	cdr = (cdr_t *)new char[sizeof(cdr_t) + len];
	time(&cdr->firstsend);
	time(&cdr->lastsend);
	strcpy(cdr->data, buffer);
	enterMutex();
	cdr->prev = last;
	cdr->next = NULL;
	if(last)
		last->next = cdr;
	last = cdr;
	if(!first)
		first = cdr;
	leaveMutex();
}

void CDRAudit::remove(cdr_t *cdr)
{
	if(cdr->next)
		cdr->next->prev = cdr->prev;

	if(cdr->prev)
		cdr->prev->next = cdr->next;

	if(cdr == first)
		first = cdr->next;

	if(cdr == last)
		last = cdr->prev;

	delete[] (char *)cdr;
}

void CDRAudit::run(void)
{
	time_t now;
	unsigned len;
	char buffer[256];
	cdr_t *cdr, *next;

	slog(Slog::levelDebug) << "audit: starting service" << endl;

	setCancel(cancelImmediate);
	for(;;)
	{
		time(&now);
		enterMutex();
		cdr = first;
		while(cdr)
		{
			next = cdr->next;
			if((time_t)cdr->firstsend + (time_t)expires < now)
			{
				slog(Slog::levelDebug) << "expire: " << cdr->data << endl;
				remove(cdr);
			}
			else if((time_t)cdr->lastsend + (time_t)retry <= now)
			{
				send(cdr->data, strlen(cdr->data) + 1);
				time(&cdr->lastsend);
			}
			cdr = next;
		}
		leaveMutex();
		if(!isPending(pendingInput, interval))
			continue;
		
		len = receive(buffer, sizeof(buffer) - 1);
		if(buffer[0] != 'A')
			continue;
		buffer[len] = 0;
		len = strlen(buffer + 1);
		enterMutex();
		cdr = first;
		while(cdr)
		{
			if(!strncmp(buffer + 1, cdr->data, len))
			{
				remove(cdr);
				break;
			}
			cdr = cdr->next;
		}
		leaveMutex();
	}		
}

#ifdef	CCXX_NAMESPACES
};
#endif
