/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef _GR_IIR_H_
#define _GR_IIR_H_

#include <vector>
using std::vector;

/*!
 * \brief base class template for Infinite Impulse Response filter (IIR)
 */
template<class o_type, class i_type, class tap_type> 
class gr_iir {
public:
  /*!
   * \brief construct new IIR with given taps.
   *
   * Note that taps must be in backwards order, e.g., coefficient 0 is
   * stored in xxtaps[N-1], coefficient 1 is stored in
   * xxtaps[N-2], etc.
   *
   * fftaps comtains the feed-forward taps, and fbtaps
   * the feedback ones
   */
  gr_iir (const vector<tap_type>& fftaps, const vector<tap_type>& fbtaps)
	  : fftaps (fftaps),fbtaps(fbtaps),
	    latest(0),prev_output(fftaps.size()),prev_input(fftaps.size()) {}

  gr_iir () :latest(0) { }

  virtual ~gr_iir () {}

  /*!
   * \brief compute a single output value.
   * \returns the filtered input value.
   */
  virtual o_type filter (const i_type input);

  /*!
   * \brief compute an array of N output values.
   * \p input must have n valid entries.
   */
  virtual void filterN (o_type output[], const i_type input[],
			unsigned long n); // const;

  /*!
   * \return number of taps in filter.
   */
  unsigned ntaps () const { return fftaps.size (); }

  /*!
   * \brief install \p new_taps as the current taps.
   */
  virtual void set_taps (const vector<tap_type> &newfftaps, 
			  const vector<tap_type> &newfbtaps) 
  { 
    fftaps = newfftaps; 
    fbtaps = newfbtaps;
    prev_output.resize (fftaps.size ());
    prev_input.resize (fftaps.size ());
  }

protected:
  vector<tap_type>	fftaps;
  vector<tap_type>	fbtaps;
  int latest;
  vector<o_type>	prev_output;
  vector<o_type>	prev_input;
};


//
// general case.  We may want to specialize this
//
template<class o_type, class i_type, class tap_type> 
o_type
gr_iir<o_type, i_type, tap_type>::filter (const i_type input)
{
  o_type	acc;	// FIXME.  Is this the best acc type?
  unsigned	i = 0;
  unsigned	n = ntaps ();

  acc = fftaps[0] * input;
  for (i = 1; i < n; i ++)
    acc += fftaps[i] * prev_input[(latest + i) % n] + fbtaps[i] * prev_output[(latest + i) % n];

  prev_output[latest] = acc;
  prev_input[latest] = input;
  latest--;
  if(latest<0)
	  latest+=n;
  return (o_type) acc;
}


template<class o_type, class i_type, class tap_type> 
void 
gr_iir<o_type, i_type, tap_type>::filterN (o_type output[],
					   const i_type input[],
					   unsigned long n)
{
  for (unsigned i = 0; i < n; i++)
    output[i] = filter (input[i]);
}


#endif /* _IIR_H_ */
