/*
 This file is part of GNU Taler
 (C) 2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  Duration,
  j2s,
  TalerCorebankApiClient,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  makeTestPaymentV2,
  useSharedTestkudosEnvironment,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

export async function runWalletTransactionsTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant, merchantAdminAccessToken } =
    await useSharedTestkudosEnvironment(t);

  {
    const txRes = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {},
    );
    t.assertDeepEqual(txRes.transactions.length, 0);
  }

  // Withdraw digital cash into the wallet.

  const bankClient = new TalerCorebankApiClient(bank.baseUrl);

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  {
    const txRes = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {},
    );
    t.assertDeepEqual(txRes.transactions.length, 1);
  }

  const order = {
    summary: "Buy me! First.",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  } satisfies TalerMerchantApi.Order;
  await makeTestPaymentV2(t, { walletClient, merchant, order, merchantAdminAccessToken });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const order2 = {
    summary: "Buy me! Second.",
    amount: "TESTKUDOS:3",
    fulfillment_url: "taler://fulfillment-success/thx",
  } satisfies TalerMerchantApi.Order;
  await makeTestPaymentV2(t, { walletClient, merchant, order: order2, merchantAdminAccessToken });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  await walletClient.call(WalletApiOperation.InitiatePeerPushDebit, {
    partialContractTerms: {
      amount: "TESTKUDOS:5",
      purse_expiration: AbsoluteTime.toProtocolTimestamp(
        AbsoluteTime.addDuration(
          AbsoluteTime.now(),
          Duration.fromSpec({
            minutes: 10,
          }),
        ),
      ),
      summary: "Test",
    },
  });

  {
    const txRes = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {},
    );

    console.log(`all transactions: ${j2s(txRes.transactions)}`);

    t.assertDeepEqual(txRes.transactions.length, 4);

    const txResSp1 = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {
        limit: 2,
      },
    );

    console.log(`transactions (limit=2): ${j2s(txResSp1.transactions)}`);

    t.assertDeepEqual(txResSp1.transactions.length, 2);

    const offset = txResSp1.transactions[1].transactionId;
    const txResSp2 = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {
        offsetTransactionId: offset,
      },
    );

    console.log(
      `transactions (offset=${offset}): ${j2s(txResSp2.transactions)}`,
    );

    const reconstructed = [...txResSp1.transactions, ...txResSp2.transactions];

    t.assertDeepEqual(reconstructed, txRes.transactions);
  }

  {
    // Test filtering
    const txRes = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {
        filterByState: "final",
      },
    );
    t.assertDeepEqual(txRes.transactions.length, 3);
  }
}

runWalletTransactionsTest.suites = ["wallet"];
